# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

from AutomationUtils.cvtestcase import CVTestCase
from Reports.storeutils import StoreUtils
from Web.Common.cvbrowser import Browser
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.Store.storeapp import ReadMe
from Web.WebConsole.Store.storeapp import StoreApp
from Web.WebConsole.webconsole import WebConsole


class TestCase(CVTestCase):

    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Store: Media Kit downloads"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.SOFTWARESTORE
        self.feature = self.features_list.WEBCONSOLE
        self.browser = None
        self.webconsole = None
        self.reports = None
        self.store: StoreApp = None
        self.util = StoreUtils(self)
        self.inputs = StoreUtils.get_store_config()
        self.readme: ReadMe = None

    def init_tc(self):
        try:
            self.browser = BrowserFactory().create_browser_object(Browser.Types.FIREFOX)
            self.browser.set_downloads_dir(self.util.get_temp_dir())
            self.browser.open()
            self.webconsole = WebConsole(
                self.browser, self.commcell.webconsole_hostname
            )
            self.webconsole.login()
            self.webconsole.goto_store()
            self.store = StoreApp(self.webconsole)
            self.readme = ReadMe(self.webconsole)
        except Exception as e:
            raise CVTestCaseInitFailure(e) from e

    @test_step
    def start_step1(self):
        """Download media kit package which has only one platform type"""
        self.util.reset_temp_dir()
        self.store.download_package(
            self.inputs.MEDIAKIT.Single.name,
            category="Media Kits"
        )
        self.store.goto_readme(
            self.inputs.MEDIAKIT.Single.name,
            category="Media Kits"
        )
        descriptions = self.readme.get_readme_description()
        self.util.poll_for_tmp_files(ends_with="exe", count=1)
        self.util.validate_tmp_files(
            ends_with="exe",
            count=1,
            hashes=[
                description[:32].lower()
                for description in descriptions.split(" ")
                if len(description.strip()) >= 32
            ]
        )

    @test_step
    def start_step2(self):
        """Download media kit from package which has multiple platform types"""
        self.util.reset_temp_dir()
        self.store.goto_store_home()
        self.store.download_packages_with_multiple_platforms(
            self.inputs.MEDIAKIT.Multi.name,
            category="Media Kits",
            platforms=[
                self.inputs.MEDIAKIT.Multi.platforms[0].type
            ]
        )
        self.util.poll_for_tmp_files(
            ends_with="exe",
            count=1
        )
        f_name = self.inputs.MEDIAKIT.Multi.platforms[0].file_name
        self.util.validate_tmp_files(
            ends_with=f_name,
            count=1,
            min_size=1
        )

    @test_step
    def start_step3(self):
        """Click Download All button on media kit download window"""
        # TODO: Hardcode hashes
        self.util.reset_temp_dir()
        self.store.goto_store_home()
        self.store.download_packages_with_multiple_platforms(
            self.inputs.MEDIAKIT.Multi.name,
            category="Media Kits"
        )
        self.store.goto_readme(
            self.inputs.MEDIAKIT.Multi.name,
            category="Media Kits"
        )
        descriptions = self.readme.get_readme_description()
        self.util.poll_for_tmp_files(
            ends_with="exe",
            count=len(self.inputs.MEDIAKIT.Multi.platforms)
        )
        self.util.validate_tmp_files(
            ends_with="exe",
            count=len(self.inputs.MEDIAKIT.Multi.platforms),
            min_size=1,
            hashes=[
                description[:32].lower()
                for description in descriptions.split(" ")
                if len(description.strip()) >= 32
            ]  # Md5 hashes are 32 chars wide
        )

    def run(self):
        try:
            self.init_tc()
            self.start_step1()
            self.start_step2()
            self.start_step3()
        except Exception as err:
            self.util.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
