# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Custom Reports: Inputs with multi selection disabled"""


from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.Reports.Custom import builder
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.Reports.Custom.inputs import (
    String,
    ListBoxController,
    Time,
    TimePickerController
)


from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.webconsole import WebConsole

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.utils import CustomReportUtils


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Inputs with multi selection disabled"
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.report_builder = None
        self.report_viewer = None
        self.table = None
        self.time_picker = None
        self.date_picker_controller = None
        self.date_time_picker_controller = None
        self.list_box = None
        self.date = None
        self.time = None
        self.date_time = None
        self.string = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()

            # Deletes any previous report if any with the same name.
            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)

            # Navigate to Custom Report Builder Page.
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()

            self.report_builder = builder.ReportBuilder(self.webconsole)
            self.report_builder.set_report_name(self.name)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def add_inputs(self):
        """ Adds inputs to the builder."""

        self.time = Time("Time_t")
        self.report_builder.add_input(self.time)
        self.time_picker = TimePickerController("Time_t")
        self.time.add_html_controller(self.time_picker)
        self.time.save()

        self.string = String("ListBox_t")
        self.report_builder.add_input(self.string)
        self.list_box = ListBoxController("ListBox_t")
        self.string.add_html_controller(self.list_box)
        self.list_box.set_labels_and_values(["A", "B", "C"], ["1", "2", "3"])
        self.string.save()

    @test_step
    def associate_inputs(self):
        """Add HTML input and parameter to Dataset and preview"""
        dataset = builder.Datasets.DatabaseDataset()
        self.report_builder.add_dataset(dataset)
        dataset.set_dataset_name("Test Dataset")
        dataset.set_sql_query(
            "SELECT @time_t [Time], @listbox [ListBox]"
        )

        dataset.add_parameter("time_t", self.time)
        dataset.add_parameter("listbox", self.string)

        dataset.save()

        self.table = builder.DataTable("Automation Table 1")
        self.report_builder.add_component(self.table, dataset)
        self.table.add_column_from_dataset()

    @test_step
    def validate_data(self):
        """Validate data on both on builder and viewer"""
        self.report_builder.save(deploy=True)
        self.populate_input_and_validate()

        self.report_builder.open_report()

        self.report_viewer = viewer.CustomReportViewer(self.webconsole)
        self.table = viewer.DataTable("Automation Table 1")
        self.report_viewer.associate_component(self.table)

        self.report_viewer.associate_input(self.time_picker)
        self.report_viewer.associate_input(self.list_box)

        self.populate_input_and_validate()

    @test_step
    def bookmark_report_url(self):
        """Bookmark the report URL with input and validate data"""
        self.browser.driver.refresh()
        self.table = viewer.DataTable("Automation Table 1")
        self.report_viewer.associate_component(self.table)
        self.populate_input_and_validate(populate=False)

    def populate_input_and_validate(self, populate=True):
        """Populates input and validates if set to true"""
        if populate:
            self.time_picker.set_time_controller("12", "30", "AM")
            self.list_box.select_value("A")
            self.list_box.apply()

        data = self.table.get_table_data()
        expected_result = {
            "Time": ["12:30:00 AM"],
            "ListBox": ["1"],
        }

        if data != expected_result:
            self.log.error(f"Expected Data {expected_result}")
            self.log.error(f"Received Data {data}")
            raise CVTestStepFailure("Data validation failed")

    def run(self):
        try:
            self.init_tc()
            self.add_inputs()
            self.associate_inputs()
            self.validate_data()
            self.bookmark_report_url()

        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
