# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Testcase to verify backward compatibility of Metrics Collection queries
"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.mailer import Mailer
from AutomationUtils import config

from Reports.utils import TestCaseUtils
from Reports.metricsutils import MetricsServer

from cvpysdk.commcell import Commcell
from cvpysdk.metricsreport import PrivateMetrics

from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep


_CONSTANTS = config.get_config()


class CommCellObj:
    """
    Class with required objects for commcells used in this testcase
    """
    def __init__(self, commcell_obj, private_metrics):
        self.commcell = commcell_obj
        self.private_metrics = private_metrics


class TestCase(CVTestCase):
    """Testcase to verify backward compatibility of Metrics Collection queries"""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Metrics: Collection queries backward compatibility"
        self.private_metrics = None
        self.utils = TestCaseUtils(self)
        self.tcinputs = {
            "v11_CS": None,  # comma seperated value for multi commcells
            "email_receiver": None
        }
        self.commcell_list = []
        self.metrics_server = None
        self.mailer = None
        self.email_receiver = None

    def create_objects(self, cc_name):
        """method to create required objects with the inputs"""
        self.log.info(f'Connecting to Commcell [{cc_name}]')
        commcell_obj = Commcell(cc_name, _CONSTANTS.ADMIN_USERNAME, _CONSTANTS.ADMIN_PASSWORD)
        self.log.info(f'Getting PrivateMetrics object of Commcell [{cc_name}]')
        return CommCellObj(commcell_obj, PrivateMetrics(commcell_obj))

    def init_tc(self):
        """Initialize the application to the state required by the testcase"""
        try:
            self.commcell_list.append(
                CommCellObj(self.commcell, PrivateMetrics(self.commcell))
            )
            commcells = self.tcinputs["v11_CS"].split(',')
            self.commcell_list.extend(
                list(map(self.create_objects, commcells))
            )
            self.metrics_server = MetricsServer(
                self.commcell.webconsole_hostname,
                self.inputJSONnode['commcell']["commcellUsername"],
                self.inputJSONnode['commcell']["commcellPassword"]
            )
            self.email_receiver = self.tcinputs["email_receiver"]
            self.mailer = Mailer({'receiver': self.email_receiver}, self.commcell)
        except Exception as msg:
            raise CVTestCaseInitFailure(msg) from msg

    @test_step
    def intiate_private_uploadnow(self):
        """Validates Private Metrics uploadNow operation"""
        for cc_obj in self.commcell_list:
            self.log.info(
                f'Initiating Private Metrics upload Now in [{cc_obj.commcell.commserv_name}]'
            )
            cc_obj.private_metrics.update_url(self.commcell.webconsole_hostname)
            cc_obj.private_metrics.enable_all_services()
            cc_obj.private_metrics.upload_now()

    @test_step
    def wait_for_upload_completion(self):
        """Wait for upload completion"""
        for cc_obj in self.commcell_list:
            try:
                self.log.info(f"Waiting for upload completion for [{cc_obj.commcell.commserv_name}] cs")
                cc_obj.private_metrics.wait_for_uploadnow_completion()
                self.log.info(f'Private Metrics Upload completed in [{cc_obj.commcell.commserv_name}]')
            except TimeoutError as timeout_error:
                raise TimeoutError(f"upload failure from [{cc_obj.commcell.commserv_name}] cs "
                                   f"with reason {timeout_error}")

    @test_step
    def wait_for_parsing_completion(self):
        """Wait for parsing completion in metrics server"""
        for cc_obj in self.commcell_list:
            try:
                self.metrics_server.wait_for_parsing(
                    cc_obj.private_metrics.get_uploaded_filename()
                )
                self.log.info(
                    f'Commcell [{cc_obj.commcell.commserv_name}] parsed in metrics server')
            except TimeoutError:
                # uploaded file name can be computed only from SP12 this can be removed once
                # backward CS moves beyond SP12
                self.log.error(
                    f"Parsing couldn't be validated for CS [{cc_obj.commcell.commserv_name}]")

    @staticmethod
    def create_parsing_error_table(parsing_failures):
        """Creates the table to sent in Email for parsing errors"""
        failure_msg = '''
        <p>Below is the list of parsing Failures in Metrics server </p>
        <table  border="1"><tr><th>Message</th></tr>
        '''
        for message in parsing_failures:
            failure_msg += '<tr><td>'
            failure_msg += message[0]
            failure_msg += '</td>'
            failure_msg += '</tr>'

        failure_msg += '</table>'
        return failure_msg

    @staticmethod
    def create_collection_errors_table(collection_failures):
        """Creates the table to sent in Email for Collection errors"""
        failure_msg = '''
        <p>Below is the list of Failure Messages From the Collection Query</p>
        <table  border="1"><tr><th width="18%">CommCell Id</th>
        <th>QueryId</th><th width="60%">Message</th></tr>
        '''
        for failures in collection_failures:
            failure_msg += '<tr><td>'

            failure_msg += str(failures['ccid'])
            failure_msg += '</td><td>'

            failure_msg += str(failures['queryid'])
            failure_msg += '</td><td>'

            failure_msg += str(failures['error'])
            failure_msg += '</td></tr>'

        failure_msg += '</table>'
        return failure_msg

    @test_step
    def check_collection_errors(self):
        """Look for collection errors in metrics server in last 1 day"""
        errors = self.metrics_server.get_collection_errors(days=1)
        if errors:
            self.log.error('Mailing Collection Errors')
            self.mailer.mail("Metrics Collection Error on " + self.commcell.webconsole_hostname,
                             self.create_collection_errors_table(errors))
            raise CVTestStepFailure('Collection Errors exist for last 1 day')

    @test_step
    def check_parsing_errors(self):
        """Look for parsing errors in metrics server in last 1 day"""
        query = """
        SELECT Message FROM CVCloud..cf_SurveyLogger WITH (NOLOCK)
        WHERE (Message like '%failed to%' or Message like '%parsing failed%' or 
        Message like '%Failed with%' or Message like '%Failed in%')
        and LogDateUTC >(GETUTCDATE()-1) order by LogDateUTC desc
        """
        errors = self.metrics_server.metrics_server_api.execute_sql(
            query,
            database_name="CVCloud",
            desc='Getting parsing failed logs from cf_SurveyLogger table'
        )
        if errors:
            self.log.error('Mailing parsing Errors')
            self.mailer.mail("Parsing Error on " + self.commcell.webconsole_hostname,
                             self.create_parsing_error_table(errors))
            raise CVTestStepFailure('Parsing Errors exist for last 1 day')

    def run(self):
        try:
            self.init_tc()
            self.intiate_private_uploadnow()
            self.wait_for_upload_completion()
            self.wait_for_parsing_completion()
            self.check_collection_errors()
            self.check_parsing_errors()

        except Exception as error:
            self.utils.handle_testcase_exception(error)
