# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase to perform Basic AmazonS3 Cloud verification via Access and Secret Access key.

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                  --  initialize TestCase class

    _validate_jmdatastats(job_id, copy_id)
                                --  validates if a job_id on copy_id exists in table jmdatastats

    _validate_archchunkmapping(job_id, copy_id)
                                --  validates if archchunkmapping entry for job id exists on copy_id

    _validate_jobs_prune(job_copy_list)
                                --  validates if a job on copy is aged by checking table entries

    _cleanup()                  --  Cleanup the entities created

    set_multiple_readers()        -- Allow multiple data readers to subclient

    setup()                     --  setup function of this test case

    run()                       --  run function of this test case

    tear_down()                 --  teardown function of this test case
"""

from time import sleep

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.idautils import CommonUtils
from MediaAgents.MAUtils.mahelper import (DedupeHelper, MMHelper)


class TestCase(CVTestCase):
    """Class for executing this test case"""
    def __init__(self):
        """Initializing the Test case file"""
        super(TestCase, self).__init__()
        self.name = "AmazonS3 Cloud verification via Access and Secret Access key"
        self.tcinputs = {
            "ClientName": None,
            "MediaAgentName": None,
            "CloudMountPath": None,
            "CloudUserName": None,
            "CloudPassword": None,
            "CloudServerType": None
        }
        self.disk_library_name = None
        self.cloud_library_name = None
        self.storage_policy_dedupe_name = None
        self.storage_policy_nondedupe_name = None
        self.backupset_name = None
        self.client_machine = None
        self.ma_machine = None
        self.mountpath1 = None
        self.mountpath2 = None
        self.partition_path = None
        self.copy_partition_path = None
        self.partition2_path = None
        self.copy_partition2_path = None
        self.content_path1 = None
        self.content_path2 = None
        self.restore_dest_path = None
        self.common_util = None
        self.dedupehelper = None
        self.mmhelper = None

    def _validate_jmdatastats(self, job_id, copy_id):
        """
        Validate if a job_id on copy_id exists in table jmdatastats
        Args:
            job_id (int) -- backup job id to query the table
            copy_id (int) -- copy id of the job to validate for
        Return:
            (Bool) True if job id exists
            (Bool) False if not exists
        """
        query = """select agedTime, agedBy, disabled&256 from JMJobDataStats where jobId = {0} and archGrpCopyId = {1}
                            """.format(job_id, copy_id)
        #self.log.info("QUERY: {0}".format(query))
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        #self.log.info("RESULT: {0}".format(cur))
        self.log.info("RESULT: %s", cur)
        if cur != ['']:
            values = [int(x) for x in cur]
            if values[0] > 0 and values[1] > 0 and values[2] == 256:
                return False
        return True

    def _validate_archchunkmapping(self, job_id, copy_id):
        """
        Validate if archchunkmapping entry for job id exists on copy_id
        Args:
            job_id (int) -- job id to query the table
            copy_id (int) -- copy id of the job to validate for
        Return:
            (Bool) True if entries exist
            (Bool) False if entries doesnt exist
        """
        query = """select archFileId, archChunkId from archchunkmapping where jobId = {0} and archCopyId = {1}
                        """.format(job_id, copy_id)
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_all_rows()
        self.log.info("RESULT: %s", cur)
        if cur != [['']]:
            return True
        self.log.info("No entries present")
        return False

    def _validate_jobs_prune(self, job_copy_list):
        """
        Validates if a job on copy is aged by checking table entries
        Args:
            job_copy_list (list) -- list of tuples, each tuple has job instance and list of copy instance
        Return:
            (Bool) True/False
        """
        for job_copy in job_copy_list:
            for copy in job_copy[1]:
                if (self._validate_jmdatastats(job_copy[0], int(copy.copy_id)) and
                        self._validate_archchunkmapping(job_copy[0], int(copy.copy_id))):
                    self.log.error("Job %s is not aged on [%s/%s]", str(job_copy[0]),
                                   copy.storage_policy.storage_policy_name, copy._copy_name)
                    raise Exception("Job {0} is not aged on [{1}/{2}]".format(str(job_copy[0]),
                                                                              copy.storage_policy.storage_policy_name,
                                                                              copy._copy_name))
                self.log.info("Job %s is aged on [%s/%s]", str(job_copy[0]), copy.storage_policy.storage_policy_name,
                              copy._copy_name)

    @staticmethod
    def set_multiple_readers(subclient, num_readers):
        """
            Allow multiple data readers to subclient
            Args:
                subclient          (object) --  instance of subclient to set data readers
                num_readers        (int)    --  Number of data readers
        """

        subclient.allow_multiple_readers = True
        subclient.data_readers = num_readers

    def _cleanup(self):
        """Cleanup the entities created"""
        self.log.info("********************** CLEANUP STARTING *************************")
        try:
            # Delete bkupset
            self.log.info("Deleting BackupSet: %s if exists", self.backupset_name)
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
                self.log.info("Deleted BackupSet: %s", self.backupset_name)
            # Delete Storage Policies
            self.log.info("Deleting Dedupe storage policy: %s if exists", self.storage_policy_dedupe_name)
            if self.commcell.storage_policies.has_policy(self.storage_policy_dedupe_name):
                self.commcell.storage_policies.delete(self.storage_policy_dedupe_name)
                self.log.info("Deleted Dedupe storage policy: %s", self.storage_policy_dedupe_name)

            self.log.info("Deleting second Dedupe storage policy: %s if exists", self.storage_policy_nondedupe_name)
            if self.commcell.storage_policies.has_policy(self.storage_policy_nondedupe_name):
                self.commcell.storage_policies.delete(self.storage_policy_nondedupe_name)
                self.log.info("Deleted second Dedupe storage policy: %s", self.storage_policy_nondedupe_name)

            # Delete Disk Library
            self.log.info("Deleting library: %s if exists", self.disk_library_name)
            if self.commcell.disk_libraries.has_library(self.disk_library_name):
                self.commcell.disk_libraries.delete(self.disk_library_name)
                self.log.info("Deleted library: %s", self.disk_library_name)
            # Delete Cloud Library
            self.log.info("Deleting library: %s if exists", self.cloud_library_name)
            if self.commcell.disk_libraries.has_library(self.cloud_library_name):
                self.commcell.disk_libraries.delete(self.cloud_library_name)
                self.log.info("Deleted library: %s", self.cloud_library_name)
            # Run DataAging
            data_aging_job = self.commcell.run_data_aging()
            self.log.info("Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)
            if self.ma_machine.check_directory_exists(self.mountpath1):
                self.ma_machine.remove_directory(self.mountpath1)
            if self.ma_machine.check_directory_exists(self.mountpath2):
                self.ma_machine.remove_directory(self.mountpath2)
            if self.ma_machine.check_directory_exists(self.partition_path):
                self.ma_machine.remove_directory(self.partition_path)
            if self.ma_machine.check_directory_exists(self.copy_partition_path):
                self.ma_machine.remove_directory(self.copy_partition_path)
        except Exception as exp:
            self.log.warning("Error encountered during cleanup : %s", str(exp))

        self.log.info("********************** CLEANUP COMPLETED *************************")

    def setup(self):
        """Setup function of this test case"""
        options_selector = OptionsSelector(self.commcell)

        self.disk_library_name = '%s_disk' % str(self.id)
        self.cloud_library_name = '%s_cloud' % str(self.id)
        self.storage_policy_dedupe_name = '%s_dedupe' % str(self.id)
        self.storage_policy_nondedupe_name = '%s_nondedupe' % str(self.id)
        self.backupset_name = '%s_bs' % str(self.id)
        self.client_machine = Machine(self.client)
        self.ma_machine = Machine(self.tcinputs['MediaAgentName'], self.commcell)
        # To select drive with space available in client machine
        self.log.info('Selecting drive in the client machine based on space available')
        client_drive = options_selector.get_drive(self.client_machine, size=5 * 1024)
        if client_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', client_drive)
        # To select drive with space available in Media agent machine
        self.log.info('Selecting drive in the Media agent machine based on space available')
        ma_drive = options_selector.get_drive(self.ma_machine, size=5 * 1024)
        if ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', ma_drive)
        self.mountpath1 = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'MP1')

        self.mountpath2 = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'MP2')

        self.partition_path = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'DDB')
        self.copy_partition_path = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'Copy_DDB3')

        self.content_path1 = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'Testdata1')
        if self.client_machine.check_directory_exists(self.content_path1):
            self.client_machine.remove_directory(self.content_path1)
        self.client_machine.create_directory(self.content_path1)

        self.content_path2 = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'Testdata2')
        if self.client_machine.check_directory_exists(self.content_path2):
            self.client_machine.remove_directory(self.content_path2)
        self.client_machine.create_directory(self.content_path2)

        self.restore_dest_path = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'Restoredata')
        if self.client_machine.check_directory_exists(self.restore_dest_path):
            self.client_machine.remove_directory(self.restore_dest_path)
        self.client_machine.create_directory(self.restore_dest_path)
        self.dedupehelper = DedupeHelper(self)
        self.mmhelper = MMHelper(self)
        self.common_util = CommonUtils(self)

    def run(self):
        """Run function of this test case"""
        try:
            disk_lib_obj = self.mmhelper.configure_disk_library(self.disk_library_name, self.tcinputs['MediaAgentName'],
                                                                self.mountpath1)
            self.mmhelper.configure_disk_mount_path(disk_lib_obj, self.mountpath2, self.tcinputs['MediaAgentName'])
            cloud_lib_obj = self.mmhelper.configure_cloud_library(self.cloud_library_name,
                                                                  self.tcinputs['MediaAgentName'],
                                                                  self.tcinputs["CloudMountPath"],
                                                                  self.tcinputs["CloudUserName"],
                                                                  self.tcinputs["CloudPassword"],
                                                                  self.tcinputs["CloudServerType"])
            self.mmhelper.configure_cloud_mount_path(cloud_lib_obj, self.tcinputs["CloudMountPath"],
                                                     self.tcinputs['MediaAgentName'], self.tcinputs["CloudUserName"],
                                                     self.tcinputs["CloudPassword"], self.tcinputs["CloudServerType"])
            sp_dedup_obj = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_dedupe_name,
                                                                             self.cloud_library_name,
                                                                             self.tcinputs['MediaAgentName'],
                                                                             self.partition_path)
            # Set retention of 0-day & 1-cycle
            self.log.info("Setting Retention: 0-days and 1-cycle on Primary Copy")
            sp_dedup_primary_obj = sp_dedup_obj.get_copy("Primary")
            retention = (0, 1, -1)
            sp_dedup_primary_obj.copy_retention = retention

            # create secondary copy to copy from cloud to disk.
            copy1_name = '%s_copy1' % str(self.id)
            dedupe_copy_obj = self.dedupehelper.configure_dedupe_secondary_copy(sp_dedup_obj, copy1_name,
                                                                                self.disk_library_name,
                                                                                self.tcinputs['MediaAgentName'],
                                                                                self.copy_partition_path,
                                                                                self.tcinputs['MediaAgentName'])
            # Removing association with System Created Automatic Auxcopy schedule
            self.log.info("Removing association with System Created Autocopy schedule on above created copy")
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_dedupe_name, copy1_name)

            # Set retention of 0day 1cycle
            self.log.info("Setting Retention: 0-days and 1-cycle on Secondary Copy")
            dedupe_copy_obj.copy_retention = retention

            # create backupset
            self.mmhelper.configure_backupset(self.backupset_name, self.agent)
            # create subclient
            subclient1_name = "%s_SC1" % str(self.id)
            sc1_obj = self.mmhelper.configure_subclient(self.backupset_name, subclient1_name,
                                                        self.storage_policy_dedupe_name, self.content_path1, self.agent)
            # Allow multiple data readers to subclient
            self.log.info("Setting Data Readers=4 on Subclient")
            self.set_multiple_readers(sc1_obj, 4)

            job_copy_list = []
            job_types_sequence_list = ['full', 'incremental', 'incremental', 'synthetic_full', 'incremental',
                                       'synthetic_full']
            for sequence_index in range(0, 6):
                # Create unique content
                if job_types_sequence_list[sequence_index] is not 'synthetic_full':
                    self.log.info("Generating Data at %s", self.content_path1)
                    if not self.client_machine.generate_test_data(self.content_path1, dirs=1, file_size=(2 * 1024),
                                                                  files=10):
                        self.log.error("unable to Generate Data at %s", self.content_path1)
                        raise Exception("unable to Generate Data at {0}".format(self.content_path1))
                    self.log.info("Generated Data at %s", self.content_path1)
                else:
                    self.log.info("Sleeping for 5 seconds before running Synth Full job.")
                    sleep(5)

                # Perform Backup
                if sequence_index < 3:
                    job_copy_list.append(
                        (self.common_util.subclient_backup(sc1_obj, job_types_sequence_list[sequence_index]).job_id,
                         [sp_dedup_primary_obj, dedupe_copy_obj]))

                else:
                    self.common_util.subclient_backup(sc1_obj, job_types_sequence_list[sequence_index])

            # Restore from cloud
            restore_job = sc1_obj.restore_out_of_place(self.client, self.restore_dest_path,
                                                       [self.content_path1])
            self.log.info("restore job [%s] from cloud has started.", restore_job.job_id)
            if not restore_job.wait_for_completion():
                self.log.error("restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                restore_job.delay_reason))
            self.log.info("restore job from cloud [%s] has completed.", restore_job.job_id)

            # Run Aux copy Job
            auxcopy_job = sp_dedup_obj.run_aux_copy()
            self.log.info("Auxcopy job to disk [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error(
                    "Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception(
                    "Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id, auxcopy_job.delay_reason))
            self.log.info("Auxcopy job to disk [%s] has completed.", auxcopy_job.job_id)

            # Run Granular DataAging
            data_aging_job = self.commcell.run_data_aging(storage_policy_name=self.storage_policy_dedupe_name,
                                                          is_granular=True,
                                                          include_all_clients=True)
            self.log.info("Granular Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

            # Restore from disk
            restore_job = sc1_obj.restore_out_of_place(self.client, self.restore_dest_path,
                                                       [self.content_path1], copy_precedence=2)
            self.log.info("restore job from disk [%s] has started.", restore_job.job_id)
            if not restore_job.wait_for_completion():
                self.log.error("restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                restore_job.delay_reason))
            self.log.info("restore job from disk [%s] has completed.", restore_job.job_id)

            # create Non-Dedupe storage policy
            sp_nondedup_obj = self.mmhelper.configure_storage_policy(self.storage_policy_nondedupe_name,
                                                                     self.disk_library_name,
                                                                     self.tcinputs['MediaAgentName'])
            # Set retention of 0day 1cycle
            self.log.info("Setting Retention: 0-days and 1-cycle on Primary Copy")
            sp_nondedup_primary_obj = sp_nondedup_obj.get_copy("Primary")
            sp_nondedup_primary_obj.copy_retention = retention

            # Creating Secondary Copy
            copy2_name = '%s_copy2' % str(self.id)
            nondedupe_copy_obj = self.mmhelper.configure_secondary_copy(copy2_name, self.storage_policy_nondedupe_name,
                                                                        self.cloud_library_name,
                                                                        self.tcinputs['MediaAgentName'])
            # Removing association with System Created Autocopy schedule
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_nondedupe_name, copy2_name)

            # Set retention of 0day 1cycle
            self.log.info("Setting Retention: 0-days and 1-cycle on Secondary Copy")
            nondedupe_copy_obj.copy_retention = retention

            # create subclient
            subclient2_name = "%s_SC2" % str(self.id)
            sc2_obj = self.mmhelper.configure_subclient(self.backupset_name, subclient2_name,
                                                        self.storage_policy_nondedupe_name, self.content_path2,
                                                        self.agent)
            # Allow multiple data readers to subclient
            self.set_multiple_readers(sc2_obj, 4)

            for sequence_index in range(0, 6):
                # Create unique content
                if job_types_sequence_list[sequence_index] is not 'synthetic_full':
                    # Create unique content
                    self.log.info("Generating Data at %s", self.content_path2)
                    if not self.client_machine.generate_test_data(self.content_path2, dirs=1, file_size=(2 * 1024),
                                                                  files=10):
                        self.log.error("unable to Generate Data at %s", self.content_path2)
                        raise Exception("unable to Generate Data at {0}".format(self.content_path2))
                    self.log.info("Generated Data at %s", self.content_path2)
                else:
                    self.log.info("Sleeping for 5 seconds before running Synth Full job.")
                    sleep(5)
                # Perform Backup
                if sequence_index < 3:
                    job_copy_list.append(
                        (self.common_util.subclient_backup(sc2_obj, job_types_sequence_list[sequence_index]).job_id,
                         [sp_nondedup_primary_obj, nondedupe_copy_obj]))
                else:
                    self.common_util.subclient_backup(sc2_obj, job_types_sequence_list[sequence_index])

            # Run Aux copy Job
            auxcopy_job = sp_nondedup_obj.run_aux_copy()
            self.log.info("Auxcopy job [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error("Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception("Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id,
                                                                                auxcopy_job.delay_reason))
            self.log.info("Auxcopy job [%s] has completed.", auxcopy_job.job_id)

            # Restore J6 from non-dedupe cloud copy
            restore_job = sc2_obj.restore_out_of_place(self.client, self.restore_dest_path, [self.content_path2],
                                                       copy_precedence=2)
            self.log.info("restore job of non-dedupe data from cloud [%s] has started.", restore_job.job_id)
            if not restore_job.wait_for_completion():
                self.log.error(
                    "restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception(
                    "restore job [{0}] has failed with {1}.".format(restore_job.job_id, restore_job.delay_reason))
            self.log.info("restore job of non-dedupe data from cloud [%s] has completed.", restore_job.job_id)

            # Run Granular DataAging
            data_aging_job = self.commcell.run_data_aging(storage_policy_name=self.storage_policy_nondedupe_name,
                                                          is_granular=True,
                                                          include_all_clients=True)
            self.log.info("Granular Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

            # Prune Validation
            self.log.info("Pruning Validation Started for non-dedupe storage policy.")
            self._validate_jobs_prune(job_copy_list)
            self.log.info("Pruning Validation completed successfully.")
        except Exception as exp:
            self.log.error('Failed to execute test case with error:%s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear Down function of this test case"""
        self._cleanup()
