# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Reports : Custom Grouping in Charts"""
from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import viewer, builder

from AutomationUtils.cvtestcase import CVTestCase


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    QUERY = """
            DECLARE @tmpTbl TABLE(id int, intMod5 int)
            DECLARE @i int = 0
            WHILE @i < 100
            BEGIN
                SET @i = @i + 1
                INSERT INTO @tmpTbl VALUES(@i, @i % 5)
            END
            SELECT * FROM @tmpTbl
            """
    VERTICAL_BAR_FIELD = "id"
    PIE_CHART_FIELD = "intMod5"
    VERTICAL_BAR_GROUPS = [['Less Than 30', '<30'], ['Greater Than 75', '>75']]
    PIE_CHART_GROUPS = [['Less Than 2', '<2'], ['Equals 3', '3'], ['Greater Than 3', '>3']]
    VERTICAL_BAR = {
        'x-axis_title': 'id',
        'y_axis_title': 'id Count',
        'count': 2,
        'height': ['29', '25'],
        'x-axis_labels': ['Less Than 30', 'Greater Than 75'],
        'y-axis_labels': ['0', '10', '20', '30', '40']
    }
    PIE_CHART = {
        'slice_count': 3,
        'slice_values': ['40', '20', '20']
    }

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports : Custom Grouping in Charts"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.builder = None
        self.dataset = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def add_dataset(self):
        """Adds Dataset"""
        self.builder = builder.ReportBuilder(self.webconsole)
        self.builder.set_report_name(self.name)
        self.dataset = builder.Datasets.DatabaseDataset()
        self.builder.add_dataset(self.dataset)
        self.dataset.set_dataset_name("Automation Dataset")
        self.dataset.set_sql_query(TestCase.QUERY)
        self.dataset.save()

    @test_step
    def create_vertical_bar(self):
        """Creates Vertical Bar chart"""
        vertical_bar = builder.VerticalBar("Automation Chart 1")
        self.builder.add_component(vertical_bar, self.dataset)
        vertical_bar.set_x_axis(TestCase.VERTICAL_BAR_FIELD)
        vertical_bar.add_custom_group(TestCase.VERTICAL_BAR_FIELD, TestCase.VERTICAL_BAR_GROUPS)
        details = vertical_bar.get_chart_details()
        SQLQueries.validate_equality(TestCase.VERTICAL_BAR, details,
                                     err_msg="Expected and received values are not equal in Vertical Bar")

    @test_step
    def create_pie_chart(self):
        """Creates Pie chart"""
        pie = builder.PieChart("Automation Chart 2")
        self.builder.add_component(pie, self.dataset)
        pie.add_column_to_dimension(TestCase.PIE_CHART_FIELD)
        pie.add_custom_group(TestCase.PIE_CHART_FIELD, TestCase.PIE_CHART_GROUPS)
        details = pie.get_chart_details()
        SQLQueries.validate_equality(TestCase.PIE_CHART, details,
                                     err_msg="Expected and received values are not equal in Pie Chart")

    def validate_viewer(self):
        """Validates chart in viewer"""
        self.builder.save(deploy=True)
        self.builder.open_report()
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        vertical_bar = viewer.VerticalBar("Automation Chart 1")
        pie_chart = viewer.PieChart("Automation Chart 2")

        report_viewer.associate_component(vertical_bar)
        report_viewer.associate_component(pie_chart)

        details = vertical_bar.get_chart_details()
        SQLQueries.validate_equality(TestCase.VERTICAL_BAR, details,
                                     err_msg="Expected and received values are not equal in Vertical Bar")

        details = pie_chart.get_chart_details()
        SQLQueries.validate_equality(TestCase.PIE_CHART, details,
                                     err_msg="Expected and received values are not equal in Pie Chart")

    def run(self):
        try:
            self.init_tc()
            self.add_dataset()
            self.create_vertical_bar()
            self.create_pie_chart()
            self.validate_viewer()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
