# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Report: Daterange Component"""
import calendar
import datetime

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.report_templates import DefaultReport
from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import (
    builder,
    viewer
)


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    CUSTOM_RANGES = {
        "Last 2 Month": ["1", "5"],
        "Last 2 Year": ["1", "2", "5", "6", "7"],
        "Last 1 hour": ["1"]}

    QUERY = """  DECLARE @currentTime DATETIME = DATEADD(MINUTE, -2, GETDATE())
                 DECLARE @result TABLE(id INT IDENTITY(1,1), dt DATETIME)

                 INSERT INTO @result VALUES(@currentTime)
                 INSERT INTO @result VALUES(DATEADD(YEAR, -1, @currentTime))
                 INSERT INTO @result VALUES(DATEADD(YEAR, -2, @currentTime))
                 INSERT INTO @result VALUES(DATEADD(YEAR, -3, @currentTime))
                 INSERT INTO @result VALUES(DATEADD(MONTH, -1, @currentTime))
                 INSERT INTO @result VALUES(DATEADD(MONTH, -2, @currentTime))
                 INSERT INTO @result VALUES(DATEADD(MONTH, -3, @currentTime))
                 SELECT *
                 FROM @result"""

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Report: Daterange Component"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.table = None
        self.date_range = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def add_custom_range_to_date_range_component(self):
        """Adds custom range to the properties panel and validates in builder"""
        report = DefaultReport(self.utils)
        report.build_default_report(sql=TestCase.QUERY, open_report=False)

        date_range_name = "Automation DateRange"
        self.date_range = builder.DateRange(date_range_name)
        report.report_builder.add_component(self.date_range, report.dataset)

        self.date_range.add_custom_range("Last 2 Year", ">-2y")
        self.date_range.add_custom_range("Last 2 Month", ">-2M")
        self.date_range.add_column_from_dataset("dt")
        self.table = report.table

        self.log.info("*****************Validate in Builder*****************")
        self.filter_range()
        self.relative_range()
        report.report_builder.save()
        report.report_builder.deploy()
        report.report_builder.open_report()

        self.log.info("*****************Validate in Viewer*****************")
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        self.date_range = viewer.DateRange(date_range_name)
        self.table = viewer.DataTable(report.table.title)
        report_viewer.associate_component(self.date_range)
        report_viewer.associate_component(self.table)

    @test_step
    def filter_range(self):
        """Filter the daterange component using the added custom range"""
        for range_, expected_result in TestCase.CUSTOM_RANGES.items():
            self.date_range.set_predefined_range(range_)
            id_ = self.table.get_table_data().get('id')
            SQLQueries.validate_list_equality(expected_result, id_)

    @test_step
    def relative_range(self):
        """Filter using relative range"""
        end_date = (datetime.datetime.now() + datetime.timedelta(days=1)).strftime("%m/%d/%Y").split("/")
        start_date = (datetime.datetime.now() + datetime.timedelta(days=-3)).strftime("%m/%d/%Y").split("/")
        month = dict((k, v) for k, v in enumerate(calendar.month_abbr))
        start_date[0] = month[int(start_date[0])]
        end_date[0] = month[int(end_date[0])]
        start_date = " ".join(start_date)
        end_date = " ".join(end_date)
        self.date_range.set_custom_range(start_date, end_date)

        id_ = self.table.get_table_data().get('id')
        SQLQueries.validate_list_equality(TestCase.CUSTOM_RANGES["Last 1 hour"], id_)

    def run(self):
        try:
            self.init_tc()
            self.add_custom_range_to_date_range_component()
            self.filter_range()
            self.relative_range()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
