# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Button Panel in Custom Reports"""
import ast

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.report_templates import DefaultReport
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import viewer


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    COLOR = "yellow"
    CSS_VALUE = "rgba(255, 255, 0, 1)"
    IMAGE = "/webconsole/OEM/1/images/commcell.ico"
    ENABLE_OPTION_EXPRESSION = "=false"
    CUSTOM_CLASS = "btn btn-danger"
    BUTTON_EXPRESSION = "alert(JSON.stringify(selectedRows))"
    EXPECTED_ROWS = [{'id': 2,
                      'text_t': 'Text00000016',
                      'datetime_t': 'Oct 2, 1992, 07:36:40 AM',
                      'datetime_t_formatted': 'Oct 2, 1992, 07:36:40 AM',
                      'timestamp_t': 718011400
                      },
                     {'id': 5,
                      'text_t': 'Text00000040',
                      'datetime_t': 'Oct 2, 1992, 09:16:40 AM',
                      'datetime_t_formatted': 'Oct 2, 1992, 09:16:40 AM',
                      'timestamp_t': 718017400
                      }
                     ]

    EXPECTED_ROW = [{'id': 3,
                     'text_t': 'Text00000024',
                     'datetime_t': 'Oct 2, 1992, 08:10:00 AM',
                     'datetime_t_formatted': 'Oct 2, 1992, 08:10:00 AM',
                     'timestamp_t': 718013400
                     }
                    ]

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Button Panel in Custom Reports"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.button = None
        self.report = None
        self.button_viewer = None
        self.report_viewer = None
        self.table_viewer = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()

            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
            self.report = DefaultReport(self.utils)
            self.report.build_default_report(open_report=False)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def add_button_and_set_properties(self):
        """Adds button to the table and sets properties of the button"""
        # Add Button
        self.button = self.report.table.Button("Button 1")
        self.report.table.toggle_button_panel()
        self.report.table.add_button(self.button)

        # Set Image
        self.button.set_image("url", TestCase.IMAGE)
        self.button.set_custom_class(TestCase.CUSTOM_CLASS)

        # Toggle row selection and multiple selection of rows
        self.report.table.toggle_row_selection()
        self.report.table.toggle_multiple_row_selection()

        # Set style
        self.button.set_button_style("color: %s;" % TestCase.COLOR)

        # Set on click expression
        self.button.set_expression(TestCase.BUTTON_EXPRESSION)

        self.save_deploy_open_report()

    @test_step
    def verify_image_on_button(self):
        """Verify image on button."""
        button_img_src = self.button.get_button_image_source()
        if TestCase.IMAGE not in button_img_src:
            raise CVTestStepFailure(f"Expected image source: {TestCase.IMAGE}.\n Actual image source: {button_img_src}")

    @test_step
    def verify_button_style(self):
        """Verify button style."""
        button_color = self.button.get_button_color()
        if button_color != TestCase.CSS_VALUE:
            raise CVTestStepFailure(f"Expected Color is {TestCase.CSS_VALUE}. \n Actual Button color is {button_color}")

    @test_step
    def verify_row_selection(self):
        """Verify row selection."""
        self.switch_to_builder()
        self.button.enable_option(self.button.EnableOption.CUSTOM, TestCase.ENABLE_OPTION_EXPRESSION)
        self.save_deploy_open_report()
        if self.button_viewer.is_button_enabled():
            raise CVTestStepFailure("Button is enabled when script is set to disable button.")

        self.switch_to_builder()
        self.button.enable_option(self.button.EnableOption.ALWAYS)
        self.save_deploy_open_report()
        if not self.button_viewer.is_button_enabled():
            raise CVTestStepFailure("Button is disabled when always enabled option is selected.")

        self.switch_to_builder()
        self.button.enable_option(self.button.EnableOption.SINGLE_SELECT)
        self.save_deploy_open_report()
        self.table_viewer.select_rows([1, 2])
        if self.button_viewer.is_button_enabled():
            raise CVTestStepFailure("Button is enabled on selecting more than one row for the singleSelect option.")

        self.switch_to_builder()
        self.table_viewer.select_rows([1])
        self.button.enable_option(self.button.EnableOption.MULTI_SELECT)
        self.save_deploy_open_report()
        self.table_viewer.select_rows([1, 2, 3])
        if not self.button.is_button_enabled():
            raise CVTestStepFailure("Button is not enabled on selecting more than one or more rows.")
        self.table_viewer.select_rows([1, 2, 3])

    @test_step
    def verify_checkbox_selected_rows(self):
        """Verify checkbox selected rows"""
        self.table_viewer.select_rows([2, 5])
        self.button_viewer.click_button()
        list_ = self.webconsole.get_all_unread_notifications()
        temp_dict = (ast.literal_eval(list_[0][1:-1]))
        for dict_ in temp_dict:
            dict_.pop("sys_rowid", None)
            dict_.pop("DataSource", None)
            dict_.pop("$$hashKey", None)
        list_ = list(temp_dict)
        if list_ != TestCase.EXPECTED_ROWS:
            raise CVTestStepFailure(f"Button click returned the rows {str(list_)}.\n "
                                    f"Expected rows: {TestCase.EXPECTED_ROWS}")
        self.switch_to_builder()

    @test_step
    def verify_radiobutton_selected_rows(self):
        """Verify radiobutton selected rows"""
        self.report.table.toggle_multiple_row_selection()
        self.save_deploy_open_report()
        self.table_viewer.select_rows([3])
        self.button_viewer.click_button()
        list_ = self.webconsole.get_all_unread_notifications()
        temp_dict = (ast.literal_eval(list_[0][1:-1]))
        temp_dict.pop("sys_rowid", None)
        temp_dict.pop("DataSource", None)
        temp_dict.pop("$$hashKey", None)
        list_ = [temp_dict]
        if list_ != TestCase.EXPECTED_ROW:
            raise CVTestStepFailure(f"Button click returned the row: {str(list_)}.\n"
                                    f"Expected row: {TestCase.EXPECTED_ROW}")

    def switch_to_builder(self):
        """Closes current tab and switches to builder"""
        self.browser.driver.close()
        self.browser.driver.switch_to_window(self.browser.driver.window_handles[-1])

    def save_deploy_open_report(self):
        """Saves deploys and opens report"""
        self.report.report_builder.save()
        self.report.report_builder.deploy()
        self.report.report_builder.open_report()

        self.table_viewer = viewer.DataTable("Automation Table")
        self.button_viewer = viewer.DataTable.Button("Button 1")
        self.report_viewer = viewer.CustomReportViewer(self.webconsole)
        self.report_viewer.associate_component(self.table_viewer)
        self.table_viewer.associate_button(self.button_viewer)

    def run(self):
        try:
            self.init_tc()
            self.add_button_and_set_properties()
            self.verify_image_on_button()
            self.verify_button_style()
            self.verify_row_selection()
            self.verify_checkbox_selected_rows()
            self.verify_radiobutton_selected_rows()
        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
