# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase to perform Inline & Parallel copy case for Aux copy

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                  --  initialize TestCase class

    _validate_inline_copy()     --  validates inline copy functionality by checking for job status on secondary copy in
                                    JMJobDataStats

    _get_stream_reader_id()     --  gets StreamReaderId is used for reading for both copies in ArchJobStreamStatus

    _validate_parallel_copy()   --  validates inline copy functionality by checking that same StreamReaderId is used for
                                    reading for both copies in ArchJobStreamStatus

    _cleanup()                  --  cleanup the entities created

    setup()                     --  setup function of this test case

    run()                       --  run function of this test case

    tear_down()                 --  teardown function of this test case
"""

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.idautils import CommonUtils
from MediaAgents.MAUtils.mahelper import (MMHelper, DedupeHelper)


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = "Inline & Parallel copy case for Aux copy"
        self.tcinputs = {
            "MediaAgentName": None
        }
        self.disk_library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.mountpath = None
        self.partition_path = None
        self.content_path = None
        self.dedupehelper = None
        self.library_name = None
        self.mmhelper = None
        self.common_util = None
        self.ma_machine = None
        self.client_machine = None

    def _validate_inline_copy(self, job_id, copy_id):
        """
        validates inline copy functionality by checking for job status on secondary copy in JMJobDataStats
        Args:
            job_id (int) -- backup job id to check in table

            copy_id (int) -- copy id to validate on
        Return:
            (Bool) True if status of job is 100
            (Bool) False if not
        """

        query = """select status from JMJobDataStats where jobId = {0} and archGrpCopyId = {1}
        """.format(job_id, copy_id)
        self.log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self.log.info("RESULT: {0}".format(cur))
        if cur != ['']:
            if int(cur[0]) == 100:
                return True
        return False

    def _get_stream_reader_id(self, copy_id):
        """
        Gets StreamReaderId is used for reading for both copies in ArchJobStreamStatus
        Args:
            copy_id (list) -- copy id
        Return:
            list of reader id
        """

        query = """select distinct StreamReaderId from ArchJobStreamStatus where DestCopyId = {0}""".format(copy_id)
        self.log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self.log.info("RESULT: {0}".format(cur))
        if cur != ['']:
            return cur
        self._log.error("No entries present")
        raise Exception("Unable to fetch reader id")

    def _validate_parallel_copy(self, copy_id_list):
        """
        validates inline copy functionality by checking that same StreamReaderId is used for reading for both copies
            in ArchJobStreamStatus
        Args:
            copy_id_list (list) -- list of copy id to validate on
        Return:
            (Bool) True if StreamReaderIds are same
            (Bool) False if not
        """

        if self._get_stream_reader_id(copy_id_list[0]) == self._get_stream_reader_id(copy_id_list[1]):
            return True
        return False

    def _cleanup(self):
        """Cleanup the entities created"""

        self.log.info("********************** CLEANUP STARTING *************************")
        try:
            # Delete bkupset
            self.log.info("Deleting BackupSet: %s if exists", self.backupset_name)
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
                self.log.info("Deleted BackupSet: %s", self.backupset_name)

            # Delete Storage Policy
            self.log.info("Deleting storage policy: %s if exists", self.storage_policy_name)
            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.commcell.storage_policies.delete(self.storage_policy_name)
                self.log.info("Deleted storage policy: %s", self.storage_policy_name)

            # Delete Library
            self.log.info("Deleting library: %s if exists", self.disk_library_name)
            if self.commcell.disk_libraries.has_library(self.disk_library_name):
                self.commcell.disk_libraries.delete(self.disk_library_name)
                self.log.info("Deleted library: %s", self.disk_library_name)

            # Run DataAging
            data_aging_job = self.commcell.run_data_aging()
            self.log.info("Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

        except Exception as exp:
            self.log.error("Error encountered during cleanup : %s", str(exp))
            raise Exception("Error encountered during cleanup: {0}".format(str(exp)))

        self.log.info("********************** CLEANUP COMPLETED *************************")

    def setup(self):
        """Setup function of this test case"""

        options_selector = OptionsSelector(self.commcell)
        self.disk_library_name = '%s_library' % str(self.id)
        self.storage_policy_name = '%s_storage_policy' % str(self.id)
        self.backupset_name = '%s_BS' % str(self.id)
        self.subclient_name = '%s_SC' % str(self.id)

        self._cleanup()

        self.ma_machine = Machine(self.tcinputs['MediaAgentName'], self.commcell)
        self.client_machine = Machine(self.client)

        # To select drive with space available in ma machine
        self.log.info('Selecting drive in the MA machine based on space available')
        ma_drive = options_selector.get_drive(self.ma_machine, size=(10 * 1024))
        if ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', ma_drive)

        # To select drive with space available in client machine
        self.log.info('Selecting drive in the client machine based on space available')
        client_drive = options_selector.get_drive(self.client_machine, size=(10 * 1024))
        if client_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', client_drive)

        self.mountpath = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'MP')

        self.partition_path = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'DDB')

        self.content_path = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'TestData')
        if self.client_machine.check_directory_exists(self.content_path):
            self.client_machine.remove_directory(self.content_path)
        self.client_machine.create_directory(self.content_path)

        self.mmhelper = MMHelper(self)
        self.dedupehelper = DedupeHelper(self)
        self.common_util = CommonUtils(self)

    def run(self):
        """Run function of this test case"""

        try:
            self.mmhelper.configure_disk_library(self.disk_library_name, self.tcinputs['MediaAgentName'],
                                                           self.mountpath)
            sp_dedup_obj = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_name,
                                                                             self.disk_library_name,
                                                                             self.tcinputs['MediaAgentName'],
                                                                             self.partition_path)

            # Enable MA side dedupe  on client
            self.client.set_dedup_property('clientSideDeduplication', 'OFF')

            # Set retention of 0day 1cycle
            self.log.info("Setting Retention on Primary Copy")
            sp_dedup_primary_obj = sp_dedup_obj.get_copy("Primary")
            retention = (1, 1, -1)
            sp_dedup_primary_obj.copy_retention = retention

            # Enable encryption on Primary copy
            encryption = (True, 'BLOWFISH', 128, False)
            sp_dedup_primary_obj.copy_reencryption = encryption

            # create first non-dedupe secondary copy
            copy1_non_dedupe = '%s_copy1_nondedupe' % str(self.id)
            copy1_non_dedupe_obj = self.mmhelper.configure_secondary_copy(copy1_non_dedupe, self.storage_policy_name,
                                                                          self.disk_library_name,
                                                                          self.tcinputs['MediaAgentName'])
            # Enable parallel copy
            copy1_non_dedupe_obj.set_parallel_copy(True)

            # Enable inline copy
            copy1_non_dedupe_obj.set_inline_copy(True)

            # create first non-dedupe secondary copy
            copy2_non_dedupe = '%s_copy2_nondedupe' % str(self.id)
            copy2_non_dedupe_obj = self.mmhelper.configure_secondary_copy(copy2_non_dedupe, self.storage_policy_name,
                                                                          self.disk_library_name,
                                                                          self.tcinputs['MediaAgentName'])

            # Removing association with System Created Automatic Auxcopy schedule
            self.log.info("Removing association with System Created Autocopy schedule on above created copy")
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_name, copy1_non_dedupe)
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_name, copy2_non_dedupe)

            # Enable parallel copy
            copy2_non_dedupe_obj.set_parallel_copy(True)

            # Enable inline copy
            copy2_non_dedupe_obj.set_inline_copy(True)

            self.backupset = self.mmhelper.configure_backupset(self.backupset_name, self.agent)
            sc_obj = self.mmhelper.configure_subclient(self.backupset_name, self.subclient_name,
                                                       self.storage_policy_name, self.content_path, self.agent)

            # Allow multiple data readers to subclient
            self.log.info("Setting Data Readers=10 on Subclient")
            sc_obj.allow_multiple_readers = True
            sc_obj.data_readers = 10

            # Create unique content
            self.log.info("Generating Data at %s", self.content_path)
            if not self.client_machine.generate_test_data(self.content_path, dirs=1, file_size=(100 * 1024),
                                                          files=10):
                self.log.error("unable to Generate Data at %s", self.content_path)
                raise Exception("unable to Generate Data at {0}".format(self.content_path))
            self.log.info("Generated Data at %s", self.content_path)

            # Backup Job J1
            job1_obj = self.common_util.subclient_backup(sc_obj, "full")

            # Backup Job J2
            job2_obj = self.common_util.subclient_backup(sc_obj, "full")

            # Validate inline copy functionality

            #   validating inline copy for job1 on copy1
            if self._validate_inline_copy(int(job1_obj.job_id), int(copy1_non_dedupe_obj.copy_id)):
                self.log.info("Inline copy for 1st Backup on copy1 was success")
            else:
                self.log.error("Inline copy for 1st Backup on copy1 failed")
                raise Exception("Inline copy for 1st Backup on copy1 failed")

            #   validating inline copy for job1 on copy2
            if self._validate_inline_copy(int(job1_obj.job_id), int(copy2_non_dedupe_obj.copy_id)):
                self.log.info("Inline copy for 1st Backup on copy2 was success")
            else:
                self.log.error("Inline copy for 1st Backup on copy2 failed")
                raise Exception("Inline copy for 1st Backup on copy2 failed")

            #   validating inline copy for job2 on copy1
            if self._validate_inline_copy(int(job2_obj.job_id), int(copy1_non_dedupe_obj.copy_id)):
                self.log.info("Inline copy for 2st Backup on copy1 was success")
            else:
                self.log.error("Inline copy for 2st Backup on copy1 failed")
                raise Exception("Inline copy for 2st Backup on copy1 failed")

            #   validating inline copy for job2 on copy2
            if self._validate_inline_copy(int(job2_obj.job_id), int(copy2_non_dedupe_obj.copy_id)):
                self.log.info("Inline copy for 2st Backup on copy2 was success")
            else:
                self.log.error("Inline copy for 2st Backup on copy2 failed")
                raise Exception("Inline copy for 2st Backup on copy2 failed")

            # picking jobs for recopy on both copies and running parallel copy

            #   Picking jobs J1 and J2 for recopy on copy1
            copy1_non_dedupe_obj.recopy_jobs(job1_obj.job_id + ", " + job2_obj.job_id)

            #   Picking jobs J1 and J2 for recopy on copy2
            copy2_non_dedupe_obj.recopy_jobs(job1_obj.job_id + ", " + job2_obj.job_id)

            # Run Aux copy Job
            auxcopy_job = sp_dedup_obj.run_aux_copy()
            self.log.info("Auxcopy job [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error("Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception("Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id,
                                                                                auxcopy_job.delay_reason))
            self.log.info("Auxcopy job [%s] has completed.", auxcopy_job.job_id)

            # Validate parallel copy functionality

            if self._validate_parallel_copy([copy1_non_dedupe_obj.copy_id, copy2_non_dedupe_obj.copy_id]):
                self.log.info("SUCCESS  Result :Pass for Parallel copy as streamReaderId for both copies is same")
            else:
                self.log.error("Fail for parallel copy as Stream Readers did not match for both copies")
                raise Exception("Fail for parallel copy as Stream Readers did not match for both copies")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear Down function of this test case"""

        # revert back dedupe default setting on client
        self.client.set_dedup_property('clientSideDeduplication', 'USE_SPSETTINGS')

        if self.status != constants.FAILED:
            self.log.info("Testcase shows successful execution, cleaning up the test environment ...")
            self._cleanup()
        else:
            self.log.error("Testcase shows failure in execution, not cleaning up the test environment ...")

        if self.client_machine.check_directory_exists(self.content_path):
            self.client_machine.remove_directory(self.content_path)
