# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Custom report: Remote DB registration """

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom.data_sources import (
    MySQLDataSource,
    OracleDataSource,
    SQLServerDataSource
)

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.utils import CustomReportUtils


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom report: Remote DB registration"
        self.browser = None
        self.webconsole = None
        self.mysql = None
        self.oracle = None
        self.sqlserver = None
        self.utils = CustomReportUtils(self)
        self.ds_name = {
            "mysql": "Automation DS MySQL",
            "oracle": "Automation DS Oracle",
            "sqlserver": "Automation DS SQL Server"
        }

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            navigator = Navigator(self.webconsole)
            navigator.goto_datasources_configuration()
            self.mysql = MySQLDataSource(self.webconsole)
            self.oracle = OracleDataSource(self.webconsole)
            self.sqlserver = SQLServerDataSource(self.webconsole)
            self.cleanup()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def cleanup(self):
        """Cleans up existing data sources created by automation"""
        if self.ds_name["mysql"] in self.mysql.get_data_source_names():
            self.mysql.delete_data_source(self.ds_name["mysql"])
        if self.ds_name["oracle"] in self.oracle.get_data_source_names():
            self.oracle.delete_data_source(self.ds_name["oracle"])
        if self.ds_name["sqlserver"] in self.sqlserver.get_data_source_names():
            self.sqlserver.delete_data_source(self.ds_name["sqlserver"])

    @test_step
    def register_data_sources(self):
        """Registers data source."""
        self.mysql.add_data_source(self.ds_name["mysql"], self._tcinputs["hostname"],
                                   "commvault", self._tcinputs["DB_user_name"],
                                   self._tcinputs["DB_password"])
        self.oracle.add_data_source(self.ds_name["oracle"], self._tcinputs["hostname"],
                                    "commvault", self._tcinputs["DB_user_name"],
                                    self._tcinputs["DB_password"])
        self.sqlserver.add_data_source(self.ds_name["sqlserver"], self._tcinputs["Sqlhostname"],
                                       "commvault", self._tcinputs["DB_user_name"],
                                       self._tcinputs["DB_password"])

    @test_step
    def check_reflected_changes(self):
        """Checks whether the added data sources in shown up in the list."""
        if self.ds_name["mysql"] not in self.mysql.get_data_source_names():
            raise CVTestStepFailure("MySQL data source is not listed")
        if self.ds_name["oracle"] not in self.oracle.get_data_source_names():
            raise CVTestStepFailure("Oracle data source is not listed")
        if self.ds_name["sqlserver"] not in self.sqlserver.get_data_source_names():
            raise CVTestStepFailure("SQLServer data source is not listed")

    @test_step
    def delete_data_sources(self):
        """Deletes the data source."""
        self.mysql.delete_data_source(self.ds_name["mysql"])
        self.oracle.delete_data_source(self.ds_name["oracle"])
        self.sqlserver.delete_data_source(self.ds_name["sqlserver"])

        if self.ds_name["mysql"] in self.mysql.get_data_source_names():
            raise CVTestStepFailure("MySQL data source '{}'is listed even after deleting".
                                    format(self.ds_name["mysql"]))
        if self.ds_name["oracle"] in self.oracle.get_data_source_names():
            raise CVTestStepFailure("Oracle data source '{}'is listed even after deleting".
                                    format(self.ds_name["oracle"]))
        if self.ds_name["sqlserver"] in self.sqlserver.get_data_source_names():
            raise CVTestStepFailure("SQLServer data source '{}'is listed even after deleting".
                                    format(self.ds_name["sqlserver"]))

    def run(self):
        try:
            self.init_tc()
            self.register_data_sources()
            self.check_reflected_changes()
            self.delete_data_sources()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)