# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Reports: Exports with post query filter"""

from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils
from Reports.Custom.report_templates import DefaultReport

from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom.builder import ReportBuilder

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import builder
from Web.WebConsole.Reports.Custom import viewer

from AutomationUtils.cvtestcase import CVTestCase


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Exports with post query filter"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.export = None
        self.utils = CustomReportUtils(self)
        self.report_builder = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.set_downloads_dir(self.utils.get_temp_dir())
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()

            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def set_post_query_filter(self):
        """Creates join dataset from two database datasets"""
        # Build default report
        self.report_builder = ReportBuilder(self.webconsole)
        default_report = DefaultReport(self.utils)
        default_report.build_default_report(open_report=False, sql=SQLQueries.sql_server_q1(10))

        # Set Post query filter
        dataset_name = default_report.dataset.dataset_name
        dataset = builder.Datasets.DatabaseDataset()
        dataset.dataset_name = dataset_name
        self.report_builder.edit_dataset(dataset)
        dataset.set_post_query_filter("select * from $this$ where id % 2 = 0")
        dataset.save()

        self.report_builder.save()
        self.report_builder.deploy()

    @test_step
    def export_to_csv(self):
        """Exports to CSV"""
        # Initialize export handler and verify table data
        self.report_builder.open_report()
        self.utils.reset_temp_dir()
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        self.export = report_viewer.export_handler()
        table_1 = viewer.DataTable("Automation Table")
        report_viewer.associate_component(table_1)
        data = {'id': ['2', '4', '6', '8', '10'],
                'text_t': ['Text00000016', 'Text00000032', 'Text00000048',
                           'Text00000064', 'Text00000080'],
                'datetime_t': ['Oct 2, 1992, 07:36:40 AM', 'Oct 2, 1992, 08:43:20 AM',
                               'Oct 2, 1992, 09:50:00 AM', 'Oct 2, 1992, 10:56:40 AM',
                               'Oct 2, 1992, 12:03:20 PM'],
                'timestamp_t': ['718011400', '718015400', '718019400', '718023400', '718027400']}
        SQLQueries.validate_equality(data, table_1.get_table_data())

        # Export to CSV
        self.export.to_csv()
        self.utils.wait_for_file_to_download('csv')
        self.utils.validate_tmp_files("csv")
        self.log.info("csv export completed successfully")
        exported_file = self.utils.get_temp_files("csv")[0]
        csv_data = self.utils.get_as_table_formatted_csv(exported_file, strip_title=True)
        SQLQueries.validate_equality(data, csv_data)

    @test_step
    def export_to_html(self):
        """Exports to HTML"""
        # Export to HTML
        self.export.to_html()
        self.utils.wait_for_file_to_download('html')
        self.utils.validate_tmp_files("html")
        self.log.info("html export completed successfully")

        # Validate table in  HTML content
        path = self.utils.get_temp_files("html")
        self.browser.driver.execute_script("window.open()")
        self.browser.driver.switch_to_window(self.browser.driver.window_handles[-1])
        self.browser.driver.get(path[0])
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        table_1 = viewer.DataTable("Automation Table")
        report_viewer.associate_component(table_1)
        data = {'id': ['2', '4', '6', '8', '10'],
                'text_t': ['Text00000016', 'Text00000032', 'Text00000048',
                           'Text00000064', 'Text00000080'],
                'datetime_t': ['Oct 2, 1992, 07:36:40 AM', 'Oct 2, 1992, 08:43:20 AM',
                               'Oct 2, 1992, 09:50:00 AM', 'Oct 2, 1992, 10:56:40 AM',
                               'Oct 2, 1992, 12:03:20 PM'],
                'timestamp_t': ['718011400', '718015400', '718019400', '718023400', '718027400']}
        SQLQueries.validate_equality(data, table_1.get_table_data())
        self.browser.driver.close()
        self.browser.driver.switch_to_window(self.browser.driver.window_handles[-1])

    def run(self):
        try:
            self.init_tc()
            self.set_post_query_filter()
            self.export_to_csv()
            self.export_to_html()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
