# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase
from VirtualServer.VSAUtils import OptionsHelper, VirtualServerUtils
from AutomationUtils import constants


class TestCase(CVTestCase):
    """Class for executing Basic acceptance Test of VSA backup and CBT Restore test case"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "VSA VMware Full/Incremental Backup & CBT Restore"
        self.product = self.products_list.VIRTUALIZATIONVMWARE
        self.feature = self.features_list.DATAPROTECTION
        self.test_individual_status = True
        self.test_individual_failure_message = ""
        self.show_to_user = True
        self.tcinputs = {}

    def run(self):
        """Main function for test case execution"""

        try:
            auto_subclient = VirtualServerUtils.subclient_initialize(self)
            VirtualServerUtils.decorative_log("Backup")
            backup_options = OptionsHelper.BackupOptions(auto_subclient)
            backup_options.backup_type = "FULL"
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")
            auto_subclient.backup(backup_options)
            backup_options.backup_type = "INCREMENTAL"
            auto_subclient.backup(backup_options)

            VirtualServerUtils.decorative_log("Copy Test Data after backup")
            #We're not running Differential backup, but we are copying a Folder called Differential.
            #After Restore, it should not exist in the VM.
            backup_options.backup_type = "DIFFERENTIAL"
            auto_subclient.backup_folder_name = backup_options.backup_type
            auto_subclient.vsa_discovery(backup_options, {})

            try:
                VirtualServerUtils.decorative_log("Inplace full vm restore")
                vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, self)
                if "browse_ma" in self.tcinputs:
                    VirtualServerUtils.set_inputs(self.tcinputs, vm_restore_options)
                vm_restore_options.power_on_after_restore = True
                vm_restore_options.in_place_overwrite = True
                auto_subclient.virtual_machine_restore(vm_restore_options)
            except Exception as exp:
                raise Exception ("Exception during Full VM In-place Restore")

            try:
                VirtualServerUtils.decorative_log("CBT Restore Validation")
                self.log.info("Restore Proxy: %s" % auto_subclient.restore_proxy_client)
                auto_subclient.verify_cbt_restore(vm_restore_options.restore_job, auto_subclient.restore_proxy_client)
                for _vm in auto_subclient.vm_list:
                    auto_subclient.verify_data_pruned(_vm)
            except Exception as exp:
                raise Exception ("CBT Validation Failed")

        except Exception as exp:
            self.log.error('Failed with error: {0}'.format(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            try:
                auto_subclient.cleanup_testdata(backup_options)
                auto_subclient.post_restore_clean_up(vm_restore_options)
            except Exception:
                self.log.warning("Testcase and/or Restored vm cleanup was not completed")
                pass
            if not self.test_individual_status:
                self.result_string = self.test_individual_failure_message
                self.status = constants.FAILED
