# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Report: Validate HTTP POST datasets"""
from base64 import b64encode

from AutomationUtils import config
from AutomationUtils.cvtestcase import CVTestCase
from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils
from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom import viewer, builder
from Web.WebConsole.Reports.Custom.inputs import String, TextBoxController
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.webconsole import WebConsole


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    HTTP_CONTENT = """<DM2ContentIndexing_CheckCredentialReq mode="Webconsole" username="%s" password="%s" />"""
    API = "SearchSvc/CVWebService.svc/Login"
    EXPECTED_ROW_EXPRESSION = "/DM2ContentIndexing_CheckCredentialResp"
    POST_QUERY_FILTER = "SELECT * FROM $this$ UNION ALL SELECT * FROM $this$"
    CONSTANTS = config.get_config()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Report: Validate HTTP POST datasets"
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.report_builder = None
        self.dataset = None
        self.EXPECTED = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            name = self.webconsole.get_login_name().strip()
            self.EXPECTED = {'aliasName': ['1', '1'], 'capability': ['4328650440704', '4328650440704'],
                             'userName': [name, name],
                             'ownerOrganization_providerDomainName': ['Commcell', 'Commcell'],
                             'providerOrganization_providerDomainName': ['Commcell', 'Commcell']}
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def create_http_dataset_with_post_query(self):
        """Creates a HTTP dataset via REST"""
        self.report_builder = builder.ReportBuilder(self.webconsole)
        self.report_builder.set_report_name(self.name)
        self.dataset = builder.HTTPDataset()
        self.report_builder.add_dataset(self.dataset)
        self.dataset.set_dataset_name("Automation Dataset")
        self.dataset.set_post(
            TestCase.API,
            TestCase.HTTP_CONTENT %
            (TestCase.CONSTANTS.ADMIN_USERNAME, b64encode(TestCase.CONSTANTS.ADMIN_PASSWORD.encode()).decode()),
            json_content_type=False,
            json_accept_type=False
        )
        self.dataset.get_preview_data()
        row_expression = self.dataset.get_row_expression()

        if row_expression != TestCase.EXPECTED_ROW_EXPRESSION:
            raise CVTestStepFailure("Expected Row Expression: %s \n Actual Row Expression: %s"
                                    % (TestCase.EXPECTED_ROW_EXPRESSION, row_expression))

        self.dataset.save()

    @test_step
    def set_username_and_password_via_input(self):
        """Sets username and password from input"""
        username = String("Username")
        textbox_1 = TextBoxController("Username")
        self.report_builder.add_input(username)
        username.add_html_controller(textbox_1)
        username.set_required()
        username.save()

        password = String("Password")
        textbox_2 = TextBoxController("Password")
        self.report_builder.add_input(password)
        username.add_html_controller(textbox_2)
        username.set_required()
        password.save()

        table = builder.DataTable("Automation Table")
        self.report_builder.add_component(table, self.dataset)
        table.add_column_from_dataset("aliasName")
        table.add_column_from_dataset("capability")
        table.add_column_from_dataset("userName")
        table.add_column_from_dataset("ownerOrganization_providerDomainName")
        table.add_column_from_dataset("providerOrganization_providerDomainName")
        self.report_builder.edit_dataset(self.dataset)
        self.dataset.set_post(
            TestCase.API,
            TestCase.HTTP_CONTENT %
            ("@username", "@password"),
            json_content_type=False,
            json_accept_type=False
        )
        self.dataset.set_post_query_filter(TestCase.POST_QUERY_FILTER)
        self.dataset.add_parameter("username", username)
        self.dataset.add_parameter("password", password)
        self.dataset.save()
        self.report_builder.save(deploy=True)
        self.report_builder.refresh()
        textbox_1.set_textbox_controller(TestCase.CONSTANTS.ADMIN_USERNAME)
        textbox_2.set_textbox_controller(b64encode(TestCase.CONSTANTS.ADMIN_PASSWORD.encode()).decode())
        textbox_2.set_textbox_controller(
            b64encode(TestCase.CONSTANTS.ADMIN_PASSWORD.encode()).decode() + '\t')
        # textbox_1.apply()
        data = table.get_table_data()
        SQLQueries.validate_equality(self.EXPECTED, data)
        self.report_builder.open_report()

    @test_step
    def validate(self):
        """Validating contents on viewer"""
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        textbox_1 = TextBoxController("Username")
        textbox_2 = TextBoxController("Password")
        report_viewer.associate_input(textbox_1)
        report_viewer.associate_input(textbox_2)
        textbox_1.set_textbox_controller(TestCase.CONSTANTS.ADMIN_USERNAME)
        textbox_2.set_textbox_controller(b64encode(TestCase.CONSTANTS.ADMIN_PASSWORD.encode()).decode())
        textbox_2.set_textbox_controller(
            b64encode(TestCase.CONSTANTS.ADMIN_PASSWORD.encode()).decode() + '\t')
        # textbox_1.apply()
        table = viewer.DataTable("Automation Table")
        report_viewer.associate_component(table)
        data = table.get_table_data()
        SQLQueries.validate_equality(self.EXPECTED, data)

    def run(self):
        try:
            self.init_tc()
            self.create_http_dataset_with_post_query()
            self.set_username_and_password_via_input()
            self.validate()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
