# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

Live sync Resiliency testcase to Introduce network errors by changing host file entry of vcenter
in proxy machine.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case

"49354": {
      "ClientName"   : "Virtualization_client",
      "AgentName"    : "Virtual Server",
      "InstanceName" : "VMWare",
      "BackupsetName":"defaultbackupset",
      "SubclientName": "group3",
      "ScheduleName" : "rep-3"
},
"""

from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Server.Scheduler.schedulerhelper import SchedulerHelper
from VirtualServer.VSAUtils import VirtualServerUtils, OptionsHelper
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from AutomationUtils.machine import Machine

from Web.Common.page_object import handle_testcase_exception
from Web.Common.page_object import TestStep
from Web.Common.exceptions import CVTestStepFailure


class TestCase(CVTestCase):
    """Class for executing VSA Fake connectivity issue to VCenter"""
    test_step = TestStep()
    
    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "VMW Live sync Resiliency - Introduce network errors"
        self.schedule_helper = None
        self.schedule_name = None
        self.proxy_machine = None
        self.vcentre_hostname = None
        self.auto_subclient = None
        self.dest_auto_subclient = None
        self.backup_options = None
        self.live_sync_utils = None

    @test_step
    def power_off_dest_vms(self):
        """Power Off destination VMs"""
        self.dest_auto_subclient.hvobj.power_off_all_vms()

    def _initialize_proxy_machine_obj(self):
        """Creates restore proxy machine object"""
        proxy_client = self.dest_auto_subclient.get_proxy_list()[0]
        self.log.info(f"Initializing proxy Machine Object of [{proxy_client}]")
        self.proxy_machine = Machine(proxy_client, self.commcell)
        self.vcentre_hostname = self.dest_auto_subclient.hvobj.prop_dict["server_name"]

    def _add_fake_host_entry(self):
        """Add fake host entry, wait for 15 secs and remove fake host entry"""
        fake_ip = "172.16.1.1"
        self.log.info(
            f"Adding fake host file entry in {self.proxy_machine.machine_name}"
            f"for vcenter {self.vcentre_hostname}"
        )
        # Add fake host entry
        self.proxy_machine.add_host_file_entry(self.vcentre_hostname, fake_ip)

    def _remove_fake_host_entry(self):
        self.log.info(f"Removing fake host file entry in {self.proxy_machine.machine_name}")
        # Remove fake host entry
        self.proxy_machine.remove_host_file_entry(self.vcentre_hostname)

    def get_replication_status(self, vm_name):
        live_sync_pair = self.subclient.live_sync.get(self.schedule_name)
        vm_pair = live_sync_pair.get(vm_name)
        return vm_pair.status

    def setup(self):
        """Setup function for test case"""
        self.schedule_name = self.tcinputs["ScheduleName"]
        schedule = self.client.schedules.get(self.schedule_name)
        self.schedule_helper = SchedulerHelper(schedule, self.commcell)
        # Get destination vsa instance object
        self.auto_subclient = VirtualServerUtils.subclient_initialize(self)
        self.dest_auto_subclient = self.auto_subclient.get_live_sync_destination_subclient(
            self.schedule_name
        )
        self.live_sync_utils = LiveSyncUtils(self.auto_subclient, self.schedule_name)

    @test_step
    def backup(self):
        """Run Incremental backup"""
        backup_options = OptionsHelper.BackupOptions(self.auto_subclient)
        backup_options.backup_type = "INCREMENTAL"
        self.auto_subclient.backup(backup_options)

    @test_step
    def replication_to_pending(self, job_obj):
        """Replication job should go to pending state"""
        self.log.info(
            f'Sleeping for 1 hour 15 minutes for the Replication job [{job_obj.job_id}] '
            f'to go to pending state'
        )
        sleep(4500)
        job_obj.refresh()
        if job_obj.status.lower() == 'running':
            self.log.info(
                f'Sleeping for 10 minutes for the replication job [{job_obj.job_id}]'
                'to go to pending state'
            )
            sleep(600)
            job_obj.refresh()
        if job_obj.status.lower() != 'pending':
            raise CVTestStepFailure(
                f"Replication job [{job_obj.job_id}] should be in pending state "
                f"but it's in status [{job_obj.status.lower()}]"
            )
        self.log.info(f"Replication job [{job_obj.job_id}] is in pending status as expected")
        self.log.info(
            f'Sleeping for 30 seconds for the replication status of VM'
            f' [{self.auto_subclient.vm_list[0]}] to go to pending state'
        )
        sleep(30)
        self.log.info(f"Replication job [{job_obj.job_id}] is in pending status as expected")
        status = self.get_replication_status(self.auto_subclient.vm_list[0])
        if status != 'SYNC_PAUSED':
            raise CVTestStepFailure(
                f'VM [{self.auto_subclient.vm_list[0]}] replication status is not in sync paused,'
                f'current status [{status}]'
            )

    @test_step
    def replication_to_complete(self, job_obj):
        """Replication job to complete after failure"""
        job_obj.refresh()
        if job_obj.status.lower() == 'pending':
            self.log.info(
                f'Sleeping for 120 seconds for the replication status of VM'
                f' [{self.auto_subclient.vm_list[0]}] to go to running state'
            )
            sleep(120)
            job_obj.refresh()
        if job_obj.status.lower() == 'pending':
            raise CVTestStepFailure(
                f"Replication job [{job_obj.job_id}] should not be in pending state "
                f"but it's in status [{job_obj.status.lower()}]"
            )
        self.log.info("Waiting for Replication job [{job.job_id}] to complete")
        if not job_obj.wait_for_completion():
            self.log.info(f"Replication job with job id: {job_obj.job_id} failed")
            raise CVTestStepFailure("Replication job [{job_obj.job_id}] failed")
        self.log.info("Replication job [{job.job_id}] completed")


    def run(self):
        """Main function for test case execution"""
        backup_options = None
        try:
            self.power_off_dest_vms()
            self._initialize_proxy_machine_obj()
            self.backup()
            self.log.info('Waiting for 1 minute for replication job to trigger')
            sleep(60)
            self._add_fake_host_entry()
            job_obj = self.live_sync_utils.get_recent_replication_job(
                self.auto_subclient.backup_job.job_id
            )
            self.replication_to_pending(job_obj)
            self._remove_fake_host_entry()
            self.log.info('Resuming job to go to running state')
            job_obj.resume()
            sleep(20)
            self.replication_to_complete(job_obj)
            self.live_sync_utils.validate_live_sync()
        except Exception as exp:
            handle_testcase_exception(self, exp)
            self._remove_fake_host_entry()
        finally:
            try:
                if self.auto_subclient and backup_options:
                    self.auto_subclient.cleanup_testdata(backup_options)
            except Exception:
                self.log.warning("Testcase cleanup was not completed")