# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case

Sample input:
"49055": {
      "ClientName": "Virtualization_client",
      "AgentName": "Virtual Server",
      "InstanceName": "VMWare",
      "BackupsetName": "defaultbackupset",
      "SubclientName": "group5",
      "ScheduleName": "rep-5"
},

"""

from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Server.Scheduler.schedulerhelper import SchedulerHelper
from VirtualServer.VSAUtils import VirtualServerUtils, OptionsHelper
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils

from Web.Common.page_object import handle_testcase_exception
from Web.Common.page_object import TestStep
from Web.Common.exceptions import CVTestStepFailure


class TestCase(CVTestCase):
    """Class for executing VSA Backup and Replication when Destination VM is powered ON"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Virtual Server - VMW Live sync - Replication with destination VM powered ON"
        self.schedule_helper = None
        self.schedule_name = None
        self.auto_subclient = None
        self.dest_auto_subclient = None
        self.backup_options = None
        self.live_sync_utils = None
        self.__dest_vm = list()

    @test_step
    def power_on_dest_vm(self):
        """Power On Destination VMs"""
        self.dest_auto_subclient.hvobj.power_on_all_vms()

    @test_step
    def power_off_dest_vm(self):
        """Power off destination VM"""
        self.dest_auto_subclient.hvobj.power_off_all_vms()

    def get_replication_status(self, vm_name):
        live_sync_pair = self.subclient.live_sync.get(self.schedule_name)
        vm_pair = live_sync_pair.get(vm_name)
        return vm_pair.status

    def setup(self):
        """Setup function for test case"""
        self.schedule_name = self.tcinputs["ScheduleName"]
        schedule = self.client.schedules.get(self.schedule_name)
        self.schedule_helper = SchedulerHelper(schedule, self.commcell)
        self.auto_subclient = VirtualServerUtils.subclient_initialize(self)
        self.dest_auto_subclient = self.auto_subclient.get_live_sync_destination_subclient(
            self.schedule_name
        )
        self.live_sync_utils = LiveSyncUtils(self.auto_subclient, self.schedule_name)

    @test_step
    def backup(self):
        """Run Incremental backup"""
        backup_options = OptionsHelper.BackupOptions(self.auto_subclient)
        backup_options.backup_type = "INCREMENTAL"
        self.auto_subclient.backup(backup_options)

    @test_step
    def replication_to_pending(self, job_obj):
        """Replication job should go to pending state"""
        job_obj.refresh()
        if job_obj.status.lower() == 'running':
            self.log.info(
                f'Sleeping for 1 minutes for the replication job [{job_obj.job_id}]'
                'to go to pending state'
            )
            sleep(60)
            job_obj.refresh()
        if job_obj.status.lower() != 'pending':
            raise CVTestStepFailure(
                f"Replication job [{job_obj.job_id}] should be in pending state "
                f"but it's in status [{job_obj.status.lower()}]"
            )
        self.log.info(
            f'Sleeping for 30 seconds for the replication status of VM'
            f' [{self.auto_subclient.vm_list[0]}] to go to pending state'
        )
        sleep(30)
        self.log.info(f"Replication job [{job_obj.job_id}] is in pending status as expected")
        status = self.get_replication_status(self.auto_subclient.vm_list[0])
        if status != 'NEEDS_SYNC':
            raise CVTestStepFailure(
                f'VM [{self.auto_subclient.vm_list[0]}] replication status is not in sync pending,'
                f'current status [{status}]'
            )

    @test_step
    def replication_to_complete(self, job_obj):
        """Replication job to complete after failure"""
        self.log.info('Waiting for 1 minute for job resume')
        sleep(60)
        job_obj.refresh()
        if job_obj.status.lower() != 'running':
            raise CVTestStepFailure(
                f"Replication job [{job_obj.job_id}] should be in running state "
                f"but it's in status [{job_obj.status.lower()}]"
            )
        self.log.info(
            f"Replication job [{job_obj.job_id}] is in {job_obj.status.lower()} status as expected"
        )
        if job_obj.status.lower() != 'completed':
            self.log.info(f"Waiting for Replication job [{job_obj.job_id}] to complete")
            if job_obj.wait_for_completion():
                self.log.info(f"Replication job with job id: {job_obj.job_id} Completed")
            else:
                raise CVTestStepFailure(f"Replication Job {job_obj.job_id} Failed")

    def run(self):
        """Main function for test case execution"""
        try:
            self.power_on_dest_vm()
            self.backup()
            self.log.info('Sleeping for 1 minutes')
            sleep(60)
            # Get latest replication job from schedule
            job_obj = self.live_sync_utils.get_recent_replication_job(
                self.auto_subclient.backup_job.job_id
            )
            self.replication_to_pending(job_obj)
            self.power_off_dest_vm()
            self.log.info('Resuming job to go to running state')
            job_obj.resume()
            self.replication_to_complete(job_obj)
            self.live_sync_utils.validate_live_sync()
        except Exception as exp:
            handle_testcase_exception(self, exp)

        finally:
            try:
                if self.auto_subclient and self.backup_options:
                    self.auto_subclient.cleanup_testdata(self.backup_options)
            except Exception:
                self.log.warning("Testcase cleanup was not completed")
