# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Metrics: Executive Summary Report PPT generation """
import time

from cvpysdk import schedules

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from Web.WebConsole.Reports.manage_schedules import ScheduleSettings
from Web.WebConsole.Reports.cte import ConfigureSchedules

from AutomationUtils import mail_box
from AutomationUtils import config
from AutomationUtils import logger
from AutomationUtils.cvtestcase import CVTestCase

from Reports.utils import TestCaseUtils


CONSTANTS = config.get_config()


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()
    REPORT_NAME = 'Activity'
    MIN_FILE_SIZE = 340000  # BYTES
    SCHEDULE_NAME = 'Automation_tc_48872_' + str(int(time.time()))

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Executive Summary Report PPT generation"
        self.show_to_user = True
        self.log = logger.get_log()
        self.browser = None
        self.webconsole = None
        self.report = None
        self.navigator = None
        self.export = None
        self.schedule_email_id = None
        self.schedules = None
        self.schedule_settings = None
        self.mail = None
        self.utils = TestCaseUtils(self)

    def _init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.utils.reset_temp_dir()
            download_directory = self.utils.get_temp_dir()
            self.log.info("Download directory:%s", download_directory)
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.set_downloads_dir(download_directory)
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.navigator = Navigator(self.webconsole)
            self.webconsole.goto_reports()
            self.schedule_settings = ScheduleSettings(self.webconsole)
            self.cleanup_schedules()
            self.navigator.goto_worldwide_report(TestCase.REPORT_NAME)
            self.webconsole.wait_till_loadmask_spin_load()
            self.report = MetricsReport(self.webconsole)
            self.export = self.report.export_handler()
            self.schedule_email_id = CONSTANTS.email.email_id
            self.schedules = schedules.Schedules(self.commcell)
            self.mail = mail_box.MailBox()
            self.mail.connect()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def cleanup_schedules(self):
        """ Deletes the schedules which contain 'Automation_tc_48872_' in schedule name """
        self.navigator.goto_schedules_configuration()
        self.schedule_settings.cleanup_schedules("Automation_tc_48872_")

    @test_step
    def _export_executive_summary_report(self):
        """
        Generates executive summary report ppt
        """
        self.export.to_executive_summary()

    @test_step
    def _validate_schedule(self):
        """
        Validate schedules
        """
        self.log.info("Creating schedule [%s]", TestCase.SCHEDULE_NAME)
        schedule = self.report.open_schedule()
        schedule.create_schedule(TestCase.SCHEDULE_NAME, self.schedule_email_id,
                                 ConfigureSchedules.Format.EXECUTIVE_SUMMARY,
                                 )
        self.log.info("Created schedule successfully")

        self.schedules.refresh()
        if not self.schedules.has_schedule(TestCase.SCHEDULE_NAME):
            err_str = "[%s] schedule does not exists. Please verify schedule is created " \
                      "successfully"
            raise CVTestStepFailure(err_str)

        self.log.info("Running the schedule [%s]", TestCase.SCHEDULE_NAME)
        _schedule = schedules.Schedule(self.commcell,
                                       schedule_name=TestCase.SCHEDULE_NAME)
        _job_id = _schedule.run_now()
        _job = self.commcell.job_controller.get(_job_id)

        self.log.info("Wait for [%s] job to complete", str(_job))
        if _job.wait_for_completion(300):  # wait for max 5 minutes
            self.log.info("Schedule job completed with job id:[%s]" % _job_id)
        else:
            err_str = "Schedule job failed with job id [%s]" % _job_id
            raise CVTestStepFailure(err_str)

        self.log.info("verify scheduled report mail has proper attachment")
        self.utils.reset_temp_dir()
        self.utils.download_mail(self.mail, TestCase.SCHEDULE_NAME)
        self.utils.get_attachment_files(ends_with="pptx")
        self.log.info("Verified attachment for the Schedule [%s] ", TestCase.SCHEDULE_NAME)
        self.cleanup_schedules()

    @test_step
    def _validate_exported_file(self):
        """
        Validate exported file
        """
        self.utils.wait_for_file_to_download("pptx")
        self.utils.validate_tmp_files("pptx", min_size=TestCase.MIN_FILE_SIZE)
        self.log.info("Validated exported file")

    def run(self):
        try:
            self._init_tc()
            self._export_executive_summary_report()
            self._validate_exported_file()
            self._validate_schedule()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.mail.disconnect()
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
