# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Validate exports for worldwide and commcell level reports"""
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from AutomationUtils import logger
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Reports import reportsutils
from Web.Common.exceptions import CVWebAutomationException


REPORTS_CONFIG = reportsutils.get_reports_config()


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Reports export validation"
        self.show_to_user = True
        self.log = logger.get_log()
        self.tcinputs = {}
        self.browser = None
        self.webconsole = None
        self.report = None
        self.navigator = None
        self.export = None
        self.worldwide_reports = None
        self.commcell_reports = None
        self.commcell_name = None
        self.utils = TestCaseUtils(self)

    def _init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.utils.reset_temp_dir()
            download_directory = self.utils.get_temp_dir()
            self.log.info("Download directory:%s", download_directory)
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.set_downloads_dir(download_directory)
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.navigator = Navigator(self.webconsole)
            self.report = MetricsReport(self.webconsole)
            self.export = self.report.export_handler()
            self.worldwide_reports = REPORTS_CONFIG.REPORTS.METRICS.WORLDWIDE
            self.worldwide_reports.extend(REPORTS_CONFIG.REPORTS.CUSTOM)
            self.commcell_name = reportsutils.get_commcell_name(self.commcell)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def verify_export_to_pdf(self):
        """
        Verify export to pdf is working fine
        """
        self.export.to_pdf()
        self.webconsole.wait_till_loadmask_spin_load()
        self.utils.wait_for_file_to_download('pdf')
        self.utils.validate_tmp_files("pdf")
        self.log.info("pdf export completed successfully")

    def verify_export_to_csv(self):
        """
        Verify export to csv is working fine
        """
        self.export.to_csv()
        self.webconsole.wait_till_loadmask_spin_load()
        self.utils.wait_for_file_to_download('csv')
        self.utils.validate_tmp_files("csv")
        self.log.info("csv export completed successfully")

    def verify_export_to_html(self):
        """
        Verify export to html is working fine
        """
        self.export.to_html()
        self.webconsole.wait_till_loadmask_spin_load()
        self.utils.wait_for_file_to_download('html')
        self.utils.validate_tmp_files("html")
        self.log.info("html export completed successfully")

    def _export_report(self):
        """
        Verify export to pdf, csv, html html is working fine
        """
        self.verify_export_to_pdf()
        self.verify_export_to_csv()
        self.verify_export_to_html()

    @test_step
    def verify_worldwide_reports_export(self):
        """
        Verify worldwide exports are working fine
        """
        for each_report in self.worldwide_reports:
            self.log.info("validating export for worldwide report %s", each_report)
            self.navigator.goto_worldwide_report(each_report)
            self.utils.reset_temp_dir()
            self._export_report()
        self.log.info("Verified export for worldwide reports")

    @test_step
    def verify_commcell_reports_exports(self):
        """
        verify commcell reports exports are working fine
        """
        for each_report in self.commcell_reports:
            self.log.info("validating export for commcell report %s", each_report)
            try:
                self.navigator.goto_commcell_reports(each_report, self.commcell_name)
            except CVWebAutomationException as ex:
                if each_report == 'Job Summary' or each_report == 'SLA' or each_report == '作业摘要':
                    continue
                else:
                    raise CVWebAutomationException(ex)
            self.utils.reset_temp_dir()
            self._export_report()
        self.log.info("Verified export for commcell level reports")

    def run(self):
        try:
            self._init_tc()
            locales = ['en', 'zh']
            self.verify_worldwide_reports_export()
            for local in locales:
                self.webconsole.set_language(local)
                if local == 'en':
                    self.commcell_reports = REPORTS_CONFIG.REPORTS.METRICS.COMMCELL_REPORTS
                elif local == 'zh':
                    self.log.info("Verify export for commcell level reports using chines local ")
                    self.commcell_reports = REPORTS_CONFIG.REPORTS.METRICS.CHINES_LOCAL_COMMCELL_REPORTS
                self.verify_commcell_reports_exports()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.webconsole.set_language('en')
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
