# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Testcase to verify Scrubbing in CommServer
"""


from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils

from Web.Common.page_object import TestStep
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.health_tiles import GenericTile
from Web.WebConsole.Reports.Metrics.components import MetricsTable

from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure

from cvpysdk.metricsreport import PrivateMetrics


class TestCase(CVTestCase):
    """Testcase to verify Metrics Scrubbing in CommServer"""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Metrics: Scrubbing in CommServer"
        self.utils = TestCaseUtils(self)
        self.browser = None
        self.webconsole = None
        self.navigator = None
        self.private_metrics = None
        self.private_metrics_name = None
        self.prune_db_tile = None

    def setup(self):
        """Intializes Private metrics object required for this testcase"""
        self.private_metrics = PrivateMetrics(self.commcell)
        self.private_metrics_name = self.private_metrics.private_metrics_server_name
        self.private_metrics.enable_health()
        self.private_metrics.save_config()

    def init_webconsole(self):
        """Initialize the application to the state required by the testcase"""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.private_metrics_name)
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.navigator = Navigator(self.webconsole)
            self.navigator.goto_commcell_dashboard(self.commcell.commserv_name)
            self.health_tile = GenericTile(self.webconsole, 'Index Cache Space Utilization')
            self.health_tile_mountPath = GenericTile(self.webconsole, 'Mount Path')
        except Exception as msg:
            raise CVTestCaseInitFailure(msg) from msg

    @test_step
    def enable_scrubbing(self):
        """Enable scrubbing in CommCell"""
        cmd = "-sn SetKeyIntoGlobalParamTbl.sql -si CommservSurveyPrivateScrubPathInfo -si y -si 1"
        exec_command = self.commcell._services['EXECUTE_QSCRIPT'] % cmd
        self.commcell._cvpysdk_object.make_request("POST", exec_command)

    @test_step
    def disable_scrubbing(self):
        """Disable scrubbing in CommCell"""
        cmd = "-sn SetKeyIntoGlobalParamTbl.sql -si CommservSurveyPrivateScrubPathInfo -si n"
        exec_command = self.commcell._services['EXECUTE_QSCRIPT'] % cmd
        self.commcell._cvpysdk_object.make_request("POST", exec_command)

    @test_step
    def check_exclude_scurbbing(self):
        """check if CommservSurveyPrivateExcludeScrubList is present in gxglobalParam table"""
        query = "SELECT value FROM GxGlobalParam WHERE name " \
                "='CommservSurveyPrivateExcludeScrubList' and modified=0"
        response = self.utils.cre_api.execute_sql(query)
        if not response:
            raise CVTestStepFailure(
                "CommservSurveyPrivateExcludeScrubList is not set in the GxGlobalParam table. "
                " Set this value [<ExcludeScrubList><Name>MountPathName</Name></ExcludeScrubList>]"
                " in the GxGlobalParam table "
                " and rerun the testcase. "
            )

    @test_step
    def verify_scrubbing(self):
        """Verifies Scubbing is successfull and reflected in reports"""
        self.navigator.goto_health_report()
        self.health_tile.access_view_details()
        table = MetricsTable(self.webconsole)
        values = table.get_data_from_column('Index Cache Path')
        values = list(set(values))
        if len(values) != 1 or values[0] != 'Masked Data':
            raise CVTestStepFailure(
                "Index cache location report path info is not masked. "
                f"expected data in path [Masked Data] received data {values}"
            )

    @test_step
    def verify_exclude_scrubbing(self):
        """Verifies Scubbing is successfull and reflected in reports"""
        self.navigator.goto_health_report()
        self.health_tile_mountPath.access_view_details()
        table = MetricsTable(self.webconsole)
        values = table.get_data_from_column('Mount Path')
        values = list(set(values))
        if len(values) != 1 and values[0] == 'Masked Data':
            raise CVTestStepFailure(
                "Mouth Path report mount path name is still masked even after setting exclusion "
                "in scrubbing."
            )

    def run(self):
        try:
            self.enable_scrubbing()
            self.check_exclude_scurbbing()
            self.utils.private_metrics_upload()
            self.init_webconsole()
            self.verify_scrubbing()
            self.verify_exclude_scrubbing()
        except Exception as error:
            self.utils.handle_testcase_exception(error)
        finally:
            self.disable_scrubbing()
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
