# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Metrics Report Alert"""
import datetime
import time

from urllib.parse import urlparse

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from Web.WebConsole.Reports.manage_alerts import AlertSettings
from Web.WebConsole.Reports.Metrics.components import AlertMail
from AutomationUtils import mail_box

from AutomationUtils.cvtestcase import CVTestCase

from Reports.utils import TestCaseUtils


class AlertDetails:
    """
    Set alert details
    """
    alert_name = None
    column_name = None
    condition = None
    hours = None
    minutes = None
    ampm = None
    report_url = None
    report_name = None

    def __init__(self, column_name, condition, report_url, report_name):
        """
        Set alert details
        Args:
            column_name               (String)  --  name of the column
            condition:                (String)  --  the condition string which is set on table
            report_url:               (String)  --  URL of the report
            report_name:              (String)  --  Name of the report
        """
        self.alert_name = "Automation_tc_47857_%s" % str(int(time.time()))
        self.column_name = column_name
        self.condition = condition
        self.report_url = report_url
        self.report_name = report_name
        self.generate_alert_time()

    def generate_alert_time(self):
        """
        Generate alert time to set in alert
        """
        now = datetime.datetime.now()
        now_plus_2mins = now + datetime.timedelta(minutes=2)
        self.hours = str(int(datetime.datetime.strftime(now_plus_2mins, "%I")))
        self.minutes = str(int(datetime.datetime.strftime(now_plus_2mins, "%M")))
        if len(self.minutes) == 1:
            self.minutes = "0" + self.minutes
        self.ampm = str(datetime.datetime.strftime(now_plus_2mins, "%p"))


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Metrics Report Alert"
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.navigator = None
        self.report = None
        self.reports = ["License Usage"]
        self.alert_window = None
        self.alerts = []
        self.metrics_table = None
        self.alert_settings = None
        self.mail = None
        self.alert_mail = None
        self.mail_browser = None
        self._driver = None
        self.utils = TestCaseUtils(self)

    def init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.mail = mail_box.MailBox()
            self.mail.connect()

            # open browser
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.open()

            # login to web console and redirect to ww reports.
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self._driver = self.browser.driver
            self.webconsole.goto_reports()

            self.navigator = Navigator(self.webconsole)
            self.report = MetricsReport(self.webconsole)
            self.alert_settings = AlertSettings(self.webconsole)
            self.cleanup_alerts()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def create_alert(self):
        """Create alert"""
        for each_report in self.reports:
            self.navigator.goto_worldwide_report(each_report)
            table = self.report.get_tables()[0]
            column_name = table.get_visible_column_names()[0]
            condition_string = table.get_data_from_column(column_name)[0]
            self.log.info("Creating alert for [%s] report for [%s] column with condition string:"
                          "[%s]", each_report, column_name, condition_string)
            alert = AlertDetails(column_name=column_name, condition=condition_string,
                                 report_url=self._driver.current_url, report_name=each_report)
            alert_window = table.open_alert()
            alert_window.set_time(alert.hours, alert.minutes, alert.ampm)
            alert_window.create_alert(alert_name=alert.alert_name, column_name=column_name,
                                      column_value=condition_string)
            self.log.info("Alert [%s] created successfully on [%s] report ", alert.alert_name,
                          each_report)
            self.alerts.append(alert)

    def cleanup_alerts(self):
        """ Deletes the alert which contain 'Automation_tc_47857_' in alert name """
        self.navigator.goto_alerts_configuration()
        self.alert_settings.cleanup_alerts("Automation_tc_47857_")

    def access_report(self, url):
        """Access report"""
        self._driver.get(url)
        self.webconsole.wait_till_load_complete()

    def access_email_file(self):
        """Access email downloaded file in browser"""
        html_path = self.utils.poll_for_tmp_files(ends_with='html')[0]
        self.mail_browser = BrowserFactory().create_browser_object(name="ClientBrowser")
        self.mail_browser.open()
        self.mail_browser.goto_file(file_path=html_path)
        self.alert_mail = AlertMail(self.mail_browser)

    def get_report_content(self, filter_value):
        """Read report data"""
        table = self.report.get_tables()[0]
        column_name = table.get_visible_column_names()[0]
        table.set_filter(column_name=column_name, value=filter_value)
        return table.get_data()

    @test_step
    def validate_alert_email(self):
        """Validate alert email"""
        for each_alert in self.alerts:
            self.utils.reset_temp_dir()
            self.utils.download_mail(self.mail, subject=each_alert.alert_name)
            self.access_report(each_alert.report_url)
            self.access_email_file()
            web_report_table_data = self.get_report_content(each_alert.condition)
            mail_report_name = self.alert_mail.get_report_name()
            if mail_report_name != each_alert.report_name:
                raise CVTestStepFailure("Report names are not matching with mail content, "
                                        "report name in mail:%s,report name in webconsole:%s" %
                                        (mail_report_name, each_alert.report_name))
            mail_report_link = self.alert_mail.get_report_link()
            if urlparse(mail_report_link).path != urlparse(each_alert.report_url).path:
                raise CVTestStepFailure("Report links are not matching with mail content, report "
                                        "link in mail:%s, report link in webconsole:%s" %
                                        (mail_report_link, each_alert.report_url))
            mail_table_data = self.alert_mail.get_table_data()
            if mail_table_data != web_report_table_data:
                raise CVTestStepFailure("Table data in report is not matching with mail content, "
                                        "table data in mail:%s,report data in webconsole:%s" %
                                        (str(mail_table_data), str(web_report_table_data)))
            self.log.info("Email content is matched successfully with report content for the "
                          "report [%s]", each_alert.report_name)

    @test_step
    def delete_alerts(self):
        """
        Delete alerts
        """
        self.navigator.goto_alerts_configuration()
        alert_names = []
        for each_alert in self.alerts:
            alert_names.append(each_alert.alert_name)
        self.alert_settings.delete_alerts(alert_names)

    def run(self):
        try:
            self.init_tc()
            self.create_alert()
            self.log.info("Wait for mails to be received for 3 minutes")
            time.sleep(180)
            self.validate_alert_email()
            self.delete_alerts()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.mail.disconnect()
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
            Browser.close_silently(self.mail_browser)
