# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

TcInputs to be passed in JSON File:

    PrimaryCopyMediaAgent:   Name of a MediaAgent machine - we create primary copy here
    SecondaryCopyMediaAgent: Name of a MediaAgent machine - we create secondary copies here

    Note: Both the MediaAgents can be the same machine

Steps:

1: Configure the environment: create a library,Storage Policy-with a Primary, 2 Secondary Copies,
                              (Source for Secondary Copy2 should be Secondary Copy1)
                              a BackupSet,a SubClient

2: Submit 2 Full Backups and run AuxCopy to Secondary Copy1

3: Seal the store, pick the jobs for recopy and run AuxCopy to Secondary Copy1 again

4: Prune and run DataAging for Backup Job1 on Copy1

5: Validate that same Enc.Key is used for recopy

6: Run AuxCopy to Secondary Copy2 and then Restore jobs from both the copies(Complete Successfully)

7: CleanUp the Environment
"""
from AutomationUtils import constants
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import DedupeHelper, MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = 'Scenario where there is arch.File with two Enc.keys in archfilesidbkeys table'
        self.tcinputs = {
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None,
        }
        self.mm_helper = None
        self.ma_machine_1 = None
        self.ma_machine_2 = None
        self.dedupe_helper = None
        self.client_machine = None
        self.ddb_path = None
        self.ma_1_path = None
        self.ma_2_path = None
        self.mount_path = None
        self.client_path = None
        self.mount_path_2 = None
        self.content_path = None
        self.restore_path = None
        self.copy_ddb_path = None
        self.subclient = None
        self.copy_name = None
        self.library_name = None
        self.storage_policy = None
        self.subclient_name = None
        self.backupset_name = None
        self.storage_policy_name = None

    def setup(self):
        """Setup function of this test case"""
        self.client_machine = Machine(self.client.client_name, self.commcell)
        self.ma_machine_1 = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.ma_machine_2 = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        utility = OptionsSelector(self.commcell)
        client_drive = utility.get_drive(self.client_machine, 1024)
        primary_ma_drive = utility.get_drive(self.ma_machine_1, 4096)
        secondary_ma_drive = utility.get_drive(self.ma_machine_2, 4096)
        self.client_path = client_drive + 'test_' + str(self.id) + self.client_machine.os_sep
        self.ma_1_path = primary_ma_drive + 'test_' + str(self.id) + self.ma_machine_1.os_sep
        self.ma_2_path = secondary_ma_drive + 'test_' + str(self.id) + self.ma_machine_2.os_sep
        self.ddb_path = self.ma_1_path + 'DDB'
        self.mount_path = self.ma_1_path + 'MP'
        self.mount_path_2 = self.ma_2_path + 'MP2'
        self.copy_ddb_path = self.ma_2_path + 'copy_DDB'
        self.content_path = self.client_path + 'Content'
        self.restore_path = self.client_path + 'Restores'
        self.copy_name = str(self.id) + '_Copy'
        self.library_name = str(self.id) + '_Lib'
        self.backupset_name = str(self.id) + '_BS'
        self.subclient_name = str(self.id) + '_SC'
        self.storage_policy_name = str(self.id) + '_SP'
        self.mm_helper = MMHelper(self)
        self.dedupe_helper = DedupeHelper(self)

    def run(self):
        """Run Function of this case"""
        try:
            # 1: Configure the environment
            self.log.info('Setting Client Properties for Encryption on Client (BlowFish, 256)')
            self.client.set_encryption_property('ON_CLIENT', 'BlowFish', '256')
            self.mm_helper.create_uncompressable_data(self.client.client_name,
                                                      self.content_path, 0.5)
            self.mm_helper.configure_disk_library(self.library_name,
                                                  self.tcinputs['PrimaryCopyMediaAgent'],
                                                  self.mount_path)
            self.mm_helper.configure_disk_library(self.library_name + '_2',
                                                  self.tcinputs['SecondaryCopyMediaAgent'],
                                                  self.mount_path_2)
            self.storage_policy = self.dedupe_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['PrimaryCopyMediaAgent'],
                self.ddb_path)
            storage_policy_copies = [self.storage_policy.get_copy('Primary')]
            for index in range(1, 3):
                storage_policy_copies.append(self.dedupe_helper.configure_dedupe_secondary_copy(
                    self.storage_policy,
                    self.copy_name + str(index),
                    self.library_name + '_2',
                    self.tcinputs['SecondaryCopyMediaAgent'],
                    self.copy_ddb_path + str(index),
                    self.tcinputs['SecondaryCopyMediaAgent']))

            # set copy1 as source for copy2
            source = {'copyId': int(storage_policy_copies[1].copy_id),
                      'copyName': self.copy_name + str(1)}
            storage_policy_copies[2]._copy_properties['sourceCopy'] = source
            storage_policy_copies[2]._set_copy_properties()

            self.mm_helper.configure_backupset(self.backupset_name)
            self.subclient = self.mm_helper.configure_subclient(content_path=self.content_path)

            # Remove Association with System Created AutoCopy Schedule
            for index in range(1, 3):
                self.mm_helper.remove_autocopy_schedule(self.storage_policy_name,
                                                        self.copy_name + str(index))

            # 2: Submit 2 Full Backups and run AuxCopy to Secondary Copy1
            self.log.info('Submitting 2 FULL Backups and AuxCopy to copy1')
            backup_job_1 = self.subclient.backup(backup_level='Full')
            if not backup_job_1.wait_for_completion():
                raise Exception('Backup Job 1 Failed')
            self.log.info('Backup Job 1 Completed Id : %s', backup_job_1.job_id)
            backup_job_2 = self.subclient.backup(backup_level='Full')
            if not backup_job_2.wait_for_completion():
                raise Exception('Backup Job 2 Failed')
            self.log.info('Backup Job 2 Completed Id : %s', backup_job_2.job_id)

            aux_copy_job = self.storage_policy.run_aux_copy(self.copy_name + str(1),
                                                            use_scale=True, all_copies=False)
            if aux_copy_job.wait_for_completion():
                self.log.info('AuxCopy Job Completed (Id: %s)', aux_copy_job.job_id)
            else:
                raise Exception('AuxCopy Job Failed (Id: %s)' % aux_copy_job.job_id)

            # 3: Seal the store, pick the jobs for recopy and run AuxCopy to Secondary Copy1 again
            self.log.info('Sealing the store, pick for Re-Copy and run AuxCopy again')
            self.storage_policy.seal_ddb(self.copy_name + str(1))
            storage_policy_copies[1].recopy_jobs(backup_job_1.job_id)
            storage_policy_copies[1].recopy_jobs(backup_job_2.job_id)

            aux_copy_job = self.storage_policy.run_aux_copy(self.copy_name + str(1),
                                                            use_scale=True, all_copies=False)
            if aux_copy_job.wait_for_completion():
                self.log.info('2nd AuxCopy Job(Re-Copy): %s Completed', aux_copy_job.job_id)
            else:
                raise Exception('2nd Aux Copy Job(Re-Copy): %s Failed' % aux_copy_job.job_id)

            # 4: Prune and run DataAging for Backup Job1 on Copy1
            self.log.info('Prune and run DataAging for Backup Job1 on Copy1')
            command = 'qoperation agedata -delbyjobid -j {0} -sp {1} -spc {2} -ft {3}'
            command = command.format(backup_job_1.job_id, self.storage_policy_name,
                                     self.copy_name + str(1), 'Q_DATA')
            self.log.info('QCommand: %s', command)
            response = self.commcell.execute_qcommand(command)
            self.log.info('Response: %s', str(response))
            self.log.info('Prune completed')

            aging_job = self.commcell.run_data_aging(self.copy_name + str(1),
                                                     self.storage_policy_name,
                                                     is_granular=True,
                                                     include_all_clients=True,
                                                     select_copies=True,
                                                     prune_selected_copies=True)
            if aging_job.wait_for_completion():
                self.log.info('Data Aging Job Completed: %s', aging_job.job_id)
            else:
                raise Exception('Data Aging Job Failed:(Id: %s)' % aging_job.job_id)

            # 5: Validate that same Enc.Key is used for recopy
            self.log.info('Validate: All archFiles in archfileSIDBKeys table refer same Enc.KeyID')
            query = '''select distinct encKeyID from archFileSIDBKeys
                    where archCopyId = %s''' % storage_policy_copies[1].copy_id
            self.log.info('Query: %s', query)
            self.csdb.execute(query)
            result = self.csdb.fetch_all_rows()
            if len(result) == 1:
                self.log.info('Validation Succeeded')
            else:
                self.log.error('Validation Failed')

            # 6: Run AuxCopy to Secondary Copy2 and then Restore jobs from both the copies
            self.log.info('Running AuxCopy to Copy2')
            aux_copy_job = self.storage_policy.run_aux_copy(self.copy_name + str(2),
                                                            use_scale=True, all_copies=False)
            if aux_copy_job.wait_for_completion():
                self.log.info('AuxCopy Job to Copy2: %s Completed', aux_copy_job.job_id)
            else:
                raise Exception('Aux Copy Job to Copy2: %s Failed' % aux_copy_job.job_id)

            restores_jobs = []
            for index in range(2, 4):
                job = self.subclient.restore_out_of_place(self.client.client_name,
                                                          self.restore_path + str(index - 1),
                                                          [self.content_path],
                                                          copy_precedence=index)
                restores_jobs.append(job)

            for job in restores_jobs:
                if job.wait_for_completion():
                    self.log.info('Restore Job: %s Completed', job.job_id)
                else:
                    raise Exception('Restore job %s Failed' % job.job_id)

            self.log.info('Validating Restored Data from 2 Copies')
            for index in range(1, 3):
                restored_path = self.restore_path + str(index) + self.client_machine.os_sep
                difference = self.client_machine.compare_folders(self.client_machine,
                                                                 self.content_path,
                                                                 restored_path + 'Content')
                if difference:
                    raise Exception('Validating Data restored from Copy %s Failed' % index)
            self.log.info('Validation SUCCESS')
        except Exception as exe:
            self.status = constants.FAILED
            self.result_string = str(exe)
            self.log.error('Exception Occurred: %s', str(exe))

    def tear_down(self):
        """Tear Down Function of this Case"""
        # 7: CleanUp the Environment
        try:
            self.log.info('reverting Client Enc.properties to default use sp_settings')
            self.client.set_encryption_property()
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting backupset %s", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)

            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("Deleting storage policy  %s", self.storage_policy_name)
                self.commcell.storage_policies.delete(self.storage_policy_name)

            self.commcell.disk_libraries.delete(self.library_name)
            self.commcell.disk_libraries.delete(self.library_name + '_2')
            self.mm_helper.remove_content(self.ma_1_path, self.ma_machine_1)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['SecondaryCopyMediaAgent']:
                self.mm_helper.remove_content(self.ma_2_path, self.ma_machine_2)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.client.client_name:
                if self.tcinputs['SecondaryCopyMediaAgent'] != self.client.client_name:
                    self.mm_helper.remove_content(self.client_path, self.client_machine)
        except Exception as exe:
            self.log.error('ERROR in TearDown. Might need to Cleanup Manually: %s', str(exe))
