# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase to perform AuxCopy with Encryption

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                  --  initialize TestCase class

    _cleanup()                  --  cleanup the entities created

    _validate_encryption_on_copy()  --  validates whether encryption flag set on copy extendedFlags in
                                        archgroupcopy table

    _get_archFiles_for_auxjob() --  gets rows for given auxcopy job id that are populated in archchunktoreplicate table

     _validate_aging()          --  validate where mmdeletedaf table entries are cleaned

     _set_prune_process_interval    --  sets config param 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS' with interval value

    setup()                     --  setup function of this test case

    run()                       --  run function of this test case

    tear_down()                 --  teardown function of this test case
"""

import time

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.idautils import CommonUtils
from MediaAgents.MAUtils.mahelper import (MMHelper, DedupeHelper)


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = "AuxCopy with Encryption"
        self.tcinputs = {
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None,
            "SqlSaPassword": None
        }
        self.options_selector = None
        self.primary_library_name = None
        self.secondary_library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.primary_mountpath = None
        self.secondary_mountpath = None
        self.primary_partition_path = None
        self.secondary_partition_path = None
        self.content_path = None
        self.dedupehelper = None
        self.library_name = None
        self.mmhelper = None
        self.common_util = None
        self.primary_ma_machine = None
        self.secondary_ma_machine = None
        self.client_machine = None

    def _validate_encryption_on_copy(self, copy_id):
        """
        validates whether encryption flag set on copy extendedFlags in archgroupcopy table
        Args:
            copy_id (int) -- copy id to validate on
        Return:
            (Bool) True/False
        """

        query = f"select count(1) from archgroupcopy where extendedflags&1024 = 1024 and id = {copy_id}"
        self.log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self.log.info("RESULT: {0}".format(cur))
        if cur != ['']:
            if int(cur[0]) == 1:
                return True
        return False

    def _get_archFiles_for_auxjob(self, job_id):
        """
            Gets rows for given auxcopy job id that are populated in archchunktoreplicate table
            Args:
                job_id (str) -- AuxCopy job id
            Return:
                list of reader id
        """

        query = f"SELECT COUNT(1) FROM ArchChunkToReplicate WHERE AdminJobId = {job_id}"
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != ['']:
            return int(cur[0])
        self._log.error("Unable to fetch rows for the given job id")
        raise Exception("Unable to fetch rows for the given job id")

    def _validate_aging(self):
        """Validate where mmdeletedaf table entries are cleaned."""

        interval = 0
        flag = 1
        while flag != 0 and interval < 300:
            time.sleep(30)
            interval = interval + 30
            query = f"""SELECT	COUNT(1), DAF.SIDBStoreId
                        FROM	MMDeletedAF DAF
                        JOIN	archGroupCopy AGC
                                ON DAF.SIDBStoreId = AGC.SIDBStoreId
                        JOIN	archGroup AG
                                ON AG.id = AGC.archGroupId
                        WHERE	(AG.name = '{self.storage_policy_name}'
                                OR AG.name = '{self.gdsp_name}')
                                AND AGC.SIDBStoreId <> 0
                        GROUP BY DAF.SIDBStoreId"""

            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            archfile_count = self.csdb.fetch_all_rows()
            if archfile_count[0][0] == '':
                flag = 0
            else:
                flag = 1
                for count in archfile_count:
                    self.log.info("POLL ---- Number of entries for DDB-%s in mmdeletedaf =[%s] " %
                                  (count[1], count[0]))
        if flag == 0:
            return True
        else:
            return False

    def _set_prune_process_interval(self, value, n_min=10):
        """
        sets config param 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS' with interval value
        Args:
            value (int): time in minutes to set as config param value

            n_min (int): minimum value that can be set for this config
        """
        self.log.info("Setting pruning process interval to %s", value)
        _query = """UPDATE MMConfigs
                    SET value = {0} , nMin = {1}
                    WHERE NAME = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'
                    """.format(value, n_min)
        self.mmhelper.execute_update_query(_query, self.tcinputs['SqlSaPassword'])

    def _cleanup(self):
        """Cleanup the entities created"""

        self.log.info("********************** CLEANUP STARTING *************************")
        try:
            # Delete bkupset
            self.log.info("Deleting BackupSet: %s if exists", self.backupset_name)
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
                self.log.info("Deleted BackupSet: %s", self.backupset_name)

            # Delete Storage Policy
            self.log.info("Deleting storage policy: %s if exists", self.storage_policy_name)
            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.commcell.storage_policies.delete(self.storage_policy_name)
                self.log.info("Deleted storage policy: %s", self.storage_policy_name)

            # Delete GDSP
            self.log.info("Deleting GDSP: %s if exists", self.gdsp_name)
            if self.commcell.storage_policies.has_policy(self.gdsp_name):
                self.commcell.storage_policies.delete(self.gdsp_name)
                self.log.info("Deleted GDSP: %s", self.gdsp_name)

            # Delete Library
            self.log.info("Deleting primary library: %s if exists", self.primary_library_name)
            if self.commcell.disk_libraries.has_library(self.primary_library_name):
                self.commcell.disk_libraries.delete(self.primary_library_name)
                self.log.info("Deleted primary library: %s", self.primary_library_name)

            # Delete Library
            self.log.info("Deleting secondary library: %s if exists", self.secondary_library_name)
            if self.commcell.disk_libraries.has_library(self.secondary_library_name):
                self.commcell.disk_libraries.delete(self.secondary_library_name)
                self.log.info("Deleted secondary library: %s", self.secondary_library_name)

            # Run DataAging
            data_aging_job = self.commcell.run_data_aging()
            self.log.info("Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

        except Exception as exp:
            self.log.error("Error encountered during cleanup : %s", str(exp))
            raise Exception("Error encountered during cleanup: {0}".format(str(exp)))

        self.log.info("********************** CLEANUP COMPLETED *************************")

    def setup(self):
        """Setup function of this test case"""

        self.options_selector = OptionsSelector(self.commcell)
        self.primary_library_name = '%s_primary_library' % str(self.id)
        self.secondary_library_name = '%s_secondary_library' % str(self.id)
        self.gdsp_name = '%s_gdsp' % str(self.id)
        self.storage_policy_name = '%s_storage_policy' % str(self.id)
        self.backupset_name = '%s_BS' % str(self.id)
        self.subclient_name = '%s_SC' % str(self.id)

        self._cleanup()

        self.primary_ma_machine = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.secondary_ma_machine = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        self.client_machine = Machine(self.client)

        # To select drive with space available in primary ma machine
        self.log.info('Selecting drive in the primary MA machine based on space available')
        primary_ma_drive = self.options_selector.get_drive(self.primary_ma_machine, size=(5 * 1024))
        if primary_ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', primary_ma_drive)

        # To select drive with space available in secondary ma machine
        self.log.info('Selecting drive in the secondary MA machine based on space available')
        secondary_ma_drive = self.options_selector.get_drive(self.secondary_ma_machine, size=(5 * 1024))
        if secondary_ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', secondary_ma_drive)

        # To select drive with space available in client machine
        self.log.info('Selecting drive in the client machine based on space available')
        client_drive = self.options_selector.get_drive(self.client_machine, size=(5 * 1024))
        if client_drive is None:
            raise Exception("No free space for generating content")
        self.log.info('selected drive: %s', client_drive)

        self.primary_mountpath = self.primary_ma_machine.join_path(primary_ma_drive, 'Automation', str(self.id), 'MP')

        self.primary_partition_path = self.primary_ma_machine.join_path(primary_ma_drive, 'Automation', str(self.id),
                                                                        'DDB')

        self.secondary_mountpath = self.secondary_ma_machine.join_path(secondary_ma_drive, 'Automation', str(self.id),
                                                                       'MP')
        self.secondary_partition_path = self.secondary_ma_machine.join_path(secondary_ma_drive, 'Automation',
                                                                            str(self.id), 'DDB')

        self.content_path = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'TestData')
        if self.client_machine.check_directory_exists(self.content_path):
            self.client_machine.remove_directory(self.content_path)
        self.client_machine.create_directory(self.content_path)

        self.mmhelper = MMHelper(self)
        self.dedupehelper = DedupeHelper(self)
        self.common_util = CommonUtils(self)

    def run(self):
        """Run function of this test case"""

        try:
            self.mmhelper.configure_disk_library(self.primary_library_name, self.tcinputs['PrimaryCopyMediaAgent'],
                                                 self.primary_mountpath)
            self.mmhelper.configure_disk_library(self.secondary_library_name, self.tcinputs['SecondaryCopyMediaAgent'],
                                                 self.secondary_mountpath)

            sp_dedup_obj = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_name,
                                                                             self.primary_library_name,
                                                                             self.tcinputs['PrimaryCopyMediaAgent'],
                                                                             self.primary_partition_path)

            # Enable default enc on client
            self.client.set_encryption_property('ON_CLIENT', 'BLOWFISH', '256')

            # create first secondary copy
            copy1 = '%s_copy1' % str(self.id)
            self.dedupehelper.configure_dedupe_secondary_copy(sp_dedup_obj, copy1, self.secondary_library_name,
                                                              self.tcinputs['SecondaryCopyMediaAgent'],
                                                              self.secondary_partition_path +
                                                              self.options_selector.get_custom_str(),
                                                              self.tcinputs['SecondaryCopyMediaAgent'])
            # create second secondary copy
            copy2 = '%s_copy2' % str(self.id)
            copy2_obj = self.dedupehelper.configure_dedupe_secondary_copy(sp_dedup_obj, copy2,
                                                                          self.secondary_library_name,
                                                                          self.tcinputs['SecondaryCopyMediaAgent'],
                                                                          self.secondary_partition_path +
                                                                          self.options_selector.get_custom_str(),
                                                                          self.tcinputs['SecondaryCopyMediaAgent'])

            # create gdsp
            self.dedupehelper.configure_global_dedupe_storage_policy(self.gdsp_name, self.secondary_library_name,
                                                                     self.tcinputs['SecondaryCopyMediaAgent'],
                                                                     self.secondary_partition_path +
                                                                     self.options_selector.get_custom_str(),
                                                                     self.tcinputs['SecondaryCopyMediaAgent'])

            # creating SP copy pointing to GDSP
            copy3 = '%s_copy3' % str(self.id)
            self._log.info("Adding secondary copy pointing to GSDP: %s", copy3)
            sp_dedup_obj.create_secondary_copy(copy3, global_policy=self.gdsp_name)
            copy3_obj = sp_dedup_obj.get_copy(copy3)

            # Enable parallel copy
            copy3_obj.parallel_copy = True

            # Removing association with System Created Automatic Auxcopy schedule
            self.log.info("Removing association with System Created Autocopy schedule on above created copy")
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_name, copy1)
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_name, copy2)
            self.mmhelper.remove_autocopy_schedule(self.storage_policy_name, copy3)

            self.backupset = self.mmhelper.configure_backupset(self.backupset_name, self.agent)
            sc_obj = self.mmhelper.configure_subclient(self.backupset_name, self.subclient_name,
                                                       self.storage_policy_name, self.content_path, self.agent)

            # Create unique content
            self.log.info("Generating Data at %s", self.content_path)
            if not self.client_machine.generate_test_data(self.content_path, dirs=1, file_size=(100 * 1024), files=10):
                self.log.error("unable to Generate Data at %s", self.content_path)
                raise Exception("unable to Generate Data at {0}".format(self.content_path))
            self.log.info("Generated Data at %s", self.content_path)

            # Backup Job J1
            job1_obj = self.common_util.subclient_backup(sc_obj, "full")

            # Backup Job J2
            job2_obj = self.common_util.subclient_backup(sc_obj, "full")

            # Run Aux copy Job
            auxcopy_job = sp_dedup_obj.run_aux_copy()
            self.log.info("Auxcopy job [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error("Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception("Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id,
                                                                                auxcopy_job.delay_reason))
            self.log.info("Auxcopy job [%s] has completed.", auxcopy_job.job_id)

            # Validating encryption flag set on extendedFlags in archgroupcopy table
            if self._validate_encryption_on_copy(copy3_obj.copy_id):
                self.log.info("Encryption flag set on copy")
            else:
                self.log.error("Encryption flag not set on copy")

            # delete copy created to GDSP
            self.log.info("Deleting storage policy copy pointed to GDSP")
            sp_dedup_obj.delete_secondary_copy(copy3)

            # Recreating SP copy pointing to same GDSP
            self._log.info("Adding secondary copy pointing to same GSDP: %s", copy3)
            sp_dedup_obj.create_secondary_copy(copy3, global_policy=self.gdsp_name)
            copy3_obj = sp_dedup_obj.get_copy(copy3)

            # Enable parallel copy
            copy3_obj.set_parallel_copy(True)

            # ReRun Aux copy Job
            auxcopy_job = sp_dedup_obj.run_aux_copy()
            self.log.info("Auxcopy job [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error("Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception("Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id,
                                                                                auxcopy_job.delay_reason))
            self.log.info("Auxcopy job [%s] has completed.", auxcopy_job.job_id)

            # Validating Aux copy job
            self.log.info("Querying ArchChunkToReplicate table to see if entries were created for this auxcopy job")
            if self._get_archFiles_for_auxjob(auxcopy_job.job_id) == 0:
                self.log.info("Successfully skipped archive files for dash aux when present in"
                              "mmdeletedaf table for same storeID")
            else:
                self.log.error("Failed to skipped archive files for dash aux when present in"
                               "mmdeletedaf table for same storeID")
                raise Exception("ERROR   Result:Fail for check to skip archive files for dash aux when present in "
                                "mmdeletedaf table for same storeID")

            # setting prune process interval to 2 mins
            self._set_prune_process_interval(2, 1)

            # Run Granular DataAging
            data_aging_job = self.commcell.run_data_aging(storage_policy_name = self.storage_policy_name,
                                                          is_granular=True, include_all_clients=True)
            self.log.info("First Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

            # Run Granular DataAging - Second
            data_aging_job = self.commcell.run_data_aging(storage_policy_name = self.storage_policy_name,
                                                          is_granular=True, include_all_clients=True)
            self.log.info("Second Data Aging job [%s] has started.", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.error(
                    "Data Aging job [%s] has failed with %s.", data_aging_job.job_id, data_aging_job.delay_reason)
                raise Exception(
                    "Data Aging job [{0}] has failed with {1}.".format(data_aging_job.job_id,
                                                                       data_aging_job.delay_reason))
            self.log.info("Data Aging job [%s] has completed.", data_aging_job.job_id)

            # Validate Aging
            if self._validate_aging():
                self.log.info("Entries in mmdeleteaf cleanup succeeded")
            else:
                self.log.error("Entries in mmdeleteaf cleanup Failed")
                raise Exception("Entries in mmdeleteaf cleanup Failed")

            # Reverting pruning process interval to 60 mins
            self._set_prune_process_interval(60)

            # Picking jobs J1 and J2 for recopy on copy2
            copy2_obj.recopy_jobs(job1_obj.job_id + ", " + job2_obj.job_id)

            # Run Aux copy Job
            auxcopy_job = sp_dedup_obj.run_aux_copy()
            self.log.info("Auxcopy job [%s] has started.", auxcopy_job.job_id)
            if not auxcopy_job.wait_for_completion():
                self.log.error("Auxcopy job [%s] has failed with %s.", auxcopy_job.job_id, auxcopy_job.delay_reason)
                raise Exception("Auxcopy job [{0}] has failed with {1}.".format(auxcopy_job.job_id,
                                                                                auxcopy_job.delay_reason))
            self.log.info("Auxcopy job [%s] has completed.", auxcopy_job.job_id)

            # Validating encryption flag set on extendedFlags in archgroupcopy table
            if self._validate_encryption_on_copy(copy3_obj.copy_id):
                self.log.info("Encryption flag set on copy")
            else:
                self.log.error("Encryption flag not set on copy")

            # Validating Restores from three copies

            #   Restore from copy1
            restore_job = sc_obj.restore_in_place([self.content_path], copy_precedence=2)
            self.log.info("restore job [%s] has started from [%s].", restore_job.job_id, copy1)
            if not restore_job.wait_for_completion():
                self.log.error("restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                restore_job.delay_reason))
            self.log.info("restore job [%s] has completed from [%s].", restore_job.job_id, copy1)

            #   Restore from copy2
            restore_job = sc_obj.restore_in_place([self.content_path], copy_precedence=3)
            self.log.info("restore job [%s] has started from [%s].", restore_job.job_id, copy2)
            if not restore_job.wait_for_completion():
                self.log.error("restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                restore_job.delay_reason))
            self.log.info("restore job [%s] has completed from [%s].", restore_job.job_id, copy2)

            #   Restore from copy3
            restore_job = sc_obj.restore_in_place([self.content_path], copy_precedence=4)
            self.log.info("restore job [%s] has started from [%s].", restore_job.job_id, copy3)
            if not restore_job.wait_for_completion():
                self.log.error("restore job [%s] has failed with %s.", restore_job.job_id, restore_job.delay_reason)
                raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                restore_job.delay_reason))
            self.log.info("restore job [%s] has completed from [%s].", restore_job.job_id, copy3)

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear Down function of this test case"""

        # revert back dedupe default setting on client
        self.client.set_dedup_property('clientSideDeduplication', 'USE_SPSETTINGS')

        # Reverting pruning process interval to 60
        self._set_prune_process_interval(60)

        if self.status != constants.FAILED:
            self.log.info("Testcase shows successful execution, cleaning up the test environment ...")
            self._cleanup()
        else:
            self.log.error("Testcase shows failure in execution, not cleaning up the test environment ...")

        if self.client_machine.check_directory_exists(self.content_path):
            self.client_machine.remove_directory(self.content_path)
