# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Reports : Validate Reports Charts"""
from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import (
    viewer,
    builder
)

from AutomationUtils.cvtestcase import CVTestCase


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    QUERY = """
            DECLARE @i INT = 0
            DECLARE @tmpTable table(emp_id VARCHAR(MAX),department_name VARCHAR(MAX),
                                    department_code INT,joining_date DATE)
            WHILE @i < 10
            BEGIN
                  SET NOCOUNT ON
                  if @i % 3 = 0
                     INSERT INTO @tmpTable (emp_id,department_name,
                                              department_code,joining_date)
                     VALUES('EMP0'+CAST(@i AS varchar),'SALES',111,DATEADD(day,30,GETDATE()))
                  else
                     INSERT INTO @tmpTable(emp_id,department_name,
                                             department_code,joining_date)
                     VALUES('EMP0'+CAST(@i AS varchar),'RESEARCH',222,DATEADD(day,45,GETDATE()))
                  SET @i = @i + 1
            END

            SELECT * from @tmpTable

    """
    COLUMNS_FOR_CHART = ['department_code', 'joining_date']
    HORIZONTAL_BAR = {
        'x-axis_title': 'department_code',
        'y_axis_title': 'department_code',
        'count': 2,
        'height': ['6', '4'],
        'x-axis_labels': ['222', '111']
    }

    VERTICAL_BAR = {
        'x-axis_title': 'department_code',
        'y_axis_title': 'department_code Sum',
        'count': 2,
        'height': ['1332', '444'],
        'x-axis_labels': ['222', '111']
    }

    PIE_CHART = {
        'slice_count': 2,
        'slice_values': ['222', '111']
    }

    DONUT_CHART = {
        'slice_count': 2,
        'slice_values': ['222', '111']
    }

    LINE_CHART = {
        'x-axis_title': 'department_code',
        'y_axis_title': 'department_code Max',
        'count': 2,
        'height': ['222', '111'],
        'x-axis_labels': ['222', '111'],
        'y-axis_labels': ['0', '50', '100', '150', '200', '250']
    }

    TIMELINE_CHART = {
        'x-axis_title': 'joining_date',
        'y_axis_title': 'department_code Count',
        'count': 2,
        'height': ['4', '6'],
        'y-axis_labels': ['0', '2', '4', '6', '8']
    }

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports : Validate Reports Charts"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.builder = None
        self.dataset = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def add_dataset(self):
        """Adds dataset"""
        self.builder = builder.ReportBuilder(self.webconsole)
        self.builder.set_report_name(self.name)
        self.dataset = builder.Datasets.DatabaseDataset()
        self.builder.add_dataset(self.dataset)
        self.dataset.set_dataset_name("Automation Dataset")
        self.dataset.set_sql_query(TestCase.QUERY)
        self.dataset.save()

    @test_step
    def create_horizontal_bar(self):
        """Creates Horizontal Bar chart"""
        horizontal_bar = builder.HorizontalBar("Automation Chart 1")
        self.builder.add_component(horizontal_bar, self.dataset)
        horizontal_bar.set_x_axis(TestCase.COLUMNS_FOR_CHART[0])
        horizontal_bar.set_aggregation("Count")
        details = horizontal_bar.get_chart_details()
        del details["y-axis_labels"]
        SQLQueries.validate_membership_equality(TestCase.HORIZONTAL_BAR, details,
                                                err_msg="Expected and received values are not equal in Horizontal Bar")

    @test_step
    def create_vertical_bar(self):
        """Creates Vertical Bar chart"""
        vertical_bar = builder.VerticalBar("Automation Chart 2")
        self.builder.add_component(vertical_bar, self.dataset)
        vertical_bar.set_x_axis(TestCase.COLUMNS_FOR_CHART[0])
        vertical_bar.set_aggregation("Sum")
        details = vertical_bar.get_chart_details()
        del details["y-axis_labels"]
        SQLQueries.validate_equality(TestCase.VERTICAL_BAR, details,
                                     err_msg="Expected and received values are not equal in Vertical Bar")

    @test_step
    def create_pie_chart(self):
        """Creates Pie chart"""
        pie = builder.PieChart("Automation Chart 3")
        self.builder.add_component(pie, self.dataset)
        pie.add_column_to_dimension(TestCase.COLUMNS_FOR_CHART[0])
        pie.set_aggregation("Min")
        details = pie.get_chart_details()
        SQLQueries.validate_equality(TestCase.PIE_CHART, details,
                                     err_msg="Expected and received values are not equal in Pie Chart")

    @test_step
    def create_donut_chart(self):
        """Creates Donut chart"""
        donut = builder.DonutChart("Automation Chart 4")
        self.builder.add_component(donut, self.dataset)
        donut.add_column_to_dimension(TestCase.COLUMNS_FOR_CHART[0])
        donut.set_aggregation("Avg")
        details = donut.get_chart_details()
        SQLQueries.validate_equality(TestCase.DONUT_CHART, details,
                                     err_msg="Expected and received values are not equal in Donut Chart")

    @test_step
    def create_line_chart(self):
        """Creates Line Chart"""
        line = builder.LineChart("Automation Chart 5")
        self.builder.add_component(line, self.dataset)
        line.set_x_axis(TestCase.COLUMNS_FOR_CHART[0])
        line.set_aggregation("Max")
        details = line.get_chart_details()
        SQLQueries.validate_equality(TestCase.LINE_CHART, details,
                                     err_msg="Expected and received values are not equal in Line Chart")

    @test_step
    def create_timeline_chart(self):
        """Creates Timeline chart """
        timeline = builder.TimelineChart("Automation Chart 6")
        self.builder.add_component(timeline, self.dataset)
        timeline.set_x_axis(TestCase.COLUMNS_FOR_CHART[1])
        timeline.set_time_grouping("Weekly")
        timeline.set_y_axis(TestCase.COLUMNS_FOR_CHART[0])
        timeline.set_aggregation("Count")
        details = timeline.get_chart_details()
        del details['x-axis_labels']
        SQLQueries.validate_equality(TestCase.TIMELINE_CHART, details,
                                     err_msg="Expected and received values are not equal in Timeline Chart")

    @test_step
    def validate_viewer(self):
        """Validates chart in viewer"""
        self.builder.save(deploy=True)
        self.builder.open_report()
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        horizontal_bar = viewer.HorizontalBar("Automation Chart 1")
        vertical_bar = viewer.VerticalBar("Automation Chart 2")
        pie_chart = viewer.PieChart("Automation Chart 3")
        donut = viewer.DonutChart("Automation Chart 4")
        line_chart = viewer.LineChart("Automation Chart 5")
        timeline_chart = viewer.TimelineChart("Automation Chart 6")

        report_viewer.associate_component(horizontal_bar)
        report_viewer.associate_component(vertical_bar)
        report_viewer.associate_component(pie_chart)
        report_viewer.associate_component(donut)
        report_viewer.associate_component(line_chart)
        report_viewer.associate_component(timeline_chart)

        details = horizontal_bar.get_chart_details()
        del details['y-axis_labels']
        SQLQueries.validate_membership_equality(TestCase.HORIZONTAL_BAR, details,
                                                err_msg="Expected and received values are not equal in Horizontal Bar")

        details = vertical_bar.get_chart_details()
        del details["y-axis_labels"]
        SQLQueries.validate_equality(TestCase.VERTICAL_BAR, details,
                                     err_msg="Expected and received values are not equal in Vertical Bar")

        details = pie_chart.get_chart_details()
        SQLQueries.validate_equality(TestCase.PIE_CHART, details,
                                     err_msg="Expected and received values are not equal in Pie Chart")

        details = donut.get_chart_details()
        SQLQueries.validate_equality(TestCase.DONUT_CHART, details,
                                     err_msg="Expected and received values are not equal in Donut Chart")

        details = line_chart.get_chart_details()
        SQLQueries.validate_equality(TestCase.LINE_CHART, details,
                                     err_msg="Expected and received values are not equal in Line Chart")

        details = timeline_chart.get_chart_details()
        del details['x-axis_labels']
        del details['y-axis_labels']
        del TestCase.TIMELINE_CHART['y-axis_labels']
        SQLQueries.validate_equality(TestCase.TIMELINE_CHART, details,
                                     err_msg="Expected and received values are not equal in Timeline Chart")

    def run(self):
        try:
            self.init_tc()
            self.add_dataset()
            self.create_horizontal_bar()
            self.create_vertical_bar()
            self.create_pie_chart()
            self.create_donut_chart()
            self.create_line_chart()
            self.create_timeline_chart()
            self.validate_viewer()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
