# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Metrics: Private metrics Health report pdf export validation"""
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.report import MetricsReport

from AutomationUtils import logger
from AutomationUtils.cvtestcase import CVTestCase

from Reports.utils import TestCaseUtils
from Reports import reportsutils


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Metrics: Private metrics Health report pdf export validation"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.log = logger.get_log()
        self.tcinputs = {}
        self.browser = None
        self.webconsole = None
        self.report = None
        self.navigator = None
        self.export = None
        self.commcell_name = None
        self.utils = TestCaseUtils(self)

    def init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.utils.reset_temp_dir()
            download_directory = self.utils.get_temp_dir()
            self.log.info("Download directory:%s", download_directory)
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.set_downloads_dir(download_directory)
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.commcell_name = reportsutils.get_commcell_name(self.commcell)
            self.webconsole.goto_commcell_dashboard()
            self.webconsole.wait_till_load_complete()
            self.navigator = Navigator(self.webconsole)
            self.navigator.goto_commcell_dashboard(self.commcell_name)
            self.navigator.goto_health_report()
            self.report = MetricsReport(self.webconsole)
            self.export = self.report.export_handler()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def verify_health_export_to_pdf(self):
        """
        Verify export to pdf in health page
        """
        self.export.to_pdf()
        self.webconsole.wait_till_loadmask_spin_load()
        self.utils.poll_for_tmp_files(ends_with='pdf', timeout=600, min_size=500)
        self.log.info("pdf export completed successfully")

    @test_step
    def verify_expected_export_options(self):
        """
        Verify only pdf export option is visible in private metrics
        """
        expected_option = ["PDF", "CSV"]
        self.webconsole.wait_till_load_complete()
        available_export_options = self.export.get_available_export_types()
        if expected_option != available_export_options:
            raise CVTestStepFailure("Expected export [%s] is not present in health export. "
                                    "Available exports are:[%s]" %
                                    (str(expected_option), str(available_export_options)))
        self.log.info("verified available export options in health page successfully")

    def run(self):
        try:
            self.init_tc()
            self.verify_expected_export_options()
            self.verify_health_export_to_pdf()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
