# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    previous_run_cleanup() -- for deleting the left over
    backupset and storage policy from the previous run

    run_full_backup_job() -- for running a full backup job

    get_unix_size() -- get size of folder on unix systems

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

This testcase verifies if the physical pruning
is taking place or not, even if one chunk
belonging to the backup job is pruned
using drill hole method this testcase will pass.

input json file arguments required:

                        "ClientName": "name of the client machine without as in commserve",
                        "AgentName": "File System",
                        "MediaAgentName": "name of the media agent as in commserve",
                        "ExpectedDeletedRecords"  - str  - the minimum number
                         of references you want in zero ref table
                        "library_name": name of the Library to be reused
                        "mount_path": path where the data is to be stored
                        "dedup_path": path where dedup store to be created

                        note --
                                ***********************************
                                if library_name_given then reuse_library
                                else:
                                    if mountpath_location_given -> create_library_with_this_mountpath
                                    else:
                                        auto_generate_mountpath_location
                                        create_library_with_this_mountpath

                                if dedup_path_given -> use_given_dedup_path
                                else it will auto_generate_dedup_path
                                ***********************************

Design Steps:
1.	Create resources
2.	Content: numerous files with random data
    amounting to 1 gb ~ roughly 180 files (total)
    2.1 Disable mark and sweep
3.	Subclient level: data reader streams : 1
4.	First backup job , get the sidb id,
    archfiles., chunks -> first backup job
5.	Delete every alternate file
6.	Second backup job
7.	Delete first backup job
8.	MMdeleted AF check
9.	Change mm prune process interval to 2 min
10.	MMdeletedArchFileTracking table check
11.	Checks ->  chunks deleted drill hole
    sidbphysicaldeletes.log, phase 3 count 0 idxsidbusagehistory,
    sidb2.exe process arch file count, restore job, md5 hash compare
"""

import time
from AutomationUtils import constants, machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils import mahelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "drill hole testcase: verifies " \
                    "if the drill hole pruning process is occurring or not"

        self.tcinputs = {
            "MediaAgentName": None,
            "ExpectedDeletedRecords": None
        }

        self.mount_path = None
        self.dedup_store_path = None
        self.restore_path = None
        self.content_path = None
        self.library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.mm_helper = None
        self.dedup_helper = None
        self.client_machine = None
        self.media_agent_machine = None
        self.opt_selector = None
        self.storage_policy_id = None
        self.sidb_id = None
        self.substore_id = None
        self.testcase_path = None
        self.testcase_path_client = None
        self.testcase_path_media_agent = None
        self.library = None
        self.storage_policy = None
        self.backup_set = None
        self.subclient = None
        self.is_user_defined_lib = False
        self.is_user_defined_mp = False
        self.is_user_defined_dedup = False

    def setup(self):
        """sets up the variables to be used in testcase"""
        if self.tcinputs.get("library_name"):
            self.is_user_defined_lib = True
        if self.tcinputs.get("mount_path"):
            self.is_user_defined_mp = True
        if self.tcinputs.get("dedup_path"):
            self.is_user_defined_dedup = True

        suffix = str(self.tcinputs["MediaAgentName"])[1:] + str(self.tcinputs["ClientName"])[1:]
        if self.is_user_defined_lib:
            self.log.info("Existing library name supplied")
            self.library_name = self.tcinputs.get("library_name")
        else:
            self.library_name = "{0}_lib{1}".format(str(self.id), suffix)
        self.storage_policy_name = "{0}_SP{1}".format(str(self.id), suffix)
        self.backupset_name = "{0}_BS{1}".format(str(self.id), suffix)
        self.subclient_name = "{0}_SC{1}".format(str(self.id), suffix)
        self.dedup_helper = mahelper.DedupeHelper(self)
        self.mm_helper = mahelper.MMHelper(self)
        self.opt_selector = OptionsSelector(self.commcell)
        self.client_machine = machine.Machine(
            self.client.client_name, self.commcell)
        self.media_agent_machine = machine.Machine(
            self.tcinputs["MediaAgentName"], self.commcell)

    def previous_run_clean_up(self):
        """delete previous run items"""
        self.log.info("********* previous run clean up **********")
        try:
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
            if self.commcell.storage_policies.has_policy(
                    self.storage_policy_name):
                self.commcell.storage_policies.delete(self.storage_policy_name)
            self.log.info("previous run clean up COMPLETED")
        except Exception as exp:
            self.log.info("previous run clean up ERROR")
            self.log.info("ERROR:%s", exp)

    def run_full_backup_job(self):
        """
            run a full backup job

            Returns
                an object of running full backup job
        """
        self.log.info("Starting backup job")
        job = self.subclient.backup("FULL")
        self.log.info("Backup job: %s", str(job.job_id))
        if not job.wait_for_completion():
            if job.status.lower() == "completed":
                self.log.info("job %s complete", job.job_id)
            else:
                raise Exception(
                    "Job {0} Failed with {1}".format(
                        job.job_id, job.delay_reason))
        return job

    @staticmethod
    def get_unix_size(machine_object, folder_path):
        """get the size of folder in float format in bytes"""
        command = 'du "{0}"'.format(folder_path)
        output = machine_object.execute(command)
        return float(output.formatted_output[-1][0])

    def run(self):
        """Run function of this test case"""
        try:
            self.previous_run_clean_up()

            # create the required resources for the testcase
            # get the drive path with required free space
            drive_path_client = self.opt_selector.get_drive(
                self.client_machine)
            if not self.is_user_defined_lib:
                drive_path_media_agent = self.opt_selector.get_drive(
                    self.media_agent_machine)
                self.testcase_path_media_agent = "%s%s" % (drive_path_media_agent, self.id)

            # creating testcase directory, mount path, content path, dedup
            # store path

            self.testcase_path_client = "%s%s" % (drive_path_client, self.id)

            self.content_path = self.client_machine.join_path(
                self.testcase_path_client, "content_path")
            if self.client_machine.check_directory_exists(self.content_path):
                self.log.info("content path directory already exists")
                self.client_machine.remove_directory(self.content_path)
                self.log.info("existing content deleted- so it doesn't interfere with dedupe")
            self.client_machine.create_directory(self.content_path)
            self.log.info("content path created")

            self.restore_path = self.client_machine.join_path(
                self.testcase_path_client, "restore_path")
            if self.client_machine.check_directory_exists(self.restore_path):
                self.log.info("restore path directory already exists")
                self.client_machine.remove_directory(self.restore_path)
                self.log.info("existing restore path deleted")
            self.client_machine.create_directory(self.restore_path)
            self.log.info("restore path created")

            if self.is_user_defined_mp:
                if self.is_user_defined_lib:
                    self.log.info("custom mount path along with library supplied")
                    self.mount_path = self.media_agent_machine.join_path(self.tcinputs["mount_path"],
                                                                         self.media_agent_machine.os_sep)
                else:
                    self.log.info("custom mount path supplied")
                    self.mount_path = self.media_agent_machine.join_path(self.tcinputs["mount_path"], self.id)
            else:
                if not self.is_user_defined_lib:
                    self.mount_path = self.media_agent_machine.join_path(
                        self.testcase_path_media_agent, "mount_path")

            if self.is_user_defined_dedup:
                self.log.info("custom dedup path supplied")
                self.dedup_store_path = self.media_agent_machine.join_path(self.tcinputs["dedup_path"], self.id)
            else:
                self.dedup_store_path = self.media_agent_machine.join_path(
                    self.testcase_path_media_agent, "dedup_store_path")

            # create library
            if not self.is_user_defined_lib:
                self.library = self.mm_helper.configure_disk_library(
                    self.library_name, self.tcinputs["MediaAgentName"], self.mount_path)

            # create SP
            self.storage_policy = self.dedup_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs["MediaAgentName"],
                self.dedup_store_path)

            # use the storage policy object
            # from it get the storage policy id
            # get the sidb store id and sidb sub store id
            # self.storage_policy_id = self.storage_policy._get_storage_policy_id()
            return_list = self.dedup_helper.get_sidb_ids(
                self.storage_policy.storage_policy_id, "Primary")
            self.sidb_id = int(return_list[0])
            self.substore_id = int(return_list[1])

            # disable mark and sweep
            # we are setting the third bit from right as 0 (flag and (1011))
            query = """UPDATE   IdxSIDBStore
                       SET      ExtendedFlags = ExtendedFlags & ~4
                       WHERE    SIDBStoreId = {0}""".format(self.sidb_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.opt_selector.update_commserve_db(query)

            # we are setting the first bit from right as 0 (flag and (1110))
            query = """UPDATE    IdxSIDBSubStore
                       SET       ExtendedFlags = ExtendedFlags & ~1
                       WHERE     SubStoreId = {0}""".format(self.substore_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.opt_selector.update_commserve_db(query)
            self.log.info("mark and sweep DISABLED")

            # create backupset
            self.backup_set = self.mm_helper.configure_backupset(
                self.backupset_name, self.agent)

            # generate unique random data for the testcase
            # factor of 0.00028 helps to ensure minimum number of records
            # which are deleted after pruning
            data_size = int(self.tcinputs["ExpectedDeletedRecords"]) * 0.00028
            if self.mm_helper.create_uncompressable_data(self.client.client_name,
                                                         self.content_path,
                                                         float("{0:.1f}".format(data_size)), 1):
                self.log.info("generated unique data for subclient")
            else:
                raise Exception("couldn't generate unique data")

            content_files = sorted(
                self.client_machine.get_files_in_path(
                    self.content_path))
            # got the files to be loaded to the subclient

            # create subclient
            self.log.info("check SC: %s", self.subclient_name)
            if not self.backup_set.subclients.has_subclient(
                    self.subclient_name):
                self.subclient = self.backup_set.subclients.add(
                    self.subclient_name, self.storage_policy_name)
                self.log.info("created subclient %s", self.subclient_name)
            else:
                self.log.info("subclient %s exists", self.subclient_name)
                self.subclient = self.backup_set.subclients.get(
                    self.subclient_name)

            # add subclient content
            self.log.info(
                "add all the generated files as content to the subclient")
            self.subclient.content = content_files

            # set the subclient data reader / streams to one
            self.log.info(
                "set the data readers for subclient %s to 1",
                self.subclient_name)
            self.subclient.data_readers = 1

            # use the default storage policy block size 128 kb

            # run the first backup job
            first_job = self.run_full_backup_job()

            query = """SELECT    archchunkid
                       FROM      archchunkmapping
                       WHERE     archfileid
                       IN       ( SELECT    id
                                  FROM      archfile 
                                  WHERE     jobid={0} 
                                  AND       filetype=1)""".format(first_job.job_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.csdb.execute(query)
            res = self.csdb.fetch_all_rows()
            chunks_first_job = []
            for i in range(len(res)):
                chunks_first_job.append(int(res[i][0]))
            self.log.info("got the chunks belonging to the first backup job")
            self.log.info("Chunks are: {0}".format(chunks_first_job))

            query = """SELECT    id
                       FROM      archFile
                       WHERE     jobId={0}""".format(first_job.job_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.csdb.execute(query)
            res = self.csdb.fetch_all_rows()
            arch_files_first_job = []
            for i in range(len(res)):
                arch_files_first_job.append(int(res[i][0]))
            self.log.info(
                "got the archfiles belonging to the first backup job")
            self.log.info("Archfiles are:{0}".format(arch_files_first_job))

            temp_files = []
            # change the content on the subclient for the second backup job
            for i in range(0, len(content_files), 2):
                temp_files.append(content_files[i])
            content_files = temp_files
            self.log.info(
                "deleted every alternate file from the content files list ")
            self.log.info(
                "maximises the chance of drill hole taking place while pruning ")

            # add the modified content files list to subclient
            self.log.info(
                "MODIFIED content files list added as content to the subclient")
            self.subclient.content = content_files

            # run the second backup job
            second_job = self.run_full_backup_job()

            # get the exact mount path location
            query = """ SELECT      mountpathName
                        FROM        MMLibrary,MMMountPath
                        WHERE       MMMountPath.LibraryId = MMLibrary.LibraryId
                        AND         MMLibrary.AliasName = '{0}'""".format(self.library_name)
            self.log.info("EXECUTING QUERY: %s", query)
            self.csdb.execute(query)
            mount_path_location = self.media_agent_machine.join_path(
                self.mount_path, str(
                    self.csdb.fetch_one_row()[0]))
            self.log.info("Getting folder size for %s", mount_path_location)
            if self.media_agent_machine.client_object.os_info.lower().count('windows') > 0:
                initial_size = self.media_agent_machine.get_folder_size(mount_path_location, True)
            else:
                initial_size = self.get_unix_size(self.media_agent_machine, mount_path_location)

            # delete the first backup job
            storage_policy_copy = self.storage_policy.get_copy("Primary")
            # because only copy under storage policy was created above
            storage_policy_copy.delete_job(first_job.job_id)
            self.log.info("deleted the first backup job: id %s", str(first_job.job_id))
            error_flag = []

            # after deletion of job, the archFiles should be moved to
            # MMdeletedAF table
            self.log.info("sleeping for 30 seconds")
            time.sleep(30)

            self.log.info("===================================="
                          "===============================================")
            self.log.info("SETUP VALIDATION 1:"
                          " Verify if the deleted job files "
                          "have been moved to MMdeletedAF")

            query = """SELECT    archFileId
                       FROM      MMDeletedAF
                       WHERE     SIDBStoreId={0}""".format(self.sidb_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.csdb.execute(query)
            res = self.csdb.fetch_all_rows()
            mm_deleted_af_list = []
            for i in range(len(res)):
                mm_deleted_af_list.append(int(res[i][0]))

            if set(arch_files_first_job) == set(mm_deleted_af_list):
                self.log.info(
                    "Result: pass _ archfiles of first job have been transferred to MMdeletedAF")
            else:
                self.log.info("Result: fail")
                error_flag = error_flag \
                             + ["archfiles of deleted job "
                                "have not been moved to MMDeletedAF"]
            self.log.info("-------------------------------------"
                          "----------------------------------------------")


            query = """UPDATE    mmconfigs
                       SET       value = 2, nmin = 0
                       WHERE     name = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"""
            self.log.info("EXECUTING QUERY %s", query)
            self.opt_selector.update_commserve_db(query)
            self.log.info("mmprune process interval set to two minute")

            self.log.info("waiting... to trigger MM Prune Process")
            for i in range(2):
                self.log.info(
                    "data aging + sleep for 240 seconds: RUN %s" %
                    (i + 1))
                job = self.commcell.run_data_aging(
                    copy_name="Primary",
                    storage_policy_name=self.storage_policy_name,
                    is_granular=True,
                    include_all_clients=True,
                    select_copies=True,
                    prune_selected_copies=True)
                self.log.info("Data Aging job: %s", str(job.job_id))
                if not job.wait_for_completion():
                    if job.status.lower() == "completed":
                        self.log.info("job %s complete", job.job_id)
                    else:
                        raise Exception(
                            "Job {0} Failed with {1}".format(
                                job.job_id, job.delay_reason))
                time.sleep(240)

            # will make sure that prune process can only start up once in this
            # interval
            self.log.info("MMPruneProcess wait over")

            # no need to move time
            # since the mark and sweep disabled

            log_file = "SIDBPhysicalDeletes.log"
            match_regex = [" H "]
            drill_hole_occurrance = False
            drill_hole_records = []
            self.log.info("==========================================="
                          "========================================")
            self.log.info("CASE VALIDATION 1: verify if the drill "
                          "hole occurred for the chunks of the first job")

            matched_lines, matched_strings = self.dedup_helper.parse_log(
                self.tcinputs['MediaAgentName'], log_file, match_regex[0], single_file=True)

            for matched_line in matched_lines:
                line = matched_line.split()
                if int(line[7]) in chunks_first_job:
                    drill_hole_occurrance = True
                    drill_hole_records.append(line[7])

            if drill_hole_occurrance:
                self.log.info("Result: Pass _ Atleast one chunk"
                              " was deleted by using Drill Hole method")
                self.log.info("Chunk IDs with drilled holes are ")
                for drilled_hole_chunk in set(drill_hole_records):
                    self.log.info(drilled_hole_chunk)
            else:
                self.log.info("Result: Fail")
                error_flag += ["No Chunk was deleted using"
                               " Drill Hole method: Drill hole did not occur"]

            time.sleep(180)
            self.log.info("===================================="
                          "===============================================")
            self.log.info("CASE VALIDATION 2: Check for"
                          " phase 3 count in idxSidbUsageHistory Table")
            query = """SELECT    ZeroRefCount
                       FROM      IdxSIDBUsageHistory
                       WHERE     SIDBStoreId = {0}
                       AND       HistoryType = 1
                       ORDER BY(ModifiedTime) DESC""".format(self.sidb_id)
            self.log.info("EXECUTING QUERY %s", query)
            self.csdb.execute(query)
            zero_ref_count_case3 = int(self.csdb.fetch_one_row()[0])
            if zero_ref_count_case3 == 0:
                self.log.info("Result:Pass")
                self.log.info("Pending delete count is 0")
            else:
                self.log.info("Result:Fail")
                self.log.info("Deletion of items with"
                              " no reference is still pending")
                error_flag += ["pending delete count in"
                               " idxSidbUsageHistory Table is not zero"]

            self.log.info("======================================"
                          "=============================================")
            self.log.info("CASE VALIDATION 3: Checking if"
                          " there is reduction in physical size of the chunks")
            # get the mount path exact folder,
            # compare the size with the physical
            # size before drill hole pruning
            # took place
            if self.media_agent_machine.client_object.os_info.lower().count('windows') > 0:
                current_size = self.media_agent_machine.get_folder_size(mount_path_location, True)
            else:
                current_size = self.get_unix_size(self.media_agent_machine, mount_path_location)
            if current_size < initial_size:
                self.log.info("physical size after pruning [%d] by "
                              "drill hole is less than initial size [%d]"
                              % (current_size, initial_size))
                self.log.info("RESULT: PASS")
            else:
                self.log.info("RESULT: FAIL")
                self.log.info("physical size after pruning [%d] by drill hole is not less than initial size [%d]" %
                              (current_size, initial_size))
                error_flag += ["physical size after pruning by drill hole is not less than initial size"]

            self.log.info("===================================="
                          "===============================================")
            self.log.info("CASE VALIDATION 4: Running a restore job"
                          " for the second backup and verifying the files")
            restore_job = self.subclient.restore_out_of_place(self.client.client_name,
                                                              self.restore_path, content_files)
            self.log.info("Restore job: " + str(restore_job.job_id))
            if not restore_job.wait_for_completion():
                if restore_job.status.lower() == "completed":
                    self.log.info("job %d complete", restore_job.job_id)
                else:
                    raise Exception(
                        "Job {0} Failed with {1}".format(
                            restore_job.job_id,
                            restore_job.delay_reason))

            self.log.info("VERIFYING IF THE RESTORED FILES ARE SAME OR NOT")
            restored_files = self.client_machine.get_files_in_path(
                self.restore_path)
            self.log.info("Comparing the files using MD5 hash")
            if len(restored_files) == len(content_files):
                restored_files.sort()
                for original_file, restored_file in zip(
                        content_files, restored_files):
                    if not self.client_machine.compare_files(
                            self.client_machine, original_file, restored_file):
                        self.log.info("Result: Fail")
                        raise ValueError("The restored file is "
                                         "not the same as the original content file")
            self.log.info("All the restored files "
                          "are same as the original content files")
            self.log.info("Result: Pass")

            if error_flag:
                # if the list is not empty then error was there, fail the test
                # case
                self.log.info(error_flag)
                raise Exception("testcase failed")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """delete all items created for the testcase"""
        try:
            self.log.info("*********************************************")
            self.log.info("Restoring defaults")

            self.log.info("setting back the "
                          "mmprune process interval to 60 mins")
            query = """update mmconfigs
                    set value = 60, nmin = 10
                    where name = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"""
            self.opt_selector.update_commserve_db(query)
            self.log.info("EXECUTING QUERY %s", query)

            # delete the generated content for this testcase
            # machine object initialised earlier
            if self.client_machine.check_directory_exists(self.content_path):
                self.client_machine.remove_directory(self.content_path)
                self.log.info("Deleted the generated data.")
            else:
                self.log.info("Content directory does not exist.")

            # delete the restored directory
            if self.client_machine.check_directory_exists(self.restore_path):
                self.client_machine.remove_directory(self.restore_path)
                self.log.info("Deleted the restored data.")
            else:
                self.log.info("Restore directory does not exist.")

            self.log.info("deleting backupset and SP of the test case")
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
                self.log.info("backup set deleted")
            else:
                self.log.info("backup set does not exist")

            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.commcell.storage_policies.delete(self.storage_policy_name)
                self.log.info("storage policy deleted")
            else:
                self.log.info("storage policy does not exist.")

            if not self.is_user_defined_lib:
                if self.commcell.disk_libraries.has_library(self.library_name):
                    self.commcell.disk_libraries.delete(self.library_name)
                    self.log.info("Library deleted")
                else:
                    self.log.info("Library does not exist.")
            self.log.info("clean up successful")

        except Exception as exp:
            self.log.info("clean up not successful")
            self.log.info("ERROR:%s", exp)
