# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

Testcase to validate basic features of Indexing like browse, find, versions, synthetic full
and restores

TestCase:
    __init__()                  --  Initializes the TestCase class

    setup()                     --  All testcase objects are initializes in this method

    run()                       --  Contains the core testcase logic and it is the one executed

    create_backupset()          --  Creates a backupset and initializes testdata directory

    create_subclient()          --  Creates subclients and initializes testdata directory

    run_sequence()              --  Executes a sequence of actions provided

    run_backup()                --  Runs a backup job and adds the job to Indexing validation

    run_combinations()          --  Executes the browse/find/restore combinations one by one

    edit_testdata()             --  Edits the testdata of the subclients

    generate_combinations()     --  Prepares a list of browse/find/versions/restore combinations

    get_timerange()             --  Gets a random timerange from the backups

    get_versions_file()         --  Gets a random file to do view all versions

    create_file()               --  Creates a file with a random name

    edit_file()                 --  Edits the content of the given file

    create_folder()             --  Creates a folder with random name

    tear_down()                 --  Cleans the data created for Indexing validation

"""

import traceback
import time
import itertools
import json
import random

from queue import Queue
from threading import Thread
from AutomationUtils import logger, constants, commonutils
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import CVEntities
from AutomationUtils.idautils import CommonUtils
from AutomationUtils.machine import Machine
from Indexing.browse_restore import validation


class TestCase(CVTestCase):

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = 'Indexing - Browse, find, versions and restore'
        self.product = self.products_list.MEDIAAGENT
        self.feature = self.features_list.INDEXING
        self.show_to_user = False

        self.tcinputs = {
            'StoragePolicyName': None,
            'TestDataPath': None,
            'RestorePath': None
        }

        self.backupset = None
        self.subclients = {}

        self.cv_entities = None
        self.cv_ops = None
        self.idx = None
        self.cl_machine = None
        self.cl_delim = None

        self.job_end_times = {}
        self.manual_testdata = {}
        self.combination_failed = False

    def setup(self):
        """All testcase objects are initializes in this method"""

        try:
            self.client_name = self.tcinputs.get('ClientName')
            self.backupset_name = self.tcinputs.get('Backupset', 'browse_find_restore')
            self.storagepolicy_name = self.tcinputs.get('StoragePolicyName')
            self.subclients_count = self.tcinputs.get('SubclientsToCreate', 1)
            self.testdata_path = self.tcinputs.get('TestDataPath').split(';')
            self.restore_path = self.tcinputs.get('RestorePath')
            self.backup_cycle = self.tcinputs.get(
                'BackupCycle', 'FULL; EDIT; INC; EDIT; SYNTHETIC_FULL')

            self.cv_ops = CommonUtils(self.commcell)
            self.cv_entities = CVEntities(self.commcell)
            self.cl_machine = Machine(self.client_name, self.commcell)

            self.cl_delim = self.cl_machine.os_sep

            self.create_backupset()

            self.idx = validation.Validation({
                'commcell': self.commcell,
                'backupset': self.backupset,
                'debug': False
            })

            self.create_subclients()

        except Exception as exp:
            self.log.error(str(traceback.format_exc()))
            raise Exception(exp)

    def run(self):
        """Contains the core testcase logic and it is the one executed"""

        try:
            self.log.info("Started executing {0} testcase".format(self.id))

            self.log.info('Running backups to perform browse, find and restore')
            for subclient in self.subclients:
                self.run_sequence(subclient)

            self.log.info('Running browse/find/version combinations')
            self.run_combination()

            if self.combination_failed:
                self.log.error('Some combinations of browse, find, versions and restore failed')
                self.log.error('Setting testcase as failed')
                self.status = constants.FAILED
            else:
                self.log.info('All combinations of browse, find, versions and restore passed')
                self.log.info('Test case passed')
                self.status = constants.PASSED

        except Exception as exp:
            self.log.error('Test case failed with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED
            self.log.error(str(traceback.format_exc()))

    def create_backupset(self):
        """Creates a backupset and initializes testdata directory"""

        if self.testdata_path is None:
            raise Exception('Directory to create testdata not provided')

        bkset_props = self.cv_entities.create({
            'backupset': {
                'name': self.backupset_name,
                'client': self.client_name,
                'agent': 'File system',
                'instance': 'defaultinstancename'
            }
        })

        self.backupset = bkset_props['backupset']['object']

        bkset_paths = [self.cl_delim.join([path, self.backupset_name])
                       for path in self.testdata_path]

        for path in bkset_paths:
            if self.cl_machine.check_directory_exists(path):
                self.log.info('Deleting previous testdata directory [{0}]'.format(path))
                self.cl_machine.remove_directory(path)

    def create_subclients(self):
        """Creates subclients and initializes testdata directory"""

        for i in range(self.subclients_count):
            name = 'subclient_' + str(i+1)
            rand_id = commonutils.get_random_string(3)
            dir_name = name + '_' + rand_id
            content = [self.cl_delim.join([path, self.backupset_name, dir_name])
                       for path in self.testdata_path]

            sc_props = self.cv_entities.create({
                'subclient': {
                    'name': name,
                    'client': self.client_name,
                    'agent': 'File system',
                    'instance': 'defaultinstancename',
                    'backupset': self.backupset_name,
                    'storagepolicy': self.storagepolicy_name,
                    'content': content,
                    'description': self.id
                }
            })

            subclient = sc_props['subclient']['object']
            subclient.scan_type = 1

            self.subclients[name] = subclient

            self.idx.register_subclient(subclient)

            for path in content:
                self.log.info('Creating new testdata directory [{0}]'.format(path))
                self.cl_machine.create_directory(path)

            self.job_end_times[name] = []
            self.manual_testdata[name] = {}

    def run_sequence(self, subclient):
        """Executes a sequence of actions provided"""

        sequence = self.backup_cycle.lower()
        sequence = 'new; ' + sequence if 'new' not in sequence else sequence
        operations = [o.strip() for o in sequence.split(';')]

        self.log.info('Running sequence [{0}] for subclient [{1}]'.format(
            sequence.upper(), subclient))

        for operation in operations:
            if operation in ['full', 'incremental', 'differential', 'synthetic_full']:
                self.run_backup(subclient, operation)

            if operation in ['edit', 'new']:
                self.edit_testdata(subclient, operation)

    def run_backup(self, subclient, backup):
        """Runs a backup job and adds the job to Indexing validation"""

        subclient_obj = self.subclients.get(subclient)
        job_obj = self.cv_ops.subclient_backup(subclient_obj, backup_type=backup)

        if job_obj.job_id:
            self.idx.record_job(job_obj)

            self.log.info('JOB BASED FIND VALIDATION')
            ret_code, results = self.idx.validate_browse({
                'job_id': job_obj.job_id
            })

            if ret_code == -1:
                raise Exception('Job based find gave unexpected results')

            end_timetamp = commonutils.convert_to_timestamp(job_obj.end_time)
            self.job_end_times[subclient].append(end_timetamp)

        else:
            raise Exception('Cannot start backup job')

    def combination_thread(self, queue):
        while True:
            combination = queue.get()

            # Set Variables for readability
            id = combination[0],
            level = combination[1],
            op_type = combination[2],
            op_range = combination[3],
            show_deleted = combination[4],
            restore = combination[5],
            filters = combination[6]

            self.log.info('Executing combination #{0}'.format(id))

            the_subclient = None
            the_subclient_obj = None
            versions_check_file = None
            from_time = 0
            to_time = 0

            options = dict()
            options['restore'] = dict()
            options['operation'] = op_type
            options['show_deleted'] = show_deleted

            # LEVEL of browse
            if level == 'subclient':
                sc_list = list(self.subclients.keys())
                the_subclient = random.choice(sc_list)
                the_subclient_obj = self.subclients[the_subclient]
                options['subclient'] = the_subclient
            else:
                options['subclient'] = None

            # FROM, TO time
            from_time, to_time = self.get_timerange(op_range, the_subclient)
            options['from_time'] = from_time
            options['to_time'] = to_time

            # OPERATION type
            if op_type == 'browse':
                options['path'] = self.cl_delim

            elif op_type == 'find':
                options['path'] = '|**|*'.replace('|', self.cl_delim)

            elif op_type == 'versions':
                versions_check_file = self.get_versions_file(the_subclient, from_time, to_time)
                options['path'] = versions_check_file

            # FILTERS
            if filters is not None:
                options['filters'] = [('FileName', '*.txt')]

            # RESTORE options
            if restore:
                options['restore']['do'] = True
                options['restore']['dest_path'] = self.restore_path

                if op_type == 'browse' and level == 'subclient':
                    options['restore']['source_items'] = the_subclient_obj.content

                if op_type == 'versions':
                    options['restore']['source_items'] = [versions_check_file]
                    options['restore']['select_version'] = 'all'

            ret_code = self.idx.validate_browse_restore(options)

            if ret_code == -1:
                self.combination_failed = True
                self.log.error('Combination finished. Result [FAILED] - #{0}\n'.format(id))
            else:
                self.log.info('Combination finished. Result [PASSED] - #{0}\n'.format(id))

            queue.task_done()

    def run_combination(self):
        """Executes the browse/find/restore combinations one by one"""

        try:
            combinations = self.generate_combinations()
            self.log.info('Creating threads')

            q = Queue()
            num_threads = 5

            for i in range(num_threads):
                worker = Thread(target=self.combination_thread, args=(q,))
                worker.setDaemon(True)
                worker.start()

            for combination in combinations:
                q.put(combination)

            q.join()

        except Exception as exp:
            self.log.error(str(traceback.format_exc()))
            raise Exception(exp)

    def edit_testdata(self, subclient, action='edit'):
        """Edits the testdata of the subclients"""

        subclient_obj = self.subclients.get(subclient)
        sc_content = subclient_obj.content
        sc_manual_testdata = self.manual_testdata[subclient]

        for path in sc_content:
            if action == 'edit':
                self.cl_machine.modify_test_data(path, modify=True)
                self.edit_file(sc_manual_testdata['to_edit'])

                self.cl_machine.delete_file(sc_manual_testdata['to_delete'])
                sc_manual_testdata['to_delete'] = self.create_file(path)

                self.cl_machine.remove_directory(sc_manual_testdata['to_delete_dir'])
                sc_manual_testdata['to_delete_dir'] = self.create_folder(path)

            if action == 'new':
                self.cl_machine.generate_test_data(path, dirs=1)
                sc_manual_testdata = {
                    'to_edit': self.create_file(path),
                    'to_delete': self.create_file(path),
                    'to_delete_dir': self.create_folder(path)
                }

        self.manual_testdata[subclient] = sc_manual_testdata
        self.log.info('Manual testdata [{0}]'.format(json.dumps(self.manual_testdata)))

    @staticmethod
    def generate_combinations():
        """Prepares a list of browse/find/versions/restore combinations"""

        levels = ['backupset', 'subclient']
        operations = ['browse', 'find', 'versions']
        types = ['latest', 'timerange', 'point_in_time']
        show_deleted = [True, False]
        restore = [True]
        filters = [None]

        combinations = itertools.product(levels, operations, types, show_deleted, restore, filters)
        comb_list = list(combinations)

        # Adding manual list to the combination
        comb_list.append(['subclient', 'browse', 'latest', True, True, True])
        comb_list.append(['subclient', 'find', 'latest', True, True, True])

        comb_list_count = len(comb_list)
        combs = []

        # Appending ID to every combination
        for i in range(comb_list_count):
            comb = list(comb_list[i])
            comb.insert(0, i+1)
            combs.append(comb)

        return combs

    def get_timerange(self, type, subclient=None):
        """Gets a random timerange from the backups"""

        # Get all the timestamps from all the subclients, else from particular subclient
        if subclient is None:
            all_times = self.job_end_times.values()
            timestamps = list(itertools.chain.from_iterable(all_times))
        else:
            timestamps = self.job_end_times[subclient]

        timestamps = [commonutils.get_int(t) for t in timestamps]

        if len(timestamps) == 0:
            raise Exception('No backup jobs seems to have run because timestamps list is empty')

        t_count = len(timestamps)
        timestamps.sort()

        if type == 'timerange':
            etime_idx = random.randint(0, t_count-1)

            # If got the first item, then generate a dummy start time 1hr behind
            if etime_idx == 0:
                return timestamps[etime_idx]-3600, timestamps[etime_idx]

            # Get a start time index lesser than the end time index
            stime_idx = random.randint(0, etime_idx-1)

            # Adding 5 secs to from time because, we save job endtime in validation DB and taking
            # jobendtime as it is in timerange will cause problem since validation will
            # include that job but Indexing will not include the job

            return timestamps[stime_idx]+10, timestamps[etime_idx]

        if type == 'point_in_time':
            etime_idx = random.randint(0, t_count - 1)
            return 0, timestamps[etime_idx]

        if type == 'latest':
            return 0, 0

    def get_versions_file(self, subclient=None, from_time=0, to_time=0):
        """Gets a random file to do view all versions"""

        if to_time == 0:
            to_time = int(time.time())

        query = ("select path from indexing where jobendtime between {0} and {1} {2} and "
                 "type = 'file' and status in ('modified', 'new') and name like '%.txt' "
                 "order by jobid desc limit 1")

        sc_query = " and subclient = '{0}'".format(subclient) if subclient is not None else ''
        query = query.format(from_time, to_time, sc_query)

        response = self.idx.db.execute(query)

        random_file = ''
        if response.rowcount != 0:
            random_file = response.rows[0][0]

        return random_file

    def create_file(self, directory):
        """Creates a file with a random name"""

        file_name = commonutils.get_random_string(5)
        file_path = directory + self.cl_delim + file_name + '-' + str(int(time.time())) + '.txt'
        file_cnt = str(random.randint(0,9)) * 4096
        self.cl_machine.create_file(file_path, file_cnt)
        return file_path

    def edit_file(self, file_path):
        """Edits the content of the given file"""

        file_cnt = str(random.randint(0,20)) * 4096
        self.cl_machine.create_file(file_path, file_cnt)

    def create_folder(self, path):
        """Creates a folder with random name"""

        dir_name = commonutils.get_random_string(5)
        dir_path = path + self.cl_delim + dir_name
        self.cl_machine.generate_test_data(dir_path, dirs=1)
        return dir_path

    def tear_down(self):
        """Cleans the data created for Indexing validation"""

        self.idx.cleanup()