# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    set_registry_keys()  -- sets the registry keys UseCacheDB, UseAuxCopyReadLessPlus

    run_validations()    --  runs the validations

    tear_down()     --  tear down function of this test case

TcInputs to be passed in JSON File:

    PrimaryCopyMediaAgent:   Name of a MediaAgent machine - we create primary copy here
    SecondaryCopyMediaAgent: Name of a MediaAgent machine - we create secondary copy here

    Note: Both the MediaAgents can be the same machine

Steps:

1: Configure the environment: create a library,Storage Policy-with Primary, Secondary Copy,
                              a BackupSet,a SubClient

2: Set Properties: Use new read-less mode(Enable DiskRead) and maxCacheDBSizeMB: 4096

3: Run a Backup Job and then AuxCopy and run Validations(properties set)

4: Submit another Backup Job and again a AuxCopy Job

5: Run Validations(new read-less, Network Optimized, Cache used, Dedupe Occurred, properties set)

6: CleanUp the environment

"""

from AutomationUtils import constants
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import DedupeHelper, MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "Automation -Dedupe-New Readless AuxCopy with cache"
        self.tcinputs = {
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None
        }
        self.utility = None
        self.mm_helper = None
        self.ma_machine_1 = None
        self.ma_machine_2 = None
        self.dedupe_helper = None
        self.client_machine = None
        self.ddb_path = None
        self.mount_path = None
        self.client_path = None
        self.mount_path_2 = None
        self.content_path = None
        self.copy_ddb_path = None
        self.primary_ma_path = None
        self.secondary_ma_path = None
        self.subclient = None
        self.copy_name = None
        self.library_name = None
        self.library_name_2 = None
        self.storage_policy = None
        self.subclient_name = None
        self.backupset_name = None
        self.storage_policy_name = None
        self.list_primary = []
        self.list_secondary = []
        self.config_strings = None
        self.is_user_defined_lib = False
        self.is_user_defined_copy_lib = False
        self.is_user_defined_mp = False
        self.is_user_defined_copy_mp = False
        self.is_user_defined_dedup = False
        self.is_user_defined_copy_dedup = False

    def setup(self):
        """Setup function of this test case"""
        self.mm_helper = MMHelper(self)
        self.dedupe_helper = DedupeHelper(self)
        self.utility = OptionsSelector(self.commcell)
        self.client_machine = Machine(self.client.client_name, self.commcell)
        self.ma_machine_1 = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.ma_machine_2 = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        client_drive = self.utility.get_drive(self.client_machine, 2048)
        self.client_path = self.client_machine.join_path(client_drive, 'test_' + str(self.id))
        self.content_path = self.client_machine.join_path(self.client_path, 'content')
        self.copy_name = '%s%s' % (str(self.id), '_Copy')
        self.subclient_name = '%s%s' % (str(self.id), '_SC')
        self.backupset_name = '%s%s%s' % (str(self.id), '_BS_',
                                          str(self.tcinputs['SecondaryCopyMediaAgent'])[2:])
        self.storage_policy_name = '%s%s%s' % (str(self.id), '_SP_',
                                               str(self.tcinputs['SecondaryCopyMediaAgent'])[2:])

        if self.tcinputs.get('library_name'):
            self.is_user_defined_lib = True
        if self.tcinputs.get('mount_path'):
            self.is_user_defined_mp = True
        if self.tcinputs.get('dedup_path'):
            self.is_user_defined_dedup = True
        if self.tcinputs.get('copy_library_name'):
            self.is_user_defined_copy_lib = True
        if self.tcinputs.get('copy_mount_path'):
            self.is_user_defined_copy_mp = True
        if self.tcinputs.get('copy_dedup_path'):
            self.is_user_defined_copy_dedup = True

        if not self.is_user_defined_mp or not self.is_user_defined_dedup:
            ma_1_drive = self.utility.get_drive(self.ma_machine_1, 2048)
            self.primary_ma_path = self.ma_machine_1.join_path(ma_1_drive, 'test_' + str(self.id))
        if not self.is_user_defined_copy_mp or not self.is_user_defined_copy_dedup:
            ma_2_drive = self.utility.get_drive(self.ma_machine_2, 2048)
            self.secondary_ma_path = self.ma_machine_2.join_path(ma_2_drive,
                                                                 'test_' + str(self.id))

        if self.is_user_defined_lib:
            self.log.info("Existing library name supplied")
            self.library_name = self.tcinputs["library_name"]
        else:
            self.library_name = '%s%s%s' % (str(self.id), '_Lib1_',
                                            str(self.tcinputs['SecondaryCopyMediaAgent'])[2:])
            if not self.is_user_defined_mp:
                self.mount_path = self.ma_machine_1.join_path(self.primary_ma_path, 'MP1')
            else:
                self.log.info("custom mount_path supplied")
                self.mount_path = self.ma_machine_1.join_path(
                    self.tcinputs['mount_path'], 'test_' + self.id, 'MP1')

        if self.is_user_defined_copy_lib:
            self.log.info("Existing library name supplied for secondary copy")
            self.library_name_2 = self.tcinputs["copy_library_name"]
        else:
            self.library_name_2 = '%s%s%s' % (str(self.id), '_Lib2_',
                                              str(self.tcinputs['SecondaryCopyMediaAgent'])[2:])
            if not self.is_user_defined_copy_mp:
                self.mount_path_2 = self.ma_machine_2.join_path(self.secondary_ma_path, 'MP2')
            else:
                self.log.info("custom copy_mount_path supplied")
                self.mount_path = self.ma_machine_2.join_path(
                    self.tcinputs['copy_mount_path'], 'test_' + self.id, 'MP2')

        if self.is_user_defined_dedup:
            self.log.info("custom dedup path supplied")
            self.ddb_path = self.ma_machine_1.join_path(self.tcinputs["dedup_path"],
                                                        'test_' + self.id, "DDB")
        else:
            self.ddb_path = self.ma_machine_1.join_path(self.primary_ma_path, "DDB")

        if self.is_user_defined_copy_dedup:
            self.log.info("custom copydedup path supplied")
            self.copy_ddb_path = self.ma_machine_2.join_path(self.tcinputs["copy_dedup_path"],
                                                             'test_' + self.id, "CopyDDB")
        else:
            self.copy_ddb_path = self.ma_machine_2.join_path(self.secondary_ma_path, "CopyDDB")

        self.config_strings = ['Using 10.0 readless mode',
                               '\[Signatures processed .*?\]', '\[Found in cache .*?\]',
                               '\[Size-processed .*?\]', '\[Size-New Data- .*?\]',
                               'Encryption Type [3]', 'encryptionType [CV_DECRYPT_AND_ENCRYPT]',
                               '\[Size-Discarded .*?\]', 'Initializing Coordinator',
                               'Initializing Controller',
                               '[Coordinator] Number of streams allocated for the agent',
                               '[Reader_1] Initializing Worker Thread',
                               '^\d+\s*\w+\s*\d+\/\d+\s*\d+:\d+:\d+\s*\d+\s*\[Coordinator\] Initializing Coordinator:',
                               'SI Block Size [128 KB]']

    def run(self):
        """Run Function of This Case"""
        try:
            # 1: Setup the Environment
            # if Both MA's are same: Enable Registry Keys : UseCacheDB, UseAuxCopyReadLessPlus
            if self.tcinputs['PrimaryCopyMediaAgent'] == self.tcinputs['SecondaryCopyMediaAgent']:
                self.set_registry_keys()
            self.mm_helper.create_uncompressable_data(self.client.client_name,
                                                      self.content_path, 0.4)
            if not self.is_user_defined_lib:
                self.mm_helper.configure_disk_library(self.library_name,
                                                      self.tcinputs['PrimaryCopyMediaAgent'],
                                                      self.mount_path)
            if not self.is_user_defined_copy_lib:
                self.mm_helper.configure_disk_library(self.library_name_2,
                                                      self.tcinputs['SecondaryCopyMediaAgent'],
                                                      self.mount_path_2)

            self.storage_policy = self.dedupe_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['PrimaryCopyMediaAgent'],
                self.ma_machine_1.join_path(self.ddb_path, 'Dir'))
            storage_policy_copy = self.dedupe_helper.configure_dedupe_secondary_copy(
                self.storage_policy,
                self.copy_name,
                self.library_name_2,
                self.tcinputs['SecondaryCopyMediaAgent'],
                self.ma_machine_2.join_path(self.copy_ddb_path,
                                            'Dir' + self.utility.get_custom_str()),
                self.tcinputs['SecondaryCopyMediaAgent'],
                dash_full=True,
                source_side_disk_cache=True)
            self.mm_helper.configure_backupset(self.backupset_name)
            self.subclient = self.mm_helper.configure_subclient(content_path=self.content_path)

            # Remove Association with System Created AutoCopy Schedule
            self.mm_helper.remove_autocopy_schedule(self.storage_policy_name, self.copy_name)

            # 2: Set Properties: Use new read-less mode(Enable DiskRead) and maxCacheDBSizeMB: 4096
            storage_policy_copy.copy_client_side_dedup = False
            storage_policy_copy.copy_reencryption = (True, "AES", 256, False)

            query = '''update archgroupcopy
                    set maxCacheDBSizeMB = 4096 where id = %s''' % storage_policy_copy.copy_id
            self.utility.update_commserve_db(query)

            # 3: Run a Backup Job and then AuxCopy and run Validations(properties set)
            # 1st Iteration
            self.log.info('Submitting 1st Full Backup')
            backup_job = self.subclient.backup(backup_level='Full')
            if backup_job.wait_for_completion():
                self.log.info('1st Backup Completed :Id - %s', backup_job.job_id)
            else:
                raise Exception('1st Backup failed :Id - %s' % backup_job.job_id)

            self.log.info('Submitting AuxCopy job with scalable resource allocation')
            aux_copy_job = self.storage_policy.run_aux_copy(use_scale=True)
            if aux_copy_job.wait_for_completion():
                self.log.info('1st AuxCopy Completed :Id - %s', aux_copy_job.job_id)
            else:
                raise Exception('1st AuxCopy failed :Id - %s' % aux_copy_job.job_id)

            self.run_validations(1, backup_job.job_id, aux_copy_job.job_id)

            # 4: Submit another Backup Job and again a AuxCopy Job
            # 2nd Iteration
            self.log.info('Submitting 2nd Full Backup')
            backup_job = self.subclient.backup(backup_level='Full')
            if backup_job.wait_for_completion():
                self.log.info('2nd Backup Completed :Id - %s', backup_job.job_id)
            else:
                raise Exception('2nd Backup failed :Id - %s' % backup_job.job_id)

            self.log.info('Submitting AuxCopy job with scalable resource allocation')
            aux_copy_job = self.storage_policy.run_aux_copy(use_scale=True)
            if aux_copy_job.wait_for_completion():
                self.log.info('2nd AuxCopy Completed :Id - %s', aux_copy_job.job_id)
            else:
                raise Exception('2nd AuxCopy failed :Id - %s' % aux_copy_job.job_id)

            # 5: Run Validations (new read-less, Network Optimized, Cache used, Dedupe Occurred)
            self.run_validations(2, backup_job.job_id, aux_copy_job.job_id)

        except Exception as exe:
            self.status = constants.FAILED
            self.result_string = str(exe)
            self.log.error('Exception Occurred: %s', str(exe))

    def run_validations(self, iteration, backup_job_id, aux_copy_job_id):
        """Runs the validations
            Args:

                    iteration             (int)  --   Iteration of the Case Validations

                    backup_job_id         (str)  --   Id of Backup Job

                    aux_copy_job_id       (str)  --   Id of the AuxCopy Job
        """

        log_file = 'CVJobReplicatorODS.log'
        self.log.info('********* ITERATION %s *********', str(iteration))

        self.log.info('*** CASE 1: 10.0 Using readless Mode ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[0], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 2: Signatures Processed ***')
        signatures_processed = ''
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[1], aux_copy_job_id,
            escape_regex=False)
        if matched_line:
            signatures_processed = int(matched_string[0].split('-')[-1].split(']')[0].strip())
            self.log.info('Signatures Processed = %s', signatures_processed)
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error Result : Failed')
            self.status = constants.FAILED
        if iteration == 1:
            self.log.info('*** CASE 3: Found in cache - 0 ***')
        else:
            self.log.info('*** CASE 3: Found in cache != 0  but equals Signatures processed***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[2], aux_copy_job_id,
            escape_regex=False)
        if matched_line:
            found_in_cache = int(matched_string[0].split('-')[-1].split(']')[0].strip())
            self.log.info('Found in Cache: %s', found_in_cache)
            if iteration == 1 and found_in_cache == 0:
                self.log.info('Success Result : Passed')
            elif iteration == 2 and abs(signatures_processed - found_in_cache) <= 10:
                self.log.info('Success Result : Passed')
            else:
                self.log.error('Error  Result : Failed')
                self.status = constants.FAILED
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        check_string = self.config_strings[4]
        if iteration == 1:
            self.log.info("*** CASE 4: Size-processed == Size-Transferred ***")
        else:
            self.log.info("*** CASE 4: Size-processed == Size-Discarded ***")
            check_string = self.config_strings[7]
        compressed_size = ''
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[3], aux_copy_job_id,
            escape_regex=False)
        if matched_line:
            compressed_size = matched_string[0].split('-')[-1]
            self.log.info('Compressed Size = %s', compressed_size.split(']')[0])

        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            check_string, aux_copy_job_id,
            escape_regex=False)
        if matched_line and iteration == 1:
            transferred_size = matched_string[0].split('-')[-1]
            self.log.info('Size-New Data = %s', transferred_size.split(']')[0])
            if compressed_size.split(']')[0] == transferred_size.split(']')[0] + ' ':
                self.log.info('Success Result : Passed')
            else:
                self.log.error('Error  Result : Failed')
                self.status = constants.FAILED
        elif matched_line and iteration == 2:
            discarded_size = matched_string[0].split('-')[-1]
            # Giving tolerance up to 10 signatures for diff b/w compressed size and discarded size
            self.log.info('Size Discarded = %s', discarded_size.split(']')[0])
            if abs(int(compressed_size.split(']')[0].split()[0].strip()) -
                   int(discarded_size.split(']')[0].split()[0].strip())) <= (128*1024*10):
                self.log.info('Success Result : Passed')
            else:
                self.log.error('Error  Result : Failed')
                self.status = constants.FAILED
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 6: Did Dedupe Occur? ***')

        primary = self.dedupe_helper.get_primary_objects_sec(backup_job_id, self.copy_name)

        secondary = self.dedupe_helper.get_secondary_objects_sec(backup_job_id,
                                                                 self.copy_name)
        self.list_primary.append(int(primary))
        self.list_secondary.append(int(secondary))
        self.log.info('Primary Objects : %s', primary)
        self.log.info('Secondary Objects : %s', secondary)
        if iteration == 1:
            self.log.info('Validation will be done in next Iteration')
        else:
            if self.list_primary[0] == self.list_secondary[1] and self.list_primary[1] == 0:
                self.log.info('Success Result : Passed')
            else:
                self.log.error('Error  Result : Failed')
                self.status = constants.FAILED

        self.log.info("*** CASE 7: Encryption Type [3] ***")
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[5], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 8: Pipeline encryptionType [CV_DECRYPT_AND_ENCRYPT] ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[6], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('** CASE 9: Initializing Coordinator ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[8], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 10: Initializing Controller ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[9], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 11: [Coordinator] Number of streams allocated for agent ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[10], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 12: [Reader_1] Initializing Worker Thread ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[11], aux_copy_job_id)
        if matched_line:
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 13: Get pid of the CVJobReplicatorODS process ***')
        (matched_line, matched_string) = self.dedupe_helper.parse_log(
            self.tcinputs['PrimaryCopyMediaAgent'], log_file,
            self.config_strings[12], aux_copy_job_id,
            escape_regex=False)
        if matched_line:
            self.log.info('PID %s', (matched_string[0].split(' ')[-4]))
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error  Result : Failed')
            self.status = constants.FAILED

        self.log.info('*** CASE 14: ArchChunkToReplicate status ***')
        query = '''select distinct status
                from archchunktoreplicatehistory where AdminJobId = {0}
                '''.format(aux_copy_job_id)
        self.csdb.execute(query)
        row_1 = self.csdb.fetch_one_row()
        query = '''select distinct status
                from archchunktoreplicate where AdminJobId = {0}
                '''.format(aux_copy_job_id)
        self.csdb.execute(query)
        row_2 = self.csdb.fetch_one_row()
        if int(row_1[0]) == 2 or int(row_2[0]) == 2:
            self.log.info('ArchChunkToReplicate status for all chunks is 2')
            self.log.info('Success Result : Passed')
        else:
            self.log.error('Error Result : Fail')
            self.status = constants.FAILED

    def set_registry_keys(self):
        """Sets the Registry keys of the Client"""
        self.ma_machine_1.create_registry('MediaAgent', 'UseCacheDB', '1', r'DWord')
        self.ma_machine_1.create_registry('MediaAgent', 'UseAuxcopyReadlessPlus', '1', 'DWord')
        self.log.info("Same MA's: Registry Keys: UseCacheDB, UseAuxCopyreadlessPlus set")

    def tear_down(self):
        """Tear Down Function of this Case"""
        # 5: CleanUp the environment
        try:
            self.mm_helper.remove_content(self.content_path, self.client_machine)
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting backupset %s", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)
            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("Deleting storage policy  %s", self.storage_policy_name)
                self.commcell.storage_policies.delete(self.storage_policy_name)
                self.mm_helper.remove_content(self.ddb_path, self.ma_machine_1)
                self.mm_helper.remove_content(self.copy_ddb_path, self.ma_machine_2)
            if not self.is_user_defined_lib:
                if self.commcell.disk_libraries.has_library(self.library_name):
                    self.commcell.disk_libraries.delete(self.library_name)
                self.mm_helper.remove_content(self.mount_path, self.ma_machine_1)
            if not self.is_user_defined_copy_lib:
                if self.commcell.disk_libraries.has_library(self.library_name_2):
                    self.commcell.disk_libraries.delete(self.library_name_2)
                self.mm_helper.remove_content(self.mount_path_2, self.ma_machine_2)
            if self.tcinputs['PrimaryCopyMediaAgent'] == self.tcinputs['SecondaryCopyMediaAgent']:
                # if both MA's are same, remove registry keys set previously
                self.ma_machine_1.remove_registry('MediaAgent', 'UseCacheDB')
                self.ma_machine_1.remove_registry('MediaAgent', 'UseAuxcopyReadlessPlus')
        except Exception as exe:
            self.log.error('ERROR in TearDown. Might need to Cleanup Manually: %s', str(exe))
