# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    previous_run_cleanup() -- for deleting the left over backupset
    and storage policy from the previous run

    run_full_backup_job()   --  run a full backup job

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case


this test case will check if client side dedupe with no cache is working or not,
having 2 partitions in the dedupe store and this all should be in pipeline data transfer mode.

prerequisites: None

the testcase will change the following settings but
they will be reverted after the test case finishes its run:
    1. client side dedupe will be set to enable.
    2. use of client side cache will be set to off.
    3. for data transfer pipeline mode will be turned on.

inputs required for the testcase:
            "ClientName"
            "AgentName"
            "MediaAgentName"

Design Steps:
1.	Resources
2.	Client side dedupe on, cache off, pipeline mode on
3.	Over two backup jobs
4.	Check in clbackup log
5.	Client side dedup enabled (log + query)
6.	Client Cache not enabled (log size 0, query)
7.	Network transfer bytes (first > second)
8.	Dedup occur or not
"""

import os
from AutomationUtils import constants, cvhelper
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils import mahelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Client side dedupe case with no cache case having 2 splits in pipeline mode"

        self.tcinputs = {
            "ClientName": None,
            "AgentName": None,
            "MediaAgentName": None
        }
        self.mount_path = None
        self.dedup_store_path = None
        self.content_path = None
        self.library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.mm_helper = None
        self.dedup_helper = None
        self.opt_selector = None
        self.storage_policy_id = None
        self.sidb_id = None
        self.substore_id = None
        self.client_machine = None
        self.media_agent_machine = None
        self.testcase_path = None
        self.password = None
        self.media_agent_client_id = None
        self.testcase_path_client = None
        self.testcase_path_media_agent = None
        self.library = None
        self.storage_policy = None
        self.backup_set = None
        self.subclient = None

    def setup(self):
        """Setup function of this test case"""
        self.library_name = str(self.id) + "_lib"
        self.storage_policy_name = str(self.id) + "_SP"
        self.backupset_name = str(self.id) + "_BS"
        self.subclient_name = str(self.id) + "_SC"
        self.dedup_helper = mahelper.DedupeHelper(self)
        self.mm_helper = mahelper.MMHelper(self)
        self.opt_selector = OptionsSelector(self.commcell)
        self.client_machine = Machine(
            self.tcinputs["ClientName"], self.commcell)
        self.media_agent_machine = Machine(
            self.tcinputs["MediaAgentName"], self.commcell)
        self.media_agent_client_id = str(self.commcell.clients.get
                             (self.tcinputs["MediaAgentName"])._get_client_id())

    def previous_run_clean_up(self):
        """delete the resources from previous run"""
        self.log.info("********* previous run clean up **********")
        try:
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.agent.backupsets.delete(self.backupset_name)
            if self.commcell.storage_policies.has_policy(
                    self.storage_policy_name):
                self.commcell.storage_policies.delete(self.storage_policy_name)
            self.log.info("previous run clean up COMPLETED")
        except Exception as exp:
            self.log.info("previous run clean up ERROR")
            self.log.info("ERROR:%s", exp)

    def run_full_backup_job(self):
        """function for running full backup job"""
        self.log.info("Starting backup job")
        job = self.subclient.backup("FULL")
        self.log.info("Backup job: %s", str(job.job_id))
        if not job.wait_for_completion():
            raise Exception("Job {0} Failed with {1}".
                            format(job.job_id, job.delay_reason))
        self.log.info("job %s complete", job.job_id)
        return job

    def run(self):
        """Run function of this test case"""
        try:
            self.previous_run_clean_up()
            dict_prim = {}  # to be used later for comparing primary
            # /secondary objects
            dict_sec = {}
            dict_nw_transfer = {}  # to be used later for network
            # transfer bytes verification

            # create the required resources for the testcase
            # get the drive path with required free space

            drive_path_client = self.opt_selector.get_drive(
                self.client_machine)
            drive_path_media_agent = self.opt_selector.get_drive(
                self.media_agent_machine)

            # creating testcase directory, mount path, content path, dedup
            # store path
            self.testcase_path_client = "%s%s" % (drive_path_client, self.id)
            self.testcase_path_media_agent = "%s%s" % (drive_path_media_agent, self.id)

            self.mount_path = self.media_agent_machine.join_path(self.testcase_path_media_agent,
                                           "mount_path")
            self.dedup_store_path = self.media_agent_machine.join_path(
                self.testcase_path_media_agent, "dedup_store_path")

            self.content_path = self.client_machine.join_path(
                self.testcase_path_client, "content_path")
            if self.client_machine.check_directory_exists(self.content_path):
                self.log.info("content path directory already exists")
                self.client_machine.remove_directory(self.content_path)
                self.log.info("old content path deleted")
            self.client_machine.create_directory(self.content_path)
            self.log.info("content path created")

            # create library
            self.library = self.mm_helper.configure_disk_library(
                self.library_name, self.tcinputs["MediaAgentName"], self.mount_path)
            # create SP
            self.storage_policy = self.dedup_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['MediaAgentName'],
                self.dedup_store_path)

            # use the storage policy object
            # from it get the storage policy id
            # get the sidb store id and sidb sub store id
            return_list = self.dedup_helper.get_sidb_ids(
                self.storage_policy.storage_policy_id, "Primary")
            self.sidb_id = int(return_list[0])
            self.substore_id = int(return_list[1])

            # add partition in SP
            self.log.info("adding split for the dedup store")
            self.storage_policy.add_ddb_partition(self.storage_policy.storage_policy_id,
                                                  str(self.sidb_id),
                                                  self.media_agent_machine.join_path(self.dedup_store_path,
                                                                                     "partition2"),
                                                  self.tcinputs["MediaAgentName"])

            # create backupset
            self.backup_set = self.mm_helper.configure_backupset(self.backupset_name, self.agent)

            # generate the content
            if self.mm_helper.create_uncompressable_data(
                    self.tcinputs['ClientName'], self.content_path, 1, 1):
                self.log.info("generated content for subclient %s", self.subclient_name)

            # create subcient and add subclient content
            self.subclient = self.mm_helper.configure_subclient(
                self.backupset_name,
                self.subclient_name,
                self.storage_policy_name,
                self.content_path,
                self.agent)

            # turn client side dedupe on
            # turn use of cache off
            self.log.info("enabling client side dedupe with no cache")
            self.client.set_dedup_property(
                "clientSideDeduplication",
                "ON_CLIENT",
                client_side_cache=False)
            self.log.info("enabling client side dedupe with no cache: Done")

            # turn pipeline mode on
            cs_machine_obj = Machine(self.commcell.commserv_client)
            encrypted_pass = cs_machine_obj.get_registry_value(
                r"Database", "pAccess")
            self.password = cvhelper.format_string(
                self._commcell, encrypted_pass).split("_cv")[1]
            self.mm_helper.set_opt_lan("disable", self.commcell.commserv_name +
                                       "\\commvault", "sqladmin_cv",
                                       self.password, self.media_agent_client_id)
            # sdt mode turned off
            self.log.info("pipeline mode turned on")
            self.log.info("set up of the environment for "
                          "the testcase is completed")

            # for twice run
            # Run FULL backup
            self.log.info("Running full backup...")
            for iterator in range(1, 3):
                job = self.run_full_backup_job()
                # do the checks
                log_file = "clBackup.log"
                config_strings_clbackup = ['[isClientSideDedupEnabled - yes]',
                                           '[CacheDBSize - 0 MB]',
                                           'Using default pipeline mode [Pipeline] for [Backup] pipe']
                error_flag = []
                self.log.info("*************** Validations ****************")
                self.log.info(
                    "CASE 1: check if the client side Deduplication is enabled ")
                (matched_line, matched_string) = self.dedup_helper.parse_log(
                    self.client.client_hostname, log_file, config_strings_clbackup[0], job.job_id)

                query = """
                    SELECT      attrVal
                    FROM        APP_Client, APP_ClientProp
                    WHERE       APP_Client.name='{0}'
                    AND         APP_ClientProp.componentNameId = APP_Client.id
                    AND         attrName='Enable Deduplication'
                    AND         APP_ClientProp.modified = 0
                """.format(self.tcinputs["ClientName"])
                self.log.info("EXECUTING QUERY %s", query)
                self.csdb.execute(query)

                result = int(self.csdb.fetch_one_row()[0])
                if result == 1:        # 1 stands for de duplication enabled on client
                    self.log.info(
                        "query returned: Client side dedup is enabled")

                if matched_line or (result == 1):
                    self.log.info("Result: Pass")
                else:
                    self.log.error("Result: Failed")
                    error_flag += ["failed to find: " +
                                   config_strings_clbackup[0]]

                self.log.info(
                    "CASE 2: check if the client side Cache is disabled ")
                (matched_line, matched_string) = self.dedup_helper.\
                    parse_log(self.client.client_hostname, log_file,
                              config_strings_clbackup[1], job.job_id)

                query = """
                        SELECT      attrVal
                        FROM        APP_Client, APP_ClientProp
                        WHERE       APP_Client.name='{0}'
                        AND         APP_ClientProp.componentNameId = APP_Client.id
                        AND         attrName='Enable Client Signature Cache'
                        AND         APP_ClientProp.modified = 0
                        """.format(self.tcinputs["ClientName"])
                self.log.info("EXECUTING QUERY %s", query)
                self.csdb.execute(query)
                result = int(self.csdb.fetch_one_row()[0])
                if result == 0:  # 0 stands for client side cache being disabled
                    self.log.info(
                        "query returned: Client side cache is disabled")

                if matched_line or (result == 0):
                    self.log.info("Result: Pass")
                else:
                    self.log.error("Result: Failed")
                    error_flag += ["failed to find: " +
                                   config_strings_clbackup[1]]


                self.log.info("CASE 3: -- Checking from logs if "
                              "pipeline mode has been activated")
                (matched_line, matched_string) = self.dedup_helper.parse_log(
                    self.client.client_hostname, log_file, config_strings_clbackup[2], job.job_id)

                if matched_line:
                    self.log.info("Result: Pass _ Pipeline mode has been activated")
                else:
                    self.log.error("Result: Failed")
                    error_flag += ["failed to find: " +
                                   config_strings_clbackup[2]]


                self.log.info(
                    "CASE 4: ------------------"
                    "-Validating: Network transfer Bytes -----------------  ")
                # client to ma flow
                # first backup job, all new data, generate signaturesult,
                # since first backup job then MA not seen the
                # signatures earlier, send alot of data to MA
                # (signatures + unique blocks)
                # second backup job signatures generated at client,
                # no cache at client, since MA seen the signature
                # earlier, not unique MA will tell the client,
                # 100% dedupe, then send only the signatures
                # thus data sent after the second backup job
                # will be lesser than the data sent after the first
                # backup job

                network_bytes = self.dedup_helper.get_network_transfer_bytes(
                    job.job_id)
                self.log.info("Network transferred bytes: " + network_bytes)
                dict_nw_transfer[iterator] = network_bytes
                self.log.info("%s = %s" %
                              (iterator, dict_nw_transfer[iterator]))
                if iterator == 2 and int(dict_nw_transfer[iterator])\
                        < int(dict_nw_transfer[iterator - 1]):
                    self.log.info("Network transfer Bytes validation: Pass")
                else:
                    if iterator == 1:
                        self.log.info(
                            "validation will be done at the end of next iterator")
                    else:
                        self.log.error(
                            "Network transfer bytes validation: Fail")
                        error_flag += ["Network transfer bytes validation: Fail"]

                self.log.info("CASE 5: Did Dedupe Occur correctly "
                              "? comparing the primary and secondary objects "
                              "in the Dedup store ")
                primary_objects_count = int(self.dedup_helper.get_primary_objects(job.job_id))
                self.log.info("Primary objects: %d", primary_objects_count)
                dict_prim[iterator] = primary_objects_count
                secondary_objects_count = int(self.dedup_helper.get_secondary_objects(job.job_id))
                self.log.info("Secondary objects: %d", secondary_objects_count)
                dict_sec[iterator] = secondary_objects_count

                # now if in second iteration then we check second
                # iter backup sec equals first iter backup primary
                # and vice versa
                if iterator == 2 and dict_sec[iterator] == dict_prim[iterator - 1] \
                        and dict_prim[iterator] == dict_sec[iterator - 1]:
                    self.log.info("Dedupe validation: Pass")
                else:
                    if iterator == 1:
                        self.log.info(
                            "validation will be done at the end of next iteration")
                    else:
                        self.log.error("Dedupe validation: Fail")
                        error_flag += ["Dedupe validation: Fail"]

            if error_flag:
                # if the list is not empty then error was there, fail the test
                # case
                self.log.info(error_flag)
                raise Exception("testcase failed")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.resultant_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        try:
            self.log.info("*********************************************")
            self.log.info("Restoring defaults")

            # set the deduplication back to default
            self.log.info("setting client deduplication to default:"
                          " Use storage policy settings ")
            self.client.set_dedup_property("clientSideDeduplication",
                                           "USE_SPSETTINGS")
            self.log.info("setting client deduplication to default"
                          " Use storage policy settings: Done")

            # set the data transfer settings to default:
            # using optimised for concurrent LAN connections
            self.log.info("setting the data transfer "
                          "settings to default: using optimised "
                          "for concurrent LAN connections")
            self.mm_helper.set_opt_lan(
                "enable",
                self.commcell.commserv_name +
                "\\commvault",
                "sqladmin_cv",
                self.password,
                self.media_agent_client_id)
            self.log.info("setting the data transfer"
                          " settings to default: Done")

            # delete the generated content for this testcase
            # self.client_machine object initialised earlier
            self.client_machine.remove_directory(self.content_path)
            self.log.info("Deleted the generated data")

            self.log.info("deleting backupset and SP of the test case")
            self._agent.backupsets.delete(self.backupset_name)
            self.log.info("backup set deleted")

            self.commcell.storage_policies.delete(self.storage_policy_name)
            self.log.info("storage policy deleted")

            self.media_agent_machine.remove_directory(self.dedup_store_path)
            self.commcell.disk_libraries.delete(self.library_name)
            self.client_machine.remove_directory(self.mount_path)

            self.log.info("clean up successful")

        except Exception as exp:
            self.log.info("clean up not successful")
            self.log.info("ERROR: %s", exp)
