# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

    run_validations()    --  runs the validations

Inputs to be passed in JSON File:

    PrimaryCopyMediaAgent:   Name of a MediaAgent machine - we create primary copy here
    SecondaryCopyMediaAgent: Name of a MediaAgent machine - we create secondary copies here

    Note: Both the MediaAgents can be the same machine

Steps:

1: Configure the environment: create a library,Storage Policy-with Primary, Secondary Copy,
                              (DSA disabled for the Policy), a BackupSet,a SubClient

2: Run a Backup Job and then an AuxCopy Job

3: Run DB Validations whether DSA is disabled? and is DSA not working?

4: Enable DSA, Create a new Secondary Copy and run AuxCopy again

5: Run DB Validations whether DSA is enabled? and is DSA working?

6: CleanUp the environment

"""

from AutomationUtils import constants
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import MMHelper, DedupeHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = 'AuxCopy Without DSA'
        self.tcinputs = {
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None
        }
        self.mm_helper = None
        self.ma_machine_1 = None
        self.ma_machine_2 = None
        self.dedupe_helper = None
        self.client_machine = None
        self.ma_1_path = None
        self.ma_2_path = None
        self.mount_path = None
        self.client_path = None
        self.mount_path_2 = None
        self.content_path = None
        self.subclient = None
        self.copy_name = None
        self.library_name = None
        self.storage_policy = None
        self.subclient_name = None
        self.backupset_name = None
        self.storage_policy_name = None
        self.secondary_copy_name = None

    def setup(self):
        """Setup function of this test case"""
        self.client_machine = Machine(self.client.client_name, self.commcell)
        self.ma_machine_1 = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.ma_machine_2 = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        utility = OptionsSelector(self.commcell)
        separator = self.ma_machine_1.os_sep
        separator_2 = self.ma_machine_2.os_sep
        client_separator = self.client_machine.os_sep
        client_drive = utility.get_drive(self.client_machine, 2048)
        ma_1_drive = utility.get_drive(self.ma_machine_1, 2048)
        ma_2_drive = utility.get_drive(self.ma_machine_2, 2048)
        self.client_path = '%s%s%s%s' % (client_drive, 'test_', str(self.id), client_separator)
        self.ma_1_path = '%s%s%s%s' % (ma_1_drive, 'test_', str(self.id), separator)
        self.ma_2_path = '%s%s%s%s' % (ma_2_drive, 'test_', str(self.id), separator_2)
        self.mount_path = self.ma_1_path + 'MP'
        self.mount_path_2 = self.ma_2_path + 'MP2'
        self.content_path = self.client_path + 'Content'
        self.copy_name = str(self.id) + '_Copy'
        self.library_name = str(self.id) + '_Lib'
        self.backupset_name = str(self.id) + '_BS'
        self.subclient_name = str(self.id) + '_SC'
        self.storage_policy_name = str(self.id) + '_SP'
        self.secondary_copy_name = str(self.id) + '_Copy2'
        self.mm_helper = MMHelper(self)
        self.dedupe_helper = DedupeHelper(self)

    def run(self):
        """Run Function of this case"""
        try:
            # 1: Configure the environment
            self.mm_helper.create_uncompressable_data(self.client.client_name,
                                                      self.content_path, 0.5)
            self.mm_helper.configure_disk_library(self.library_name,
                                                  self.tcinputs['PrimaryCopyMediaAgent'],
                                                  self.mount_path)
            self.mm_helper.configure_disk_library(self.library_name + '_2',
                                                  self.tcinputs['SecondaryCopyMediaAgent'],
                                                  self.mount_path_2)

            # Create SP with 50 Streams. By default it is 50
            self.storage_policy = self.mm_helper.configure_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['PrimaryCopyMediaAgent'])
            # Disabling the DSA
            self.log.info('Disabling DSA for StoragePolicy')
            self.storage_policy.modify_dynamic_stream_allocation(False)
            self.mm_helper.configure_secondary_copy(self.copy_name,
                                                    self.storage_policy_name,
                                                    self.library_name + '_2',
                                                    self.tcinputs['SecondaryCopyMediaAgent'])
            self.storage_policy.refresh()
            self.mm_helper.configure_backupset(self.backupset_name)
            self.subclient = self.mm_helper.configure_subclient(content_path=self.content_path)
            # Remove Association with System Created AutoCopy Schedule
            self.mm_helper.remove_autocopy_schedule(self.storage_policy_name, self.copy_name)

            # 2: Run a Backup Job and then an AuxCopy Job
            self.log.info('Running Backup Job')
            backup_job = self.subclient.backup(backup_level='Full')
            if not backup_job.wait_for_completion():
                raise Exception('Backup Job Failed: %s' % backup_job.job_id)
            self.log.info('Backup Job Completed: %s', backup_job.job_id)

            self.log.info('Running AuxCopy Job')
            aux_copy_job = self.storage_policy.run_aux_copy(use_scale=False)
            if not aux_copy_job.wait_for_completion():
                raise Exception('AuxCopy Job Failed: %s' % aux_copy_job.job_id)
            self.log.info('AuxCopy Job Completed: %s', aux_copy_job.job_id)

            # 3: Run DB Validations whether DSA is disabled? and is DSA not working?
            self.run_validations(1, aux_copy_job.job_id)

            # 4: Enable DSA, Create a new Secondary Copy and run AuxCopy again
            # Enabling DSA for the Storage Policy
            self.log.info('Enabling DSA for StoragePolicy')
            self.storage_policy.modify_dynamic_stream_allocation(True)

            self.log.info('Creating Second SP copy for old Aux with DSA Case')
            self.mm_helper.configure_secondary_copy(self.secondary_copy_name,
                                                    self.storage_policy_name,
                                                    self.library_name + '_2',
                                                    self.tcinputs['SecondaryCopyMediaAgent'])
            self.mm_helper.remove_autocopy_schedule(self.storage_policy_name,
                                                    self.secondary_copy_name)
            self.storage_policy.refresh()

            self.log.info('Running AuxCopy Job')
            aux_copy_job = self.storage_policy.run_aux_copy(use_scale=False)
            if not aux_copy_job.wait_for_completion():
                raise Exception('AuxCopy Job Failed: %s' % aux_copy_job.job_id)
            self.log.info('AuxCopy Job Completed: %s', aux_copy_job.job_id)

            # 5: Run DB Validations whether DSA is enabled? and is DSA working?
            self.run_validations(2, aux_copy_job.job_id)

        except Exception as exe:
            self.status = constants.FAILED
            self.result_string = str(exe)
            self.log.error('Case Failed. Exception %s', str(exe))

    def run_validations(self, case_number, job_id):
        """Runs The Validations
            Args:

                case_number             (int)  --   1- DSA Disabled
                                                    2- DSA Enabled

                job_id                  (str)  --   Id of AuxCopy Job
            """
        self.log.info('******* Validations *******')

        to_check = ['Disabled', 'Enabled']
        self.log.info('**** Validation 1 : DSA %s****', to_check[case_number - 1])
        query = """select flags & 131072
                from archGroup where name = '{0}'""".format(self.storage_policy_name)
        self.log.info('QUERY: %s', query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        if int(row[0]) == 0:
            flag = 'Disabled'
        else:
            flag = 'Enabled'
        if case_number == 1 and flag == 'Disabled':
            self.log.info('Validation Success - DSA %s', flag)
        elif case_number == 2 and flag == 'Enabled':
            self.log.info('Validation Success - DSA %s', flag)
        else:
            self.log.error('Validation Failed - DSA %s', flag)
            self.status = constants.FAILED

        to_check = ['<>0', '=0']
        self.log.info('**** Validation 2 : No Entries with Segment Id%s ****',
                      to_check[case_number - 1])
        query = """select count(*) from archchunktocopyDSA
                where AdminJobId = {0} and segmentID{1}
                """.format(job_id, to_check[case_number - 1])
        self.log.info('QUERY: %s', query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        if int(row[0]) == 0:
            self.log.info('Validation Passed : No entries with SegmentId%s',
                          to_check[case_number - 1])
        else:
            self.log.error('Validation Failed : Populated entries with SegmentId%s',
                           to_check[case_number - 1])
            self.status = constants.FAILED

    def tear_down(self):
        """Tear Down Function of this Case"""
        try:
            # 6: CleanUp the environment
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting backupset %s", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)

            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("Deleting storage policy  %s", self.storage_policy_name)
                self.commcell.storage_policies.delete(self.storage_policy_name)

            if self.commcell.disk_libraries.has_library(self.library_name):
                self.commcell.disk_libraries.delete(self.library_name)
            if self.commcell.disk_libraries.has_library(self.library_name + '_2'):
                self.commcell.disk_libraries.delete(self.library_name + '_2')
            self.mm_helper.remove_content(self.ma_1_path, self.ma_machine_1)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['SecondaryCopyMediaAgent']:
                self.mm_helper.remove_content(self.ma_2_path, self.ma_machine_2)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.client.client_name:
                if self.tcinputs['SecondaryCopyMediaAgent'] != self.client.client_name:
                    self.mm_helper.remove_content(self.client_path, self.client_machine)
        except Exception as exe:
            self.log.error('ERROR in TearDown. Might need to Cleanup Manually: %s', str(exe))
