
# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright ©2018 Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import constants
from Application.AD.ms_ad import ADOps, cv_ad_objectscomapre
from Application.AD.exceptions import ADException

class TestCase(CVTestCase):
    """Class for executing Basic functionality of Lotus Notes Database agent test case"""

    def __init__(self):
        """Initializes test case class object
            Properties to be initialized:
                name                    (str)           --  name of this test case
                applicable_os           (str)           --  applicable os for this test case
                product                 (str)           --  applicable product for AD
                features                (str)           --  Backup and Restore
                show_to_user            (bool)          --  True, Customer should see this test case
                tcinputs                (dict)          --  dict of test case inputs
                    ClientName    (str)    -- required
                                                ClientName (Required)
                    AgentName     (str)    -- required
                                                "ACTIVE DIRECOTRY"
                    BackupsetName (str)    -- optional
                        Backupset Name,will use "defaultbaupset" if not specified
                    InstanceName  (str)    -- optional
                        Instance Name, will use "defaultinstancename" if not specified
                    SubclientName (str)    -- optional
                        Subclient Name, will use "TC_casenumber" if not specified
                    StoragePolicy (str)    -- optional
                        Storage Policy used for subclient
                    AD_server     (str)    -- optional
                        AD server to connect. wil connect to agent machine
                    AD_user       (str)    -- required
                        User to access AD
                        Currenlty, we need input, should get from cs
                    AD_password   (str)    -- required
                        password for AD_user
                subclientname      (str)        -- subclientclient name
                ad_ins             (object)     -- Application AD object
                ad_basedn          (str)        -- AD object base DN
        """
        super().__init__()
        self.name = "Active Directory - Basic Backup & Restore"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.ACTIVEDIRECTORY
        self.feature = self.features_list.DATAPROTECTION
        self.show_to_user = True
        self.ad_ins = None
        self.ad_basedn = ""
        self.domainname = ""
        self.subclientname = ""
        self.tcinputs = {
            "ClientName" : None,
            "AgentName" : "ACTIVE DIRECTORY",
            "StoragePolicy" : None,
            "AD_user" : None,
            "AD_password" : None,
        }
        if "cleanup" in self.tcinputs:
            self.cleanup_objs = self.tcinputs['cleanup']
        else:
            self.cleanup_objs = True
    def setup(self):
        """Setup function of this test case
            This function will do the following steps:
             1. create object for backupset and subclient if answer file doens't provide
        """
        self.log.debug(f"start to process testcase setup {self.name}")
        try:
            if "AD_server" not in self.tcinputs:
                self.log.debug("No AD server defined, will use agent hostname")
                ad_server = self.client.client_hostname
            else:
                ad_server = self.tcinputs['AD_server']
            ad_user = self.tcinputs['AD_user']
            ad_password = self.tcinputs['AD_password']
            self.log.debug(f"""
Will create AD objects with the following information:
server: {ad_server}
user: {ad_user}
password: {ad_password}""")
            self.ad_ins = ADOps(server=ad_server,
                                user=ad_user,
                                password=ad_password,
                                log=self.log)
            self.ad_basedn = self.ad_ins.ldap_info['basedn']
            self.log.debug(f"The AD object base dn is {self.ad_basedn}")
            if "BackupsetName" not in self.tcinputs:
                self.log.debug("No backupset is found in answer file, use defaultbackupset")
                self._backupset = self._agent.backupsets.get("defaultbackupset")
                self.log.debug(f"""
backupset object is created. The object is:
{self._backupset}""")

            if "SubclientName" not in self.tcinputs or self.tcinputs.get("SubclientName") is None:
                self.log.debug("No subclient is found in asnwer file, use default Name")
                self.subclientname = "TC_{0}".format(str(self.id))
                self.log.debug(f"""
No subclient name is assigned,will use default subclinet name:
{self.subclientname}""")
            else:
                self.subclientname = self.tcinputs.get("SubclientName")
            sc_content = [f"OU={self.ad_basedn},OU=Automation,OU={str(self.id)}"]
            self.subclient = self.backupset.check_subclient(\
                                self.backupset,\
                                self.subclientname,\
                                storagepolicy=self.tcinputs.get("StoragePolicy"),\
                                subclientcontent=sc_content)
            self.log.debug(f"""
subclient object is created. the object is:{self.subclient}""")
            self.log.debug("setup phase is completed")
        except ADException as exp:
            self.status = constants.FAILED
            self.log.exception("there is exception happened, here is the detail %s" % \
                               exp.report)
        except Exception as exp:
            self.status = constants.FAILED
            self.log.exception("there is not AD exception happened, here is the detail %s" % \
                               exp)
    def run(self):
        """Run function of this test case"""
        try:
            self.log.debug(f"Get all Content from subclient {self.subclientname}")
            sc_contents = self.subclient.content
            sc_contentsdisplay = '\n'.join(sc_contents)
            self.log.debug(f"""
current subclient {self.subclientname} has the following content:
{sc_contentsdisplay}""")
            self.log.debug("Convert subclient content to AD format")
            ad_content = self.subclient.cv_contents(sc_contents, entrypoint=self.ad_ins.basedn)
            self.log.debug(f"subclient content in AD format:{ad_content}")
            ad_objlists_base = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug(f"here is the AD objects in content:{ad_objlists_base}")
            if self.cleanup_objs:
                self.log.debug("Will clean up all objects in the entry")
                self.ad_ins.cv_ugo_delete(ad_objlists_base, ad_content)
                self.log.debug(f"the following objects are deleted: {ad_objlists_base}")
            new_ad_objs = []
            new_ad_objs += self.ad_ins.ugo_package(entrypoint=ad_content[0][1],
                                                   prestring="Base",
                                                   fixname="Objs")
            self.log.debug("get all AD objects in existing content")



            self.log.debug("start a full backup first ")
            jobf = self.subclient.backup(backup_level="Full")
            self.log.debug(f"full backup is started, job id is {jobf.job_id}")
            jobresult = jobf.wait_for_completion()
            self.log.debug(f"""
backup job {jobf.job_id} is completed. The status is {jobresult}""")
            self.log.debug("start a restore job after full backup")
            jobfr = self.subclient.restore_in_place(paths=["\\"])
            self.log.debug(f"restore job started, job id is {jobfr.job_id}")
            jobresult = jobfr.wait_for_completion()
            self.log.debug(f"""
restore job {jobfr.job_id} is completed. The status is {jobresult}""")



            self.log.debug("add more AD objects before run inc jobs")
            new_ad_objs += self.ad_ins.ugo_package(entrypoint=ad_content[0][1],
                                                   prestring="Inc")
            self.log.debug("check ad objects after add new objects")
            ad_objlists_inc1 = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug(f"""
here is the AD object after add new entry:{ad_objlists_inc1}""")
            self.log.debug("start a incremental jobs")
            jobi = self.subclient.backup(backup_level="Incremental")
            self.log.debug(f"""
incremental backup is started, job id is {jobi.job_id}""")
            jobresult = jobi.wait_for_completion()
            self.log.debug(f"""
backup job {jobi.job_id} is completed. The status is {jobresult}""")
            self.ad_ins.cv_ugo_delete(new_ad_objs, ad_content)
            ad_objlists_inc_delete = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug("check if all objects are deleted")
            result, diff_check = cv_ad_objectscomapre(ad_objlists_inc_delete,
                                                      None,
                                                      diff_objs_list=new_ad_objs,
                                                      judgement_value=True)
            if result:
                self.log.debug(f"""
no email found in list, {len(diff_check)} objects are deleted""")
            self.log.debug("run restore from latest after incremental")
            jobr = self.subclient.restore_in_place(paths=["\\"])
            jobresult = jobr.wait_for_completion()
            self.log.debug(f'restore job {jobr.job_id} is completed')
            ad_objlists_inc_restore = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug("check restore result after inc")
            result, diff_check = cv_ad_objectscomapre(ad_objlists_inc_restore,
                                                      None,
                                                      diff_objs_list=new_ad_objs,
                                                      judgement_value=False)
            if result:
                self.log.debug(f"""
all {len(diff_check)} found in list, the new objects are restored""")



            self.log.debug("create more AD objects after run first incremental")
            new_ad_objs += self.ad_ins.ugo_package(entrypoint=ad_content[0][1],
                                                   prestring="Sync")
            self.log.debug("AD objects are created")
            self.log.debug("run sync with inc first")
            jobs = self.subclient.backup(backup_level="synthetic_full", incremental_backup=True)
            self.log.debug(f"Sync full job started, job id is {jobs.job_id}")
            jobresult = jobs.wait_for_completion()
            self.log.debug("Sync full is compeleted, new inc job run before this ")
            self.log.debug(f"""
will delete all the following objects before restore:{new_ad_objs}""")
            self.ad_ins.cv_ugo_delete(new_ad_objs, ad_content)
            self.log.debug("list all objects after delete new objects")
            ad_objlists_sync = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug("compare result after delete the new objects")
            result, diff_check = cv_ad_objectscomapre(ad_objlists_sync,
                                                      None,
                                                      diff_objs_list=new_ad_objs,
                                                      judgement_value=False)
            if result:
                self.log.debug(f"all {len(diff_check)} are deleted after sync full")
            self.log.debug("run restore from sync full")
            jobr = self.subclient.restore_in_place(paths=["\\"])
            jobresult = jobr.wait_for_completion()
            self.log.debug(f'restore job {jobr.job_id} is completed')
            ad_objlists_sync_restore = self.ad_ins.cv_ad_objectlist(ad_content)
            self.log.debug("compare the result based after restore ")
            result, diff_check = cv_ad_objectscomapre(ad_objlists_sync_restore,
                                                      None,
                                                      diff_objs_list=new_ad_objs,
                                                      judgement_value=True)
            if result:
                self.log.debug(f"all {len(diff_check)} objects are restored")

            self.cleanup_objs = new_ad_objs

            if self.cleanup_objs:
                self.ad_ins.cv_ugo_delete(new_ad_objs, ad_content)
                self.log.debug("all new objects are cleaned")
            self.log.debug("run phase completed")
            self.teardown()

        except ADException as exp:
            self.status = constants.FAILED
            self.log.exception("there is exception happened, here is the detail %s" % \
                               exp.report)
        except Exception as exp:
            self.status = constants.FAILED
            self.log.exception("there is not AD exception happened, here is the detail %s" % \
                               exp)

    def teardown(self):
        """Teardown function of this test case"""
        self.log.debug("tear down phase completed")
