# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for performing Router Commcell related operations on Commcell
RouterCommcell:

    __init__()                              -- intializing the router commcell details

    get_service_commcell()                  -- Returns the service commcell object

    register_service_commcell()             -- Registers the Service Commcell to router commcell

    check_registration()                    -- checks if commcell is registered successfully or not

    unregister_service_commcell()           -- Unregisters the Service Commcell to router commcell

    _get_redirect_rules_on_router()         -- Gets the redirect rules from Router commcell

    _get_users_on_router()            -- Gets the users space from Router commcell

    create_service_db_connection()          -- creates a db connection to service database

    get_redirect_rules_on_servicedb()      -- gets the redirect rules from Service commcell database

    get_redirect_rules_on_service()         -- gets the redirect rules from Service commcell

    get_users_on_service()            -- gets the users space from Service commcell

    validate_redirect_rules()               -- validates the redirect rules

    validate_users_space()                  -- validates the users space

    get_redirect_service_commcells()        -- gets the valid service commcells to redirect for username given

    check_if_user_exists_on_service()       -- checks if the user exists on service commcell

    check_if_mail_exists_on_service()       -- checks if the user with mail id exists on service commcell

    validate_sync()                         -- validates sync operation after registration

    validate_user_user_group()              -- validates user to usergroup association

    validate_commcells_idp()                -- validates the commcells list registered for IdP

    get_commcell_object()                   -- gets the commcell object of the specified user

"""

from cvpysdk.commcell import Commcell
from AutomationUtils import database_helper, cvhelper
from AutomationUtils import logger
from AutomationUtils.machine import Machine
import xmltodict

class RouterCommcell(object):
    """Initializing the router commcell details. """

    def __init__(self, commcell):
        """Intializes the Router Commcell with commcell object. """

        self._commcell = commcell
        self.log = logger.get_log()
        self.csdb = database_helper.CommServDatabase(self._commcell)
        self._service_commcell = None
        self._service_commcell_db_connect = None

    def get_service_commcell(self, service_commcell_host_name, service_user_name, service_password):
        """Returns the service commcell object

        Args:

            service_commcell_host_name     (str)      --      service commcell host name

            service_user_name            (str)        --      service commcell user name

            service_password             (str)        --      service commcell password

        """
        self.log.info("Getting the service commcell object")
        try:

            self._service_commcell = Commcell(service_commcell_host_name,
                                              service_user_name,
                                              service_password)
            return self.service_commcell

        except Exception as excp:

            self.log.error('service commcell object creation failed : %s ',
                           str(excp))
            raise Exception('service commcell object creation failed : %s  ',
                            str(excp))

    @property
    def service_commcell(self):
        """returns the service commcell object"""
        if self._service_commcell is None:
            raise Exception("Service commcell is not initialized")

        return self._service_commcell

    def register_service_commcell(self, service_commcell_host_name, service_user_name, service_password,
                                  registered_for_idp=None):
        """Registers the Service Commcell to router/IdP commcell

        Args:

            service_commcell_host_name     (str)      --      service commcell host name

            service_user_name            (str)   --      service commcell user name

            service_password             (str)   --      service commcell password

            registered_for_idp           (bool)  --   True - if we want the commcell to be registered for Identity Provider
                                                      False - if we dont want the commcell to be registered for Identity Provider

        """
        self.log.info("Registering a Commcell")
        self._commcell.register_commcell(commcell_name=service_commcell_host_name,
                                         registered_for_routing=True,
                                         admin_username=service_user_name,
                                         admin_password=service_password,
                                         register_for_idp=registered_for_idp)

    def check_registration(self):
        """checks if commcell is registered successfully or not
        Args:
            commcell_name (str)   -- Name of the commcell

        Returns:
            boolean value

        """
        self.log.info("Check for registration")
        if not self._commcell.is_commcell_registered(self.service_commcell.commserv_name):
            raise Exception("Service Commcell is not registered successfully")

    def unregister_service_commcell(self):
        """Unregisters the Service Commcell to router commcell."""
        
        self.log.info("UnRegistering a Commcell")
        self._commcell.unregister_commcell(self.service_commcell.commserv_name)

    def _get_redirect_rules_on_router(self):
        """Gets the redirect rules from Router commcell

        Returns:

            list_of_rules_on_router     (list)  -- list of redirect rules on router for service commcell

            for example:    [['commvault.com'],['company1'],['company2']]

        """
        self.log.info("Getting the redirect rules of service commcell on router")
        query2 = "select stringVal from app_componentprop where componentType = 1047 and componentId = " \
                 "(select id from app_commcell where aliasName like '{0}')".format(self.service_commcell.commserv_name)
        self.csdb.execute(query2)
        return self.csdb.rows

    def _get_users_on_router(self):
        """Gets the users space from Router commcell

        Returns:

            list_of_users_space_on_router     (list)  -- list of users space on router for service commcell

            for example:    [['user1'],['user2'],['user3']]

        """
        self.log.info("Getting users of service commcell on router")
        query2 = "select login from UMUsersServiceCommcell where origCCId = " \
                 "(select id from app_commcell where aliasName like '{0}')".format(self.service_commcell.commserv_name)
        self.csdb.execute(query2)
        return self.csdb.rows

    def create_service_db_connection(self, machine_name):
        """
        creates a db connection to service database.

        Args:

        machine_name        (str)   -- name of the machine

        """
        service_machine_object = Machine(machine_name.lower(), self.service_commcell)
        encrypted_password = service_machine_object.get_registry_value(r"Database", "pAccess")
        sql_password = cvhelper.format_string(self.service_commcell, encrypted_password).split("_cv")[1]
        sql_instancename = "{0}\\commvault".format(machine_name.lower())
        self.log.info("Getting a db connectivity for service commcell")
        self._service_commcell_db_connect = database_helper.MSSQL(
            server=sql_instancename,
            user='sqladmin_cv',
            password=sql_password,
            database='commserv',
            as_dict=False
        )

    @property
    def service_commcell_db(self):
        """returns a db object for service commcell"""
        if self._service_commcell_db_connect is None:
            raise Exception("Service commcell db is not connected")

        return self._service_commcell_db_connect

    def get_redirect_rules_on_servicedb(self):
        """gets the redirect rules from Service commcell database.

        Returns:

            list_of_redirect_rules_service_db   (list)  -- list of redirect rules from service commcell

        """
        self.log.info("Getting the redirect rules from service commcell")
        xml_response = self.service_commcell_db.execute_stored_procedure(
            procedure_name='MCC_FetchRedirectsforRouterCommcell',
            parameters='')
        json_response = xmltodict.parse(xml_response.rows[0][0])
        list_of_rules = json_response['RedirectRules']['rules'] + json_response['RedirectRules']['domains']
        return list_of_rules

    def get_redirect_rules_on_service(self):
        """gets the redirect rules from Service commcell

        Returns:

            lis_of_redirect_rules_service   (list)  -- list of redirect rules from service commcell
        """
        return self.service_commcell.redirect_rules_of_service

    @property
    def get_users_on_service(self):
        """gets the users space from service commcell

        Returns:

            list_of_users_space_service     (list)  -- list of users space from service commcell
        """
        return self.service_commcell.users.service_commcell_users_space

    def validate_redirect_rules(self):
        """validates the redirect rules on router from service"""

        self.log.info("Getting the list of rules from service")
        redirect_rules_from_service = self.get_redirect_rules_on_service()
        self.log.info("Getting the list of rules of service on router")
        redirect_rules_on_router = self._get_redirect_rules_on_router()
        temp_list = [values[0] for values in redirect_rules_on_router]
        self.log.info("Comparing redirect rules")
        result_list_compare = set(redirect_rules_from_service) ^ set(temp_list)
        if result_list_compare:
            raise Exception("Validation Failed")

    def validate_users_space(self):
        """Validates the users space on router from service"""

        self.log.info("Getting the list of users from service")
        users_space_from_service = self.get_users_on_service.keys()
        self.log.info("Getting the list of users of service on router")
        users_space_from_router = self._get_users_on_router()
        temp_list = [values[0] for values in users_space_from_router]
        self.log.info("Comparing users")
        result_list_compare = set(users_space_from_service) ^ set(temp_list)
        if result_list_compare:
            raise Exception("Validation Failed")

    def get_redirect_service_commcells(self, user_name_or_mail_id):
        """gets the valid service commcells to redirect for username given

        Args:

            user_name_or_mail_id        (str)   -- login name or the mail id of the user

        Returns:

            list_of_service_commcells   (list)  -- list of service commcells to be redirected

        """
        self.log.info("Getting the list of service commcells eligible for redirecting")
        list_of_service_commcells = self._commcell.get_eligible_service_commcells(user_name_or_mail_id)
        if not self.service_commcell.commserv_name in list_of_service_commcells:
            raise Exception("No Service Commcell Found")
        else:
            return list_of_service_commcells

    def check_if_user_exists_on_service(self, user_name):
        """checks if user exists on service commcell

        Args:

            user_name       (str)   -- checks if user exists on service commcell

        Raises:

            when user doesnt exists on Service commcell

        """
        self.log.info("Checking if user exists on service")
        if not user_name in self.get_users_on_service.keys():
            raise Exception("user doesnt exist on service commcell")

    def check_if_mail_exists_on_service(self, mail_id):
        """checks if user exists on service commcell

        Args:

            mail_id       (str)   -- checks if user with mailid exists on service commcell

        Raises:

            when user user with mail doesnt exists on Service commcell

        """
        self.log.info("Checking if user with given mailid exists")
        list_of_mail = []
        for user_name in self.get_users_on_service:
            list_of_mail.append(self.get_users_on_service[user_name]['email'])
        if not mail_id in list_of_mail:
                raise Exception("user with mail id doesnt exist on service commcell")

    @property
    def user_groups_on_service_commcell(self):
        """gets the users groups space from Service commcell

        Returns:

            list_of_user_groups     (list)  -- list of user groups from service commcell
        """

        user_groups_on_service = self.service_commcell.user_groups.all_user_groups
        return user_groups_on_service.keys()

    def validate_sync(self):
        """checks if users, usergroups, companies ,AD and SAML apps are synced"""

        self.log.info("validating sync for users,usergroups,companies and AD")
        if not ((set(self.service_commcell.users.all_users.keys()).issubset(set(self._commcell.users.all_users.keys())))
                and (set(self.service_commcell.user_groups.all_user_groups.keys()).issubset(
                    set(self._commcell.user_groups.all_user_groups.keys())))
                and (set(self.service_commcell.organizations.all_organizations.keys()).issubset(
                    set(self._commcell.organizations.all_organizations.keys())))
                and (set(self.service_commcell.domains.all_domains.keys()).issubset(
                    self._commcell.domains.all_domains.keys()))
                and self.validate_user_user_group()):
            raise Exception("sync validation is not successfull")

    def validate_user_user_group(self):
        """Validates user to usergroup association of service on IdP """

        self.log.info("validates user to usergroup association of service on IdP")
        for ug in self.user_groups_on_service_commcell:
            if not (set(self.service_commcell.user_groups.get(ug).users).issubset(
                    set(self._commcell.user_groups.get(ug).users))):
                self.log.info("validation failed for user to ug association")
                return False
        return True

    def validate_commcells_idp(self, commcell_obj):
        """validates the list of accessible commcells to logged in user"""

        self.cc_object = commcell_obj
        commcells_for_idp = self.cc_object.commcells_for_user
        for self.service_commcell.commserv_name in commcells_for_idp:
            raise Exception("service commcell is not shown in IdP commcell list")

    def get_commcell_object(self, user_name=None, password=None, authtoken=None, service_commcell=None):
        """Returns the  commcell object

        Args:

            user_name            (str)        --      commcell user name

            password             (str)        --      commcell password

            authtoken            (str)        --    saml token generated on master commcell

            service_commcell      (boolean)   --   true- if login into service
                                                    false - if login

        """

        self.log.info("Getting the commcell object")
        try:
            if user_name and password:
                return Commcell(self._commcell.commserv_hostname, user_name, password)

            if authtoken and service_commcell:
                return Commcell(self.service_commcell.commserv_hostname,
                                authtoken=authtoken, is_service_commcell=service_commcell)

            if authtoken:
                return Commcell(self.service_commcell.commserv_hostname, authtoken=authtoken)

        except Exception as excp:

            self.log.error('service commcell object creation failed : %s ',
                           str(excp))
            raise Exception('service commcell object creation failed : %s  ',
                            str(excp))

