# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for performing User related operations on Commcell
UserProperties:
    __init__()                      --  initializing the user details

UserHelper:
    __init__()                      --  Initialize UserHelper object

    create_user()                   --  Creates new user on the CommCell

    delete_user()                   --  Deletes user passed

    _browser_setup()                --  Sets up browser constants

    gui_login()                     --  Performs auth token logins

    web_login()                     --  Performs admin console and admin console login

    gen_TFA_password()              --  Fetches one time pin for users

    enable_TFA()                    --  Enables Two Factor Authentication on the CommCell

    disable_TFA()                   --  Disables Two Factor Authentication on the CommCell

    modify_security_associations()  --  Validates security Associations on user and userggroup

    check_if_hidden_user_exists()    --  Checks if hidden user exits or not on the CommCell

"""
import random
from cvpysdk.commcell import Commcell
from AutomationUtils import database_helper
from Server.Security.securityhelper import RoleHelper
from AutomationUtils import logger, options_selector
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.adminconsole import AdminConsole
from Web.WebConsole.webconsole import WebConsole
from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase


class UserProperties(object):
    """initializing the user details """
    def __init__(self, name, cs_host, full_name=None, email=None, password=None, domain=None):
        """Initializes UserProperties object

        Args:
            name        (str)   --  name of the user

            cs_host     (str)   --  commcell host name

            full_name   (str)   --  full name for user

            email       (str)   --  user's email id

            password    (str)   --  password for user

            domain      (str)   --  domain name to which user is belongs to
        """
        self.username = name
        self.cs_host = cs_host
        self.full_name = full_name
        self.email = email
        self.password = password
        self.domain = domain


class UserHelper(object):
    """Helper class to perform User related operations"""

    def __init__(self, commcell, user=None):
        """Initializes UserHelper object

        Args:
            commcell    (obj)   --  Commcell object

            user        (obj)   --  user object
        """
        self._user = None
        if user:
            self._user = user
        self.log = logger.get_log()
        self.commcell_obj = commcell
        self.single_user = None
        self.usergroup = None
        self._csdb = database_helper.get_csdb()
        self.role_helper = RoleHelper(self.commcell_obj)
        self._utility = options_selector.OptionsSelector(self.commcell_obj)
        self.cl_obj = self.commcell_obj.commserv_client
        self.login_obj = None
        self.options_selector = options_selector.OptionsSelector(commcell)

    def create_user(self, user_name, email, full_name=None, domain=None, password=None,
                    local_usergroups=None, security_dict=None):
        """Adds a local/external user to this commcell

            Args:
                user_name                     (str)     --  name of the user to be
                                                            created

                full_name                     (str)     --  full name of the user to be
                                                            created

                email                         (str)     --  email of the user to be
                                                            created

                domain                        (str)     --  Needed in case you are adding
                                                            external user

                password                      (str)     --  password of the user to be
                                                            created
                                                            default: None

                local_usergroups              (str)     --  user can be member of
                                                            these user groups

                security_dict                 (str)     --  Role-Entity combination

                                e.g.: Security_dict={
                                'assoc1':
                                    {
                                        'entity_type':['entity_name'],
                                        'entity_type':['entity_name', 'entity_name'],
                                        'role': ['role1']
                                    },
                                'assoc2':
                                    {
                                        'mediaAgentName': ['networktestcs', 'standbycs'],
                                        'clientName': ['Linux1'],
                                        'role': ['New1']
                                        }
                                    }
         """
        self.log.info("Creating User %s", user_name)
        self._user = self.commcell_obj.users.add(user_name=user_name,
                                                 full_name=full_name,
                                                 domain=domain,
                                                 password=password,
                                                 email=email,
                                                 local_usergroups=local_usergroups,
                                                 entity_dictionary=security_dict)

        self.log.info("User [{0}] created successfully".format(user_name))
        return self._user

    def delete_user(self, user_name, new_user=None, new_group=None):
        """Deletes the user passed
        Args:
            user_name       (str)   -- object of user to be deleted

            new_user        (str)   -- user to whom ownership of entities will be transferred

            new_group       (str)   -- user group to whom ownership will be transferred

        """
        self.log.info("Deleting user: %s", user_name)

        self.commcell_obj.users.refresh()
        user = self.commcell_obj.users.has_user(user_name=user_name)
        if user:
            self.commcell_obj.users.delete(user_name=user_name, new_user=new_user, new_usergroup=new_group)
            self.log.info("Deleted user [{0}] successfully".format(user_name))
        else:
            self.log.info("Specified user is not present on the CommCell %s", user_name)

    def _browser_setup(self):
        """sets up browser constants

            returns:
                browser object
         """
        self.log.info("Initializing browser objects.")
        factory = BrowserFactory()
        browser = factory.create_browser_object()
        browser.open()
        return browser

    def web_login(self, user_name, password, web):
        """Performs admin console and admin console login
        Args:
            user_name   (str)   --  name of the user

            password    (str)   --  password for user

            web         (object)   --  object of class WebConstants

            Raises:
                Exception:
                    if user deletion is unsuccessful

        """

        web_urls = [web.adminconsole_url, web.webconsole_url]
        self.log.info("Login to webconsole and admin console")
        browser = self._browser_setup()
        for url in web_urls:
            try:
                self.log.info("Create the login object.")
                self.log.info("Login to %s.", url)
                try:
                    self.log.info("trying login with password: %s", password)
                    if url.lower().endswith("adminconsole"):
                        self.login_obj = AdminConsole(browser, self.commcell_obj.webconsole_hostname)
                    elif url.lower().endswith("webconsole"):
                        self.login_obj = WebConsole(browser, self.commcell_obj.webconsole_hostname)
                    else:
                        raise Exception('please pass valid interface name')
                    self.login_obj.login(user_name, password)
                    base = AdminConsoleBase(browser.driver)
                    base.check_error_message()
                    self.log.info("Successfully logged into %s.", url)
                except Exception as exp:
                    if 'Two-Factor Authentication is enabled' in str(exp):
                        new_password = self.gen_tfa_password(user_name=user_name, password=password)
                        self.log.info("trying login with new password: %s", new_password)
                        self.login_obj.login(user_name, new_password)
                        self.log.info("Successfully logged into %s.", url)
                    else:
                        raise Exception("Login failed due to %s", exp)
            except Exception as exp:
                self.log.error("Failed: %s", exp)
                Browser.close_silently(browser)
                raise Exception("Something went wrong. Please check logs for more details. error = {0}".format(exp))
            finally:
                getattr(self.login_obj, "logout_silently")(self.login_obj)
        Browser.close_silently(browser)

    def gui_login(self, cs_host, user_name, password):
        """performs authtoken logins
        Args:

            cs_host     (str)   --  CommCell host name

            user_name   (str)   --  user object to perform web login operation

            password    (str)   --  password for user

            Raises:
                Exception:
                    if user login is unsuccessful

        """
        try:
            Commcell(cs_host, user_name, password)
            self.log.info("login successful")
        except Exception as exp:
            if 'Two-Factor' in str(exp):
                self.log.info("Two-Factor Authentication is enabled on this CommServe")
                new_password = self.gen_tfa_password(user_name=user_name, password=password)
                self.log.info("trying gui login with new password %s", new_password)
                try:
                    Commcell(cs_host, user_name, new_password)
                    self.log.info("Two Factor Authentication login is successful")
                except Exception as exp:
                    raise Exception("login is not successful due to:", exp)
            else:
                raise Exception("login is not successful due to:", exp)

    def gen_tfa_password(self, user_name, password):
        """Fetches one time pin for users
        Args:
            user_name   (str)   --  user object to perform web login operation

            password    (str)   --  password for user

        Returns:
            New password(str)   --  returns new password (old password+OTP)

            Raises:
                Exception:
                    if user login is unsuccessful

        """
        self.log.info("Generating OTP for user: %s ", user_name)
        self.single_user = self.commcell_obj.users.get(user_name)
        otp = self.single_user.request_otp()
        new_password = password + otp
        return new_password

    def enable_tfa(self):
        """Enables Two Factor Authentication on the CommCell
        """
        self.log.info("Enbling Two-Factor Authentication on Commcell")

        self.cl_obj.add_additional_setting(category='CommServDB.GxGlobalParam',
                                           key_name='EnableTwoFactorAuthentication',
                                           data_type='BOOLEAN', value='1')
        self.log.info("*****Successfully Enabled Two-Factor Authentication*****")
        self.log.info("*****Failed Login attempt is required to generate One Time Pin*****")

    def disable_tfa(self):
        """Disables Two Factor Authentication on the CommCell
        """
        self.log.info("Dissabling Two-Factor Authentication on Commcell")

        self.cl_obj.add_additional_setting(category='CommServDB.GxGlobalParam',
                                           key_name='EnableTwoFactorAuthentication',
                                           data_type='BOOLEAN', value='0')
        self.log.info("*****Successfully Disabled Two-Factor Authentication*****")

    def modify_security_associations(self, entity_dict, user_name, request='UPDATE'):
        """Validates security Associations on user and userggroup
        Args:
            entity_dict         (Dict)  :   entity-role association dict

            request             (Str)   :   decides whether to UPDATE, DELETE or
                                            OVERWRITE user security association
        Raises:
                Exception:
                    if any of the association from passed entity_dict doesn't present on the user
                    or usergroup.

        """
        if request not in ['UPDATE', 'ADD', 'OVERWRITE', 'DELETE']:
            raise Exception("Invalid Request type is sent to the  function!!")
        else:
            self.user = self.commcell_obj.users.get(user_name)
            self.user.update_security_associations(entity_dictionary=entity_dict,
                                                   request_type=request)
            self.log.info("Secussful !!")

    def check_if_hidden_user_exists(self, user_name):
        """Checks if the hidden user exists in db or not
        Args:
            user_name           (Str)   :   name of the user

            request             (Str)   :   decides whether to UPDATE, DELETE or
                                            OVERWRITE user security association
        Returns:
            True : If user exists
            False : If user doesnt exists.
        """

        query_1 = "select 1 from UMUsers where login = '{0}'".format(user_name)
        self._csdb.execute(query_1)
        if (self._csdb.rows[0][0] == '1'):
            return True
        return False

    def get_user_clients(self):
        """Returns the list of clients for the user
        """
        clients = []
        user_id = self._user.user_id
        query = f"""CREATE TABLE #getIdaObjects 
                    (clientId INT, apptypeId INT, instanceID INT, backupsetId INT, subclientID INT,primary key(clientId,appTypeId,instanceId,backupsetId,subclientId))
                    EXEC sec_getIdaObjectsForUser {user_id}, 3, 0,0, '#getIdaObjects', 0, '2'
                    select name from app_client where id in (select clientId from #getIdaObjects);"""
        db_response = self.options_selector.update_commserve_db(query)
        for client in db_response.rows:
            clients.append(client['name'])

        return clients

    @staticmethod
    def password_generator(complexity_level=2, min_length=8):
        """
        generates password based on complexity level and min length

        Args:
            complexity_level    (int) -- complexity level
                supported values: 1 or 2 or 3

                1 - There are no requirements for passwords.
                2 - Passwords must have at least eight characters, one uppercase letter, one lowercase letter,
                    one number, and one special character.
                    Example: Wer1f*gd
                3 - Passwords must have at least eight characters, two uppercase letters, two lowercase letters,
                    two numbers, and two special characters.
                    Example: We1*G!9d

            min_length          (int) -- length of the password

        Returns:
            password        --  (str)

        Raises:
            Exception:
                if invalid values are passed
        """
        if complexity_level < 1 or complexity_level > 3 or min_length < 1:
            raise Exception("Please pass supported values")

        if complexity_level > 1 and min_length < 8:
            raise Exception("password length cannot be less than 8 for complexity level greater than 1")

        numbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9']

        lower = ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h',
                 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q',
                 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']

        upper = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
                 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q',
                 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z']

        special = ['@', '#', '$', '%', '=', ':', '?', '.', '/', '|', '~', '>', '*', '(', ')', '!']

        combined = numbers + lower + upper + special

        temp_list = None
        if complexity_level == 1:
            temp_list = random.choices(combined, k=min_length)

        if complexity_level in (2, 3):
            rand_nums = random.choices(numbers, k=complexity_level - 1)
            rand_lower = random.choices(lower, k=complexity_level - 1)
            rand_upper = random.choices(upper, k=complexity_level - 1)
            rand_special = random.choices(special, k=complexity_level - 1)

            temp = rand_nums + rand_lower + rand_upper + rand_special

            # shuffle once to avoid same patterns
            random.shuffle(temp)

            # picking random char from lower + upper + numbers only to avoid generating level 3 password rather than 2.
            if complexity_level == 2:
                combined = numbers + lower + upper

            for i in range(min_length - len(temp)):
                temp = temp + random.choices(combined, k=1)
                random.shuffle(temp)
            temp_list = temp

        final_password = ""
        for char in temp_list:
            final_password = final_password + char

        return final_password
