# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for performing User Group related operations on Commcell

UsergroupHelper:
    __init__()                      --  Initialize Usergroup Helper object

    delete_usergroup()              --  Deletes the usergroup passed

    create_usergroup()              --  Adds local/external user group on this commcell based
                                        domain parameter provided

    modify_security_associations()  --  Validates security Associations on user and usergroup
"""

import os
from AutomationUtils import database_helper
from AutomationUtils import logger, options_selector


class UsergroupHelper(object):
    """Helper class to perform User related operations"""

    def __init__(self, commcell, usergroup=None):
        """Initializes Usergroup Helper object

        Args:
            commcell    (obj)   --  Commcell object

            usergroup   (obj)   --  usergroup object
        """
        if usergroup:
            self._usergroup = usergroup
        self.log = logger.get_log()
        self.commcell_obj = commcell
        self._csdb = database_helper.get_csdb()
        self._utility = options_selector.OptionsSelector(self.commcell_obj)
        self.cl_obj = self.commcell_obj.commserv_client

    def delete_usergroup(self, group_name, new_user=None, new_group=None):
        """Deletes the usergroup passed
        Args:
            group_name       (str)   -- object of usergroup to be deleted

            new_user        (str)   -- user to whom ownership of entities will be transferred

            new_group       (str)   -- user group to whom ownership will be transferred

        """
        self.log.info("performing Delete Operation on user group: %s", group_name)

        user = self.commcell_obj.user_groups.has_user_group(user_group_name=group_name)
        if user:
            self.commcell_obj.user_groups.delete(user_group=group_name, new_user=new_user,
                                                 new_usergroup=new_group)
            self.log.info("usergroup deletion is Successful")
        else:
            self.log.info("Specified usergroup is not present on the CommCell %s", group_name)

    def create_usergroup(self, group_name, domain=None, users=None, entity_dict=None,
                         external_groups=None, local_groups=None):
        """Adds local/external user group on this commcell based domain parameter provided

            Args:
                group_name          (str)   --  name of the user group

                domain              (str)   --  name of the domain to which user group
                                                belongs to

                users               (list)  --  list which contains users who will be
                                                members of this group

                entity_dict         (dict)  --  combination of entity_type, entity
                                                names and role
                e.g.: entity_dict={
                                'assoc1':
                                    {
                                        'entity_type':['entity_name'],
                                        'entity_type':['entity_name', 'entity_name'],
                                        'role': ['role1']
                                    },
                                'assoc2':
                                    {
                                        'mediaAgentName': ['networktestcs', 'standbycs'],
                                        'clientName': ['Linux1'],
                                        'role': ['New1']
                                        }
                                    }
                entity_type         --      key for the entity present in dictionary
                                            on which user will have access
                entity_name         --      Value of the key
                role                --      key for role name you specify
                e.g:   e.g.: {"clientName":"Linux1"}
                Entity Types are:   clientName, mediaAgentName, libraryName, userName,
                                    userGroupName, storagePolicyName, clientGroupName,
                                    schedulePolicyName, locationName, providerDomainName,
                                    alertName, workflowName, policyName, roleName

                entity_name = "Linux1", "ClientMachine1"

                external_groups     (list)  --  list of domain user group which could
                                                be added as members to this group

                local_groups        (list)  --  list of commcell usergroup which could
                                                be added as members to this group
        """
        args = [group_name, domain, users, entity_dict, external_groups, local_groups]
        self.log.info("Creating usergroup with name = {0}, domain = {1}, users_list = {2}, entity_dict = {3},"
                      "external_group = {4}, local_usergroups = {5}".format(*args))
        user_group_object = self.commcell_obj.user_groups.add(*args)
        self.log.info("UserGroup {0} creation is Successful!!".format(group_name))
        return user_group_object

    def modify_security_associations(self, entity_dict, group_name, request='UPDATE'):
        """Validates security Associations on user and userggroup
        Args:
            entity_dict         (Dict)  :   entity-role association dict

            group_name          (Str)   :   Name of the usergroup

            request             (Str)   :   decides whether to UPDATE, DELETE or
                                            OVERWRITE user security association

        Raises:
                Exception:
                    if request type is not valid

        """

        if request not in ['UPDATE', 'ADD', 'OVERWRITE', 'DELETE']:
            raise Exception("Invalid Request type is sent to the  function!!")
        else:
            self._usergroup = self.commcell_obj.user_groups.get(user_group_name=group_name)
            self._usergroup.update_security_associations(entity_dictionary=entity_dict,
                                                         request_type=request)
            self.log.info("""Sucessfully modified security association for entity [{0}], group [{1}], request: [{2}]"""
                          .format(entity_dict, group_name, request))

    def get_usergroup_clients(self):
        """Returns the list of clients for the user group
        """
        clients = []
        usergroup_id = self._usergroup.user_group_id
        # Now we read in the stored procedure query for getting clients of a user group
        # usergroup_stored_proc.txt must be present in SmartClientGroups Folder
        script_dir = os.path.dirname(__file__)
        rel_path = '../SmartClientGroups/usergroup_stored_proc.txt'
        abs_file_path = os.path.join(script_dir, rel_path)
        filename = os.path.abspath(os.path.realpath(abs_file_path))
        file = open(filename, 'r')
        stored_proc = file.read()
        # Execute create stored_procedure query
        try:
            self._utility.update_commserve_db(stored_proc)
        except Exception as excp:
            self.log.info(excp)
        file.close()
        # Execute query to get usergroup clients
        query = f"""CREATE TABLE #getIdaObjects 
                    (clientId INT, apptypeId INT, instanceID INT, backupsetId INT, subclientID INT,primary key(clientId,appTypeId,instanceId,backupsetId,subclientId))
                    EXEC sec_getIdaObjectsForUser {usergroup_id}, 3, 0,0, '#getIdaObjects', 0, '2'
                    select name from app_client where id in (select clientId from #getIdaObjects);"""
        db_response = self._utility.update_commserve_db(query)
        drop_stored_proc = """DROP PROCEDURE sec_getIdaObjectsForUserGroup;"""
        # Execute drop stored procedure query
        try:
            self._utility.update_commserve_db(drop_stored_proc)
        except Exception as drop_proc_excp:
            self.log.info(drop_proc_excp)
        for client in db_response.rows:
            clients.append(client['name'])

        return clients
