# -*- coding: utf-8 -*-
# pylint: disable=W1202

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for performing user's login related validations.

LoginValidator is the only class defined in this file

LoginValidator:

    __init__(test_object)                   --  initialize instance of the LoginValidator class

    validate()                              --  entry point for validation

    cleanup()                               --  deletes entities which got created part of test case run.

    create_local_user_entity()              --  creates user/user group

    _users_to_be_validated()                --  adds users to global list of users to be verified.

    retry_webconsole_user_login             --  retries webconsole login for 'n' number of times

    _update_security_associations_for_tenants() --  updates security associations for tenant users.

    prerequisites()                         --  Registers Directories with commcell/company and
                                                imports External groups/users

    validate_user_login()                   --  Validates user's login with username/email from adminconsole,
                                                webconsole, GUI

    validate_login_attempt_limit_and_account_lock() --  validates attempt limit and account lock feature

    validate_domain_less_login()            --  validates domain less user login feature

    validate_password_complexity_and_history_features   --  validates password history, complexity and
                                                            expired password features
"""

import json
import random
import time
from datetime import datetime

from cvpysdk.commcell import Commcell
from cvpysdk.domains import Domains
from cvpysdk.organization import Organizations
from cvpysdk.client import Client
from cvpysdk.security.user import User

from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils import logger, config
from AutomationUtils.machine import Machine

from Server.Security.securityhelper import SecurityHelper
from Server.Security.usergrouphelper import UsergroupHelper
from Server.Security.userhelper import UserHelper
from Server.Security.userconstants import WebConstants


class LoginValidator:
    """DRValidator helper class to perform DR validations"""

    def __init__(self, test_object):
        """
        Initialize instance of the LoginValidator class

            Args:
                test_object     (object)            --      instance of testcase.
        """
        self.log = logger.get_log()
        self.test_object = test_object
        self._commcell = test_object.commcell
        self._csdb = test_object.csdb
        self._tcinputs = test_object.tcinputs

        self.usergrouphelper = UsergroupHelper(self._commcell)
        self.userhelper = UserHelper(self._commcell)
        self.organizations = Organizations(self._commcell)
        self.securityhelper = SecurityHelper(self._commcell)
        self.domains = Domains(self._commcell)
        self.config_json = config.get_config()
        self.utility = OptionsSelector(self._commcell)
        self.client_obj = Client(self._commcell, client_name=self._commcell.commserv_name)
        if self.config_json.Schedule.cs_machine_uname:
            self.client_machine = Machine(self._commcell.commserv_client.client_hostname,
                                          username=self.config_json.Schedule.cs_machine_uname,
                                          password=self.config_json.Schedule.cs_machine_password)
        else:
            self.client_machine = Machine(self.client_obj)
        self.users_login_details = []
        self.company_object = None
        self._commcell.refresh()

    def validate(self, feature, **inputs_required_for_feature):
        """
        Entry point for feature validations

            Args:
                feature        (list)      --      list of features to be validated.

                inputs_required_for_feature      (kwargs)   --  required parameters for the corresponding
                                                                feature validate method.

        Returns :
            None
        """
        try:
            self.prerequisites()
            if self.company_object:
                self._update_security_associations_for_tenants()
            if feature in ('user_login', 'login_attempt_limit_and_account_lock',
                           'domain_less_login', 'password_complexity_and_history_features'):
                getattr(self, 'validate_' + feature)(**inputs_required_for_feature)
            else:
                raise Exception('please pass the validate feature name')
        finally:
            self.cleanup()

    def cleanup(self):
        """
        Deletes entities which got created from test run.

        Returns:
             None
        """
        # cleanup order:
        # first, delete company, it will delete company associated entities like directories, groups etc
        # second, commcell level directories, it will delete external users and groups as well.
        # third, commcell level local entities like user groups and users..
        self.log.info('Cleaning up entities...')
        if self.company_object:
            # deleting company, will delete company associated entities like directories, groups etc
            self.log.info('deactivating and deleting company {0}'.format(self.company_object.name))
            self.organizations.delete(self.company_object.name)
        self.domains.refresh()
        for entity in self.users_login_details:
            if entity.get('level') == 'commcell':
                if entity.get('domain'):
                    if self.domains.has_domain(entity.get('domain')):
                        self.log.info('Deleting domain {0}'.format(entity.get('domain')))
                        self.domains.delete(entity.get('domain'))
                    else:
                        self.log.info("{0} domain doesn't exist on commcell".format(entity.get('domain')))
                else:
                    if entity.get('usergroupname'):
                        self.usergrouphelper.delete_usergroup(entity.get('usergroupname'),
                                                              new_user=self.config_json.ADMIN_USERNAME)
                    if entity.get('username'):
                        self.userhelper.delete_user(entity.get('username'), new_user=self.config_json.ADMIN_USERNAME)

    def create_local_user_entity(self, entity_inputs, tenant_admin_commcell_obj=None):
        """
        creates entities user/user group at commcell/company level
        imports external users/groups at commcell/company level
        Args:
             entity_inputs      (dict)      --      dict consisting of user/user group properties

             sample dict1:
                {
                "usergroup": {},
                "user": {}
                }
                with above dict passed,creates usergroup with random(name, permissions)
                and user with random(name, permissions, email, fullname), password from config.json
                and associates this user with above created user group.

             sample dict2;
                {
                "usergroup":
                    {
                        "group_name": "",
                        "domain": "",
                        "users": [],
                        "entity_dict": {},
                        "external_groups": [],
                        "local_groups": []
                    },
                "user":
                    {
                        "user_name": "",
                        "email": "",
                        "full_name": "",
                        "domain": "",
                        "password": "",
                        "local_usergroups": "",
                        "security_dict": {}
                    }
                }
                for more details about individual property of user/usergroup,
                please refer create_user() method from userhelper.py and create_usergroup from usergrouphelper.py

             tenant_admin_commcell_obj      (object)    --

            Adds created user details to global list self.users_login_details

         Returns:
               dict
        """
        timestamp = datetime.strftime(datetime.now(), '%H%M%S')
        user_group_object = None
        prefix = (tenant_admin_commcell_obj.commcell_username.split("\\")[0] + "\\") if tenant_admin_commcell_obj else ""
        final_dict = {}
        if "usergroup" in entity_inputs:
            user_group_props = entity_inputs.get("usergroup")
            create_user_group_props = {
                "usergroup_name": ("" if user_group_props.get(
                    'domain') else prefix) + (user_group_props.get("group_name", "usergroup_{0}".format(timestamp))),
                "domain": user_group_props.get('domain'),
                "users_list": user_group_props.get('users'),
                "entity_dictionary": user_group_props.get(
                    'entity_dict', self.securityhelper.gen_random_entity_types_dict(2)),
                "external_usergroup": user_group_props.get('external_groups'),
                "local_usergroup": user_group_props.get('local_groups')
            }
            self.log.info("Creating usergroup with props {0}".format(create_user_group_props))
            if tenant_admin_commcell_obj:
                user_group_object = tenant_admin_commcell_obj.user_groups.add(**create_user_group_props)
                tenant_admin_commcell_obj.user_groups.refresh()
            else:
                user_group_object = self._commcell.user_groups.add(**create_user_group_props)
                self._commcell.user_groups.refresh()
        if "user" in entity_inputs:
            user_props = dict(entity_inputs.get('user'))
            username = ("" if user_props.get("domain")
                        else prefix) + (user_props.get('user_name', "testuser_{0}".format(timestamp)))
            password = '' if user_props.get('domain') else user_props.get(
                "password", UserHelper.password_generator(complexity_level=3, min_length=12))
            create_user_props = {
                "user_name":  username,
                "email": user_props.get("email", "{0}@dummy.com".format(username.replace("\\", "_"))),
                "full_name": user_props.get("full_name", username.replace("\\", "_")),
                "domain": user_props.get("domain"),
                "password": password,
                "local_usergroups": user_props.get("local_usergroups",
                                                   [user_group_object.name] if user_group_object else None),
                "entity_dictionary": user_props.get(
                    "security_dict", self.securityhelper.gen_random_entity_types_dict(2))
            }
            self.log.info("Creating user with props {0}".format(create_user_props))
            if tenant_admin_commcell_obj:
                tenant_admin_commcell_obj.users.add(**create_user_props)
                tenant_admin_commcell_obj.users.refresh()
            else:
                self._commcell.users.add(**create_user_props)
                self._commcell.users.refresh()
            final_dict['username'] = create_user_props.get('user_name')
            final_dict['password'] = user_props.get('password') if user_props.get('domain') else password
            final_dict['email'] = create_user_props.get('email')
            final_dict['domain'] = create_user_props.get('domain')
        final_dict['level'] = "commcell" if not tenant_admin_commcell_obj else "company"
        final_dict['usergroupname'] = user_group_object.name if user_group_object else None
        return final_dict

    def _users_to_be_validated(self, username, password, email=None, level='commcell',
                               usergroupname=None, domain=None):
        """
        adds users to global list, which can be used by validate methods for various feature validation.

        Args:
            username        (str)       --  username

            password        (str)       --  password

            email           (str)       --  email

            level           (str)       --  level at which user is available/created
                Values:
                "commcell"
                "company"

            usergroupname   (str)       --  name of the user group, given user is part of..

            domain          (str)       --  short name of the domain

        Returns:
            None

        """
        user_template = {
            "level": level,
            "usergroupname": usergroupname,
            "username": username,
            "password": password,
            "email": email,
            "domain": domain
        }
        self.users_login_details.append(user_template)

    def retry_webconsole_user_login(self, username, password, retry_count=3):
        """
        Retries webconsole/adminconsole login for 'n' number of times

        Args:
             username   (str)       --      name of the user

             password   (str)       --      password of the user

             retry_count    (int)   --      numbers of times user login needs to be attempted.

        Returns:
            bool        --  if login succeeds for at least once.

        Raises:
            Exception:
                if user login fails 'n' times.
        """
        for count in range(retry_count):
            self.log.info("retry count = {0} for user = {1}".format(count, username))
            try:
                self.userhelper.web_login(username, password,
                                          web=WebConstants(self._commcell.commserv_hostname))
            except Exception as exp:
                if "Invalid Username or Password" in str(exp):
                    raise Exception(exp)
                if count == retry_count-1:
                    raise Exception(exp)
                self.log.info("retry count {0}, error={1}".format(count, exp))
                continue
            return True

    def _update_security_associations_for_tenants(self):
        """
        associates commcell level permissions for tenant users
        """
        self._commcell.users.refresh()
        for user in self.users_login_details:
            if user.get('level').lower() == 'company':
                username = ((user.get('domain') + "\\" + user.get('username'))
                            if user.get('domain') else user.get('username'))
                user_obj = self._commcell.users.get(username)
                associations = self.securityhelper.gen_random_entity_types_dict(2)
                associations["view_permission"] = {"providerDomainName": [self.company_object.name],
                                                   "role": ['view']
                                                   }
                user_obj.update_security_associations(
                    entity_dictionary=associations,
                    request_type='UPDATE'
                )
                self.log.info("{0} security associations got associated to user {1}".format(
                    user_obj.user_security_associations,
                    user_obj.user_name
                ))

    def prerequisites(self):
        """
        Creates company based on input json passed.
        Registers entities LDAP's(Active directory, LDAP Server) based on the input JSON passed.
        imports entities(External User/External User Group) based on the input JSON passed.

        Returns:
            None.
        """
        for level in self._tcinputs.keys():
            tenant_admin_commell_obj = None
            if level.lower() in ("commcell", "company"):
                commcell_obj = self._commcell
                ldap_entity_details = json.loads(self._tcinputs.get(level)).get('LDAPs', None)
                local_entity_details = json.loads(self._tcinputs.get(level)).get('Local', None)
                timestamp = datetime.strftime(datetime.now(), '%H%M%S')
                if level.lower() == 'company':
                    # create company/organization
                    company_name = 'company{0}'.format(timestamp)
                    self.log.info("creating company {0}".format(company_name))
                    self.company_object = self.organizations.add(
                        name=company_name, email=self.config_json.email.email_id,
                        contact_name=company_name,
                        company_alias=company_name)
                    self.log.info("company {0} got created successfully".format(company_name))
                    tenant_admin_name = company_name + "\\" + self.config_json.email.email_id.split('@')[0]
                    user_obj = User(self._commcell, user_name=tenant_admin_name)
                    new_password = UserHelper.password_generator(complexity_level=3, min_length=12)
                    user_obj.update_user_password(new_password=new_password,
                                                  logged_in_user_password=self.config_json.ADMIN_PASSWORD)
                    self._users_to_be_validated(username=tenant_admin_name,
                                                password=new_password,
                                                email=self.config_json.email.email_id,
                                                level='company')
                    if user_obj.is_tfa_enabled:
                        # call tfa api such that pin will get populated in db
                        self.userhelper.gui_login(self._commcell.commserv_hostname, tenant_admin_name, new_password)
                        new_password = new_password + str(user_obj.request_otp())
                    tenant_admin_commell_obj = Commcell(self._commcell.commserv_hostname,
                                                        commcell_username=tenant_admin_name,
                                                        commcell_password=new_password)
                    commcell_obj = tenant_admin_commell_obj

                # create local user/usergroups
                if local_entity_details:
                    for entity in local_entity_details:
                        entity_details = self.create_local_user_entity(
                            entity_inputs=entity,
                            tenant_admin_commcell_obj=tenant_admin_commell_obj if level.lower() == 'company' else None)
                        self._users_to_be_validated(**entity_details)

                # Register given directories with commcell/company
                if ldap_entity_details:
                    for server in ldap_entity_details:
                        # server_details = ldap_entity_details.get(server)
                        server_details = self.config_json.Security.LDAPs._asdict().get(server)._asdict()
                        self.log.info("Trying to register domain {0}"
                                      " at {1} level".format(server_details.get('NETBIOSName'), level))
                        commcell_obj.domains.add(domain_name=server_details.get('DomainName'),
                                                 netbios_name=server_details.get('NETBIOSName'),
                                                 user_name=server_details.get('UserName'),
                                                 password=server_details.get('Password'),
                                                 company_id=int(self.company_object.organization_id)
                                                 if level.lower == 'company' else 0,
                                                 ad_proxy_list=server_details.get('ViaProxy'),
                                                 type_of_server=server.replace("_", " "),
                                                 group_filter=server_details.get('group_filter'),
                                                 user_filter=server_details.get('user_filter'),
                                                 unique_identifier=server_details.get('unique_identifier'),
                                                 base_dn=server_details.get('base_dn'))
                        self.log.info("Domain {0} got added successfully"
                                      " at level {1}".format(server_details.get('NETBIOSName'), level))

                        # import external groups at (commcell\company) level
                        if server_details.get("UserGroupsToImport"):
                            for group in server_details.get("UserGroupsToImport"):
                                group = group._asdict()
                                assocs = (self.securityhelper.gen_random_entity_types_dict(2)
                                          if level.lower() == "commcell" else "")
                                self.log.info("Trying to add external group {0}"
                                              " at level {1}".format(group.get('externalgroup'), level))
                                self.create_local_user_entity(
                                    entity_inputs={
                                        "usergroup": {
                                            "group_name": group.get('externalgroup'),
                                            "domain": server_details.get('NETBIOSName'),
                                            "entity_dict": group.get('permissions', assocs)
                                        }
                                    },
                                    tenant_admin_commcell_obj=tenant_admin_commell_obj if level.lower() == 'company'
                                    else None)
                                self.log.info("External group {0} at level {1} is"
                                              " imported successfully".format(group.get('externalgroup'), level))

                        # import external users at (commcell\company) level
                        if server_details.get("UsersToImport"):
                            for user in server_details.get("UsersToImport"):
                                user = user._asdict()
                                assocs = (self.securityhelper.gen_random_entity_types_dict(2)
                                          if level.lower() == "commcell" else "")
                                self.log.info("Trying to add external user {0}"
                                              " at level {1}".format(user.get('UserName'), level))
                                entity_details = self.create_local_user_entity(entity_inputs={
                                    "user": {
                                        "user_name": user.get('UserName'),
                                        "email": user.get('email', ''),
                                        "domain": server_details.get('NETBIOSName'),
                                        "security_dict": user.get('permissions', assocs),
                                        "password": user.get('Password')
                                    }
                                }, tenant_admin_commcell_obj=tenant_admin_commell_obj if level.lower() == 'company'
                                    else None)
                                self._users_to_be_validated(**entity_details)
                                self.log.info("External user {0} is imported"
                                              " successfully".format(user.get('UserName')))
            else:
                raise Exception('Please pass valid inputs...')

    def validate_user_login(self, login_with="username", additional_setting=None):
        """
        validates user's login with username/email from adminconsole, webconsole, GUI.

        Args:
             login_with         (str)   --      login with username or email
                values:
                "username"
                "email"

            additional_setting  (dict)  --  required details for adding key 'nAllowUserWithoutRightToLogin'
                eg:
                {
                    "category": "CommServDB.GxGlobalParam",
                    "key_name": "nAllowUserWithoutRightToLogin",
                    "data_type": "INTEGER",
                    "value": "0"
                }

        Returns:
            None
        Raises:
            Exception:
                when user login fails
        """
        self.log.info("Validating users login with {0} and additional setting {1}".format(
            login_with, additional_setting))

        failed_login_counter = 0
        if additional_setting:
            # add additional setting if provided
            self.log.info("Adding Additional setting {0}".format(additional_setting))
            self._commcell.add_additional_setting(category=additional_setting["category"],
                                                  key_name=additional_setting["key_name"],
                                                  data_type=additional_setting["data_type"],
                                                  value=additional_setting["value"])
        try:
            for user in self.users_login_details:
                redundant_emails_found = False
                try:
                    self.log.info("Login attempt for user with username : {0}".format(user))
                    if login_with.lower() == 'username':
                        username = ((user.get('domain') + "\\" + user.get('username'))
                                    if user.get('domain') else user.get('username'))
                    elif login_with.lower() == 'email':
                        username = user.get('email')
                        record = self.utility.exec_commserv_query(query="select count(email) as count from umusers"
                                                                        " where email='{0}' and enabled=1"
                                                                        " group by email".format(username))
                        redundant_emails_found = int(record[0][0]) > 1
                    else:
                        raise Exception('please pass validate param')
                    if user.get('level').lower() == 'company' and not user.get('domain') and additional_setting:
                        failed_login_counter = failed_login_counter + 1
                        continue
                    if additional_setting:
                        user_obj = self._commcell.users.get(((user.get('domain') + "\\" + user.get('username'))
                                                             if user.get('domain') else user.get('username')))
                        user_obj.update_security_associations(entity_dictionary={}, request_type="OVERWRITE")

                    # perform gui login and web login's
                    self.userhelper.gui_login(self._commcell.commserv_hostname, username, user.get('password'))
                    self.retry_webconsole_user_login(username=username, password=user.get('password'))
                except Exception as exp:
                    # if login fails for one user, catch the exception and continue login validations for other users
                    if not redundant_emails_found:
                        failed_login_counter = failed_login_counter + 1
                    self.log.info(exp)
            if additional_setting and not len(self.users_login_details) == failed_login_counter:
                raise Exception("users with no permissions should not be able to login,"
                                " when nAllowUserWithoutRightLogin is set with value 0,"
                                " login users counter = {0}".format(len(
                                    self.users_login_details) - failed_login_counter))
            if failed_login_counter and not additional_setting:
                raise Exception('Login with {1} failed for {0} users,'
                                ' please check logs for more info'.format(failed_login_counter, login_with))
        finally:
            if additional_setting:
                self.log.info("Deleting Additional setting {0}".format(additional_setting))
                self._commcell.delete_additional_setting(category=additional_setting["category"],
                                                         key_name=additional_setting["key_name"])

    def validate_login_attempt_limit_and_account_lock(self, user_attempt_limit=3, account_lock_duration=180):
        """
        Validates featues FailedLoginAttemptLimit and AccountLockDuration for various user's

        Args:
            user_attempt_limit      (int)       --      number of failed attempts before user account gets locked

            account_lock_duration   (int)       --      user account lock duration in *secs*

        Returns:
            None

        Raises:
            Exception:
                if feature validation fails
        """
        attempt_limit = user_attempt_limit
        lock_duration = account_lock_duration
        failed_counter = 0
        self.log.info("Validating user's login attempt with limit = {0}"
                      " and account lock feature with duration = {1}".format(attempt_limit, lock_duration))
        self.create_local_user_entity(entity_inputs={"user": {}})
        self.log.info("Adding keys FailedLoginAttemptLimit and AccountLockDuration")
        # Add keys FailedLoginAttemptLimit and AccountLockDuration
        self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                              key_name="FailedLoginAttemptLimit",
                                              data_type="INTEGER",
                                              value=str(attempt_limit))
        self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                              key_name="AccountLockDuration",
                                              data_type="INTEGER",
                                              value=str(lock_duration))
        self.log.info("Keys got added successfully")
        try:
            self.log.info("Services restart is required after adding advanced"
                          " additional settings, restarting services..")
            # restart services
            self.client_obj.restart_services()
            self.log.info("Performing IISRESET")
            self.client_machine.execute_command('iisreset')
            time.sleep(300)
            self.log.info("Services got restarted successfully")
            for user in self.users_login_details:
                try:
                    username = ((user.get('domain') + "\\" + user.get('username'))
                                if user.get('domain') else user.get('username'))
                    self.log.info("Validating feature for user {0}".format(username))

                    # login with invalid creds for attempt_limit times from GUI or web interface
                    for attempt in range(attempt_limit):
                        try:
                            interface = random.sample(["gui", "web"], 1)
                            self.log.info("Attempt = {2} user = {0} login with invalid "
                                          "creds from = {1}".format(username, interface, attempt+1))
                            if interface[0] == "web":
                                self.userhelper.web_login(username, "*******",
                                                          web=WebConstants(self._commcell.commserv_hostname))
                            elif interface[0] == "gui":
                                self.userhelper.gui_login(self._commcell.commserv_hostname, username, "*******")
                        except Exception as excp:
                            if attempt+1 == attempt_limit and ("locked" not in str(excp)
                                                               and "incorrect" not in str(excp)):
                                raise Exception("'locked' keyword is not present on error masg,"
                                                " error = {0}".format(excp))
                            self.log.info(excp)

                    self.log.info("Failed login limit is successfully validated for user = {0}".format(username))
                    # verify that account got locked (i.e) try login with valid creds from interfaces
                    # (GUI, WEBCONSOLE, ADMINCONSOLE)
                    for interface in range(2):
                        logged_in = 1
                        try:
                            if interface == 1:
                                self.userhelper.gui_login(self._commcell.commserv_hostname,
                                                          username, user.get('password'))
                            elif interface == 0:
                                self.userhelper.gui_login(self._commcell.commserv_hostname,
                                                          username, user.get('password'))
                                self.userhelper.web_login(username, user.get("password"),
                                                          web=WebConstants(self._commcell.commserv_hostname))
                        except Exception as excp:
                            if 'locked' not in str(excp) and "incorrect" not in str(excp):
                                raise Exception("user account must be locked , excp = {0}".format(excp))
                            logged_in = 0
                            self.log.info(excp)
                        if logged_in:
                            raise Exception('login must fail with remaining duration to get account unlock')
                    self.log.info("Waiting for some time so that account get's "
                                  "unlocked secs = {0}".format(lock_duration))
                    time.sleep(lock_duration+120)
                    # attempting login with valid creds after account lock duration
                    self.retry_webconsole_user_login(username=username, password=user.get('password'))
                    self.userhelper.gui_login(self._commcell.commserv_hostname, username, user.get('password'))
                except Exception as excp:
                    failed_counter = failed_counter + 1
                    self.log.info(excp)
            if failed_counter:
                raise Exception("FailedLoginAttemptLimit and AccountLockDuration"
                                " is failed for {0} users".format(failed_counter))
            self.log.info("FailedLoginAttemptLimit and AccountLockDuration are validated successfully")
        finally:
            self.log.info("Deleting additional keys...")
            self._commcell.delete_additional_setting(category="CommServDB.GxGlobalParam",
                                                     key_name="FailedLoginAttemptLimit")
            self._commcell.delete_additional_setting(category="CommServDB.GxGlobalParam",
                                                     key_name="AccountLockDuration")
            self.log.info("removing addtional settings requires service restart, restarting services...")
            self.client_obj.restart_services()
            self.log.info("Performing IISRESET")
            self.client_machine.execute_command('iisreset')
            time.sleep(300)
            self.log.info("Services restarted successfully")

    def validate_domain_less_login(self, default_ad_user_domain, default_cs_domain='local'):
        """
        Validates domainless login feature for various user's

        Args:
            default_ad_user_domain      (str)       --      name of the domain, which will be set as default domain

            default_cs_domain            (str)      --      name of the default cs domain
                default:
                "local"

        Returns:
            None

        Raises:
            Exception:
                if feature validations fails
        """
        self.log.info("Adding Additional settings DefaultADUserDomain with value = {0}".format(default_ad_user_domain))
        self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                              key_name="DefaultADUserDomain",
                                              data_type="STRING",
                                              value=default_ad_user_domain)
        if default_cs_domain.lower() != "local":
            self.log.info("Adding additional setting DefaultCSDomain with value = {0}".format(default_cs_domain))
            self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                                  key_name="DefaultCSDomain",
                                                  data_type="STRING",
                                                  value=default_cs_domain)
        self.log.info("Keys got added successfully.")
        try:
            self.log.info("Performing IISRESET")
            self.client_machine.execute_command('iisreset')
            time.sleep(300)
            # below we are trying to login without any prefix for all user's[commcelll user, other AD users]
            # bascially negative cases are handled
            unallowed_user = 0
            # create commcell local user
            self.create_local_user_entity(entity_inputs={"user": {}})
            for user in self.users_login_details:
                try:
                    self.log.info("Performing login without prefix for user = {0}".format(user.get('username')))
                    self.userhelper.gui_login(self._commcell.commserv_hostname,
                                              user.get('username'), user.get('password'))
                    self.retry_webconsole_user_login(username=user.get('username'), password=user.get('password'))
                    if user.get('domain') and user.get('domain').lower() != default_ad_user_domain.lower():
                        unallowed_user = unallowed_user + 1
                except Exception as excp:
                    if user.get('domain') and user.get('domain').lower() == default_ad_user_domain.lower():
                        raise Exception("User {0} must be able to login,"
                                        " something went wrong excp = {1}".format(user.get('username'), excp))
                    self.log.info("user {0} login failed, Expected"
                                  " behavior excp = {1}".format(user.get('username'), excp))
                if unallowed_user:
                    raise Exception('Login must fail for username {0}, as'
                                    ' key is set DefaultADUserDomain'.format(user.get('username')))
            # below we are trying to login with prefix for all user's(i.e)
            # for commcell local user default prefix is local\\
            failed_user_login = 0
            for user in self.users_login_details:
                if user.get('domain'):
                    username = user.get('domain') + "\\" + user.get('username')
                elif user.get('level').lower() == "commcell":
                    username = default_cs_domain + "\\" + user.get('username')
                else:
                    username = user.get('username')
                self.log.info("Performing user login with prefix for user {0}".format(username))
                try:
                    self.userhelper.gui_login(self._commcell.commserv_hostname, username, user.get('password'))
                    self.retry_webconsole_user_login(username=username, password=user.get('password'))
                except Exception as excp:
                    failed_user_login = failed_user_login + 1
                    self.log.info("Login failed for user {0}, excp={1}".format(username, excp))
            if failed_user_login:
                raise Exception("login failed for {0} users, please check logs".format(failed_user_login))
            self.log.info("Domainless login feature is validated successfully")
        finally:
            self.log.info("Deleting additional keys...")
            self._commcell.delete_additional_setting(category="CommServDB.GxGlobalParam",
                                                     key_name="DefaultADUserDomain")
            if default_cs_domain.lower() != "local":
                self._commcell.delete_additional_setting(category="CommServDB.GxGlobalParam",
                                                         key_name="DefaultCSDomain")
                self.log.info("Performing IISRESET")
            self.client_machine.execute_command('iisreset')
            time.sleep(120)

    def validate_password_complexity_and_history_features(self, password_history, complexity_level=2,
                                                          age_password_days=1):
        """
        validates features password history, complexity level and age user password.

        Args:
            password_history    (dict)  --  required values for enabling password history feature
                eg:- {
                        "level" : "commcell",
                        "value" : 2
                    }

            complexity_level    (int)   --  level of complex password to be verified.

            age_password_days   (int)   --  number of days for password expiry.

        Returns:
            None

        Raises:
            Exception:
                if feature validations fails
        """
        self.log.info("Adding Additional settings passwordComplexityLevel with"
                      " value = {0}".format(complexity_level))
        self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                              key_name="passwordComplexityLevel",
                                              data_type="INTEGER",
                                              value=str(complexity_level))
        self.log.info("Adding Additional settings bMaintainPasswordHistory with"
                      " value = {0}".format(password_history))
        self._commcell.add_additional_setting(category="CommServDB.Console",
                                              key_name="bMaintainPasswordHistory",
                                              data_type="BOOLEAN",
                                              value='true')
        if password_history.get('level') == "commcell":
            self._commcell.add_additional_setting(category="CommServDB.GxGlobalParam",
                                                  key_name="Password History Count",
                                                  data_type="INTEGER",
                                                  value=str(password_history.get('value')))
        self.log.info("Keys got added successfully.")
        try:
            # try to create a local user with less complex password(i.e) using
            # password_generator generate less complex password and verify exception
            self.log.info("Performing IISRESET")
            self.client_machine.execute_command('iisreset')
            time.sleep(300)

            self.log.info("try to create a local user with less complex password..")
            try:
                self.create_local_user_entity(entity_inputs={
                    "user": {"password": self.userhelper.password_generator(complexity_level-1)}})
                raise Exception("user creation with less complex should fail")
            except Exception as excp:
                if "User password doesn't follow minimum complexity requirements" not in str(excp):
                    raise Exception(excp)
                self.log.info(excp)

            # now try creating user with complex password
            self.log.info("try creating user with password of complexity level = {0}".format(complexity_level))
            self.create_local_user_entity(entity_inputs={
                "user": {"password": self.userhelper.password_generator(complexity_level, min_length=12)}})

            for user in self.users_login_details:
                user_obj = User(self._commcell, user_name=user.get('username'))

                # expire user password
                user_obj.age_password_days = age_password_days

                # create old timestamp
                old_timestamp = int(datetime.timestamp(datetime.now())) - ((age_password_days + 2) * 24 * 60 * 60)
                self.log.info("Forcing user password expiry with age password = {0} and old timetsamp = {1}".format(
                    age_password_days, old_timestamp
                ))
                self.utility.update_commserve_db(
                    "update umusers set datePasswordSet={0} where login like '%{1}%'".format(
                        old_timestamp, user.get('username'))
                )

                try:
                    self.userhelper.gui_login(self._commcell.commserv_hostname,
                                              user.get('username'),
                                              user.get('password'))
                except Exception as exp:
                    if "Password Expired" not in str(exp):
                        raise Exception(exp)
                    self.log.info(exp)
                self.log.info("user Password expiry scenario is validated successfully")
                # try password change with less complex password and verify exception
                self.log.info("try password change with less complex password..")
                try:
                    user_obj.update_user_password(new_password=self.userhelper.password_generator(complexity_level-1),
                                                  logged_in_user_password=self.config_json.ADMIN_PASSWORD)
                    raise Exception("password change operation with less complex should fail")
                except Exception as exp:
                    if "User password doesn't follow minimum complexity requirements" not in str(exp):
                        raise Exception(exp)
                    self.log.info(exp)

                # try password change with existing password and verify exception
                self.log.info("password change with existing password..")
                try:
                    user_obj.update_user_password(new_password=user.get('password'),
                                                  logged_in_user_password=self.config_json.ADMIN_PASSWORD)
                    raise Exception("password change operation with existing password should fail")
                except Exception as exp:
                    if "Please use a different password" not in str(exp):
                        raise Exception(exp)
                    self.log.info(exp)

                # try password change with complex password
                new_password = self.userhelper.password_generator(complexity_level, min_length=12)
                user_obj.update_user_password(new_password=new_password,
                                              logged_in_user_password=self.config_json.ADMIN_PASSWORD)
                self.retry_webconsole_user_login(username=user.get('username'), password=new_password)
                self.userhelper.gui_login(self._commcell.commserv_hostname, user.get('username'), new_password)
                self.log.info("Password history = {0}, Password complexity = {1}"
                              " and age password = {2} are validated"
                              " successfully".format(password_history, complexity_level, age_password_days))
        finally:
            self.log.info("Deleting additional keys...")
            self._commcell.delete_additional_setting(category="CommServDB.GxGlobalParam",
                                                     key_name="passwordComplexityLevel")
            self._commcell.delete_additional_setting(category="CommServDB.Console",
                                                     key_name="bMaintainPasswordHistory")
