# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing REST API operations

RESTAPIHelper is the only class defined in this file

RESTAPIHelper: Helper class to perform REST API operations

RESTAPIHelper:

 __init__()                      --  initializes REST API helper object

run_newman()                     --  executes the collection file using newman

clean_up()                       -- deletes temporary files generated during newman execution

"""

import os
import json
from base64 import b64encode
from shutil import copyfile
from Server.RestAPI import restapiconstants as RC
from AutomationUtils import logger


class RESTAPIHelper(object):
    """Helper class to perform REST API operations"""

    def __init__(self):
        """Initializes RESTAPIhelper object """
        self.log = logger.get_log()

    def __repr__(self):
        """Representation string for the instance of the RESTAPIHelper class."""
        return "RESTAPIHelper class instance"

    def run_newman(self, collection_json, tc_input, delay=2, **kwargs):
        """Executes the given collection file using newman

        Args:
            collection_json  (dict)   --  dictionary value with test case id as key
                                        and collection file name created using postman as value

                                        Example:
                                        {'tc_id': self.id, 'c_name': collection_json}

            tc_input         (dict)  --      test case input dictionary

                                        Example:
                                        self.tcinputs

            delay            (int)   -- delay in execution between two APIs. Value is in ms.

                                        Example:
                                        5000

            **kwargs                 -- Provides additional run options for newman run

            Raises:
                Exception:
                    if newman execution fails
        """
        run_options =  dict(kwargs.items())

        if 'custom_endpoint' in run_options:
            webserver_url = run_options.get('custom_endpoint')
            tc_input['ServerURL'] = webserver_url
        else:
            webserver_url = 'http://' + tc_input["webserver"] + ':81/SearchSvc/CVWebService.svc'
            tc_input['ServerURL'] = webserver_url

        tc_input.pop('webserver', None)

        base64pwd = b64encode(tc_input["password"].encode()).decode()

        tc_input['Password'] = base64pwd
        tc_input.pop('password', None)

        tc_input['UserName'] = tc_input["username"]
        tc_input.pop('username', None)

        env_file = RC.ENVIRONMENT_FILE

        collection_file = os.path.join(RC.COLLECTION_FILE, collection_json['c_name'])

        temp_env_file = RC.TEMP_ENVIRONMENT_FILE

        if not os.path.exists(RC.NEWMAN_LOG_LOCATION):
            os.makedirs(RC.NEWMAN_LOG_LOCATION)

        newman_log_file_name = os.path.join(RC.NEWMAN_LOG_LOCATION,
                                            'newman_'+collection_json['tc_id']+'.log')

        self.log.info("Creating copy of environment file to location {0}".format(RC.COLLECTION_FILE))

        copyfile(env_file, temp_env_file)
        missing_keys= []

        with open(temp_env_file, 'r+') as f:
            json_data = json.load(f)

            self.log.info("Replacing values in copied environment file with test case answers")

            for tc_key in tc_input:
                key_present = True
                for value in json_data['values']:
                    if tc_key in value['key']:
                        value['value'] = tc_input[tc_key]
                    else:
                        if tc_key not in missing_keys:
                            missing_keys.append(tc_key)

                        else:
                            self.log.info("Duplicate key , ignoring it as it is already part of missing keys ")

            if missing_keys:
                for each_key in missing_keys:
                    temp_data = {'enabled': True, 'key': each_key, 'type': "text", 'value': tc_input[each_key]}
                    json_data['values'].append(temp_data)

            json_data['values'] = [i for n, i in enumerate(json_data['values']) if i not in json_data['values'][n + 1:]]
            f.seek(0)
            f.write(json.dumps(json_data))
            f.truncate()
            self.log.info("File copied and values replaced....starting newman execution...")

        cmd = 'newman run "' + collection_file + '" --environment "' + \
              temp_env_file + '" >> "' + newman_log_file_name + '" --delay-request "' + str(delay) + '"'

        if('run_flags' in run_options):
            for flag in run_options.get('run_flags'):
                cmd += " --{0}".format(flag)

        self.log.info("Command being executed is: %s ", cmd)

        res = os.system(cmd)

        self.log.info("Generating Newman logs at the location: %s", RC.NEWMAN_LOG_LOCATION)

        self.log.debug("Value returned from newman execution: %s", res)

        if res == 0:
            self.log.info("Newman tests executed successfully")

        else:
            raise Exception("Failure in execution.Please check Newman logs located at: {0}".
                            format(newman_log_file_name))

    def clean_up(self):
        """clean-up function to delete temporary files"""

        self.log.info("Starting clean-up phase")
        self.log.info("Deleting temporary environment file")
        os.remove(RC.TEMP_ENVIRONMENT_FILE)
        self.log.info("Temporary environment file deleted")
