# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""" Helper file for validating Operation window

OpValidate:
        __init__()                              -- Initialises OpValidate object

        validate()                              -- Validates an operation rule

        validate_full_data_management()         -- Validates Full data Management feature

        validate_non_full_data_management()     -- Validates Non full data Management feature

        validate_data_recovery()                -- Validates Data Recovery feature

        validate_synthetic_full()               -- Validates Synthetic full feature

        validate_aux_copy()                     -- Validates Aux copy feature

        validate_dr_backup()                    -- Validates Disaster Recovery feature

"""

import time
from AutomationUtils import logger


class OpValidate:
    """Helper class to validate operation window"""

    def __init__(self, testcase, op_rule):
        """
        Initialize instance of the opvalidate class.
        Args:
            testcase     (obj)     --Testcase object

            op_rule (obj)          --Instance of operationWindowDetails class
        """
        self.op_rule = op_rule
        self.testcase = testcase
        self.log = logger.get_log()

    def _init_job_object_verify(self):
        """
        Verifies whether the job object honors the operation window or not
        Requires a testcase initialised subclient on which operations should be performed.
        Returns:
            Flag      (int)     --- 1 if job object honors operation window else 0
            subclient (object)  --- testcase initialised subclient
        Raises:
            Exception :
                if subclient is not initialised
                                or
                if operation rule entity level is client group and clientgroup object is not initialised in testcase
        """
        self.log.info("Initialising subclient object")
        flag = 0
        if self.testcase.subclient is None:
            self.log.info("No subclient object is present in the test case %s", self.testcase.id)
            raise Exception("No subclient object is present in the test case {0}".format(self.testcase.id))
        testcase_dict = {"commserv": self.testcase.commcell.commcell_id,
                         "client": self.testcase.client.client_id,
                         "agent": self.testcase.agent.agent_id,
                         "instance": self.testcase.instance.instance_id,
                         "backupset": self.testcase.backupset.backupset_id,
                         "subclient": self.testcase.subclient.subclient_id}

        op_rule_dict = {"commserv": self.op_rule.commcell_id,
                        "client": self.op_rule.client_id,
                        "agent": self.op_rule.agent_id,
                        "instance": self.op_rule.instance_id,
                        "backupset": self.op_rule.backupset_id,
                        "subclient": self.op_rule.subclient_id}
        self.log.info("Checking whether the subclient honors the operation window")
        op_rule_entity_level = self.op_rule.entity_level.lower()
        if op_rule_entity_level == "clientgroup":
            self.log.info("Checking whether the subclient belongs to the client group")
            if self.testcase.clientgroup is None:
                self.log.info("No clientgroup object is present in the test case %s", self.testcase.id)
                raise Exception("No clientgroup object is present in the test case {0}".format(self.testcase.id))
            if self.testcase.client.name in self.testcase.clientgroup.associated_clients \
                    and self.testcase.clientgroup.clientgroup_id == self.op_rule.clientgroup_id:
                self.log.info("Subclient does honor the operation window and belongs to clientgroup")
                flag = 1
            else:
                self.log.info("Subclient does not honor the operation window")
        elif op_rule_dict[op_rule_entity_level] == testcase_dict[op_rule_entity_level]:
            self.log.info("Subclient does honor the operation window")
            flag = 1
        else:
            self.log.info("Subclient does not honor the operation window")
        self.log.info("Initialised subclient object")
        return self.testcase.subclient, flag

    def validate(self, features=None):
        """if the features are None..It'll validate
            the initialised op_rule at it's entity level for all the features
        if features are provided then it'll validate for that
            set of features for the initialised op_rule at it's entity_level
        Args:
            features (List)  --- List of features that need to be verified
        """
        if features is None:
            features = self.op_rule.operations

        for feature in features:
            if feature in self.op_rule.operations:
                getattr(self, "validate_"+feature.lower())()

    def validate_full_data_management(self):
        """Validates Full Data Management feature"""
        self.log.info("Validating Full Data Management feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "FULL_DATA_MANAGEMENT" in operations:
            operations.remove("FULL_DATA_MANAGEMENT")
            self.log.info("Full Data Management feature do honor operation window")
        else:
            self.log.info("Full Data Management feature does not honor operation window")
        job_object, flag = self._init_job_object_verify()
        self.log.info("Starting a full backup job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.backup("full")
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.backup("full")
            self.log.info("Successfully started a full backup job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Full Data Management feature".center(50, "*"))

    def validate_non_full_data_management(self):
        """Validates Non Full Data Management feature"""
        self.log.info("Validating Non Full Data Management feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "NON_FULL_DATA_MANAGEMENT" in operations:
            operations.remove("NON_FULL_DATA_MANAGEMENT")
            self.log.info("Non Full Data Management feature do honor operation window")
        else:
            self.log.info("Non Full Data Management feature does not honor operation window")
        job_object, flag = self._init_job_object_verify()
        self.log.info("Starting an incremental backup job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.backup("incremental")
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.backup("incremental")
            self.log.info("Successfully started an incremental backup "
                          "job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Non Full Data Management feature".center(50, "*"))

    def validate_data_recovery(self):
        """Validates Data Recovery feature"""
        self.log.info("Validating Data Recovery feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "DATA_RECOVERY" in operations:
            operations.remove("DATA_RECOVERY")
            self.log.info("Data Recovery feature do honor operation window")
        else:
            self.log.info("Data Recovery feature does not honor operation window")
        job_object, flag = self._init_job_object_verify()
        if self.op_rule.entity_level == "backupset":
            flag = 0
        if self.op_rule.entity_level == "subclient":
            flag = 0
        self.log.info("Starting an Restore in place job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.restore_in_place(job_object.content)
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.restore_in_place(job_object.content)
            self.log.info("Successfully started an restore in place "
                          "job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Data Recovery feature".center(50, "*"))

    def validate_synthetic_full(self):
        """Validates Synthetic full feature feature"""
        self.log.info("Validating Synthetic Full feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "SYNTHETIC_FULL" in operations:
            operations.remove("SYNTHETIC_FULL")
            self.log.info("Synthetic Full  feature do honor operation window")
        else:
            self.log.info("Synthetic Full feature does not honor operation window")
        job_object, flag = self._init_job_object_verify()
        if self.op_rule.entity_level.lower() == "commserv":
            flag = 1
        else:
            flag = 0
        self.log.info("Starting an Synthetic full backup job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.backup("Synthetic_full")
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.backup("Synthetic_full")
            self.log.info("Successfully started a Synthetic full backup "
                          "job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Synthetic Full feature".center(50, "*"))

    def validate_aux_copy(self):
        """Validates Aux copy feature feature"""
        from cvpysdk.policies.storage_policies import StoragePolicies
        self.log.info("Validating Aux Copy feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "AUX_COPY" in operations:
            operations.remove("AUX_COPY")
            self.log.info("Aux Copy feature do honor operation window")
        else:
            self.log.info("Aux Copy feature does not honor operation window")
        job_object, flag = self._init_job_object_verify()
        job_object = StoragePolicies(self.testcase.commcell).get(job_object.storage_policy)
        self.log.info("Starting a aux copy job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.run_aux_copy()
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.run_aux_copy()
            self.log.info("Successfully started an aux copy job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Aux Copy feature".center(50, "*"))

    def validate_dr_backup(self):
        """Validates Disaster Recovery feature feature"""
        self.log.info("Validating Disaster Recovery feature".center(50, "*"))
        operations = list(self.op_rule.operations)
        if "DR_BACKUP" in operations:
            operations.remove("DR_BACKUP")
            self.log.info("Disaster Recovery feature do honor operation window")
        else:
            self.log.info("Disaster Recovery feature does not honor operation window")
        job_object = self.testcase.commcell.disasterrecovery
        if self.op_rule.entity_level.lower() == "commserv":
            flag = 1
        else:
            flag = 0
        self.log.info("Starting a Disaster Recovery Backup job")
        if self.op_rule.do_not_submit_job:
            try:
                job_object.disaster_recovery_backup()
            except Exception:
                self.log.info('Job is not submitted, expected behaviour')
        else:
            job = job_object.disaster_recovery_backup()
            self.log.info("Successfully started Disaster Recovery "
                          "Backup job with job id %s", job.job_id)
            self._job_verify(flag, job, operations)
        self.log.info("Validated Disaster Recovery feature".center(50, "*"))

    def _job_verify(self, flag, job, operations):
        """
        Verifying whether job is going according to the rules of operation window
        Args:
            flag (int)          --flag is used to know whether the Enitiy created in
                                testcase is same as our entity object

            job  (object)       --job object to check status and waiting for the job to complete

            operations (List)   --List of operations after removing the feature that needs to be validated
        """
        # operations == self.op_rule.operations for individual validations
        if flag == 0 or operations == self.op_rule.operations:
            self.log.info("%s should be Running", job.job_type)
            if job.status == "Queued":
                self.log.error("Failed to run %s with error: %s", job.job_type, job.delay_reason)
                raise Exception(
                    "Failed to run {0} with error: {1}".format(job.job_type, job.delay_reason)
                )
            self.log.info("Success:%s is running", job.job_type)
            self.log.info("Waiting for the %s job to complete", job.job_type)
            job.wait_for_completion(timeout=300)
            self.log.info("Completed the %s job", job.job_type)
        else:
            self.log.info("%s should be queued", job.job_type)
            if not job.status == "Queued":
                self.log.error("Failed to Queue %s with error: %s", job.job_type, job.delay_reason)
                raise Exception(
                    "Failed to Queue {0} with error: {1}".format(job.job_type, job.delay_reason)
                )
            self.log.info("Success:%s is queued", job.job_type)
            old_operations = list(self.op_rule.operations)
            self.log.info("Modifying the operation window to check if the job will be resumed")
            self.op_rule.operations = operations
            self.log.info("successfully modified the operation window")
            time.sleep(60)
            self.log.info("%s should be resumed", job.job_type)
            if job.status == "Queued":
                self.log.error("Failed to run %s with error: %s", job.job_type, job.delay_reason)
                raise Exception(
                    "Failed to run {0} with error: {1}".format(job.job_type, job.delay_reason)
                )
            self.log.info("Success : %s is resumed", job.job_type)
            self.log.info("Waiting for the %s job to complete", job.job_type)
            job.wait_for_completion(timeout=300)
            self.log.info("Completed the %s job", job.job_type)
            self.log.info("Modifying the operation window to it's initial state")
            self.op_rule.operations = old_operations
            self.log.info("Successfully modified the operation window")
