# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing multicommcell operations

MultiCommcellHelper is the only class defined in this file

MultiCommcellHelper
===================

    __init__                        --  initializes multicommcellhelper object

    validate_client_listing()       --  Validates the clients listed when
    logged into commcell using username-password and when logged in using SAML token

    validate_incorrect_authtoken()  --  Validates whether exception is raised when
    an incorrect authtoken is passed for operations

    cleanup_certificates()          -- deletes certificates created by test cases

    saml_config()                   -- performs initial configuration for multicommcell
    operations

    token_validity_check()          -- Validates that the commcell login fails if
    token is used after expiry

    tampered_validity_check()       -- Validates that the commcell login fails if token's
    time is tampered

    job_validate()                  -- validates the jobs run by multicommcell user
    on SP commcell

"""
import time
import base64
import re

from AutomationUtils import logger
from AutomationUtils import options_selector
from AutomationUtils.idautils import CommonUtils
from cvpysdk.commcell import Commcell
from cvpysdk import identity_management


class MultiCommcellHelper(object):
    """Helper class to perform multicommcell operations"""

    def __init__(self, inputs):
        """Initializes MultiCommcellHelper object and gets the commserv database
           object if not specified

            Args:
                inputs    (dict)    --  inputs passed to initialise the object
        """
        self.log = logger.get_log()
        self.sp_commcell = Commcell(
            inputs["SPCommserver"],
            inputs["SPadminUser"],
            inputs["SPadminUserPwd"]
        )
        self.idp_commcell = Commcell(
            inputs["IDPCommserver"],
            inputs["IDPadminUser"],
            inputs["IDPadminUserPwd"]
        )
        self._idp_options_selector = options_selector.OptionsSelector(self.idp_commcell)
        self.identity_app = None
        self.service_app = None

    def validate_client_listing(self, regular_commcell, saml_commcell):
        """Validates the clients listed when logged into commcell using
           username-password and when logged in using SAML token

            Args:
                regular_commcell    (obj)   -   Commcell object wehn logged in
                                                using username & password

                saml_commcell       (obj)   -   Commcell object wehn logged in
                                                using authtoken

            Returns:
                bool    -   True if validated. False otherwise
        """
        self.log.info("\tValidating the listing of clients")
        return regular_commcell.clients.all_clients == saml_commcell.clients.all_clients

    def validate_incorrect_authtoken(self, sp_commcell, idp_commcell):
        """Validates whether exception is raised when an incorrect authtoken
           is passed for operations

            Args:
                sp_commcell         (obj)   -   Commcell object of service provider

                idp_commcell        (obj)   -   Commcell object of ID provider

            Returns:
                bool    -   True if validated. False otherwise
        """
        idp_token = idp_commcell.get_saml_token()
        self.log.info("\tValidating invalid authtoken failure")
        try:
            idp_token = '{0}xyzabc'.format(idp_token)
            sp_user_commcell = Commcell(
                sp_commcell.commserv_hostname,
                authtoken=idp_token
            )
            if sp_user_commcell:
                return False
        except Exception as exp:
            self.log.info('Login with tampered authtoken successfully prevented - {0}'.format(
                str(exp))
            )
            return True

    def cleanup_certificates(self):
        """Helper method to delete the certificates created by the TC

        Returns:
            bool    -   True if deletion successful. False otherwise.
        """
        sp_apps = identity_management.IdentityManagementApps(self.sp_commcell)
        idp_apps = identity_management.IdentityManagementApps(self.idp_commcell)

        try:
            if not self.service_app:
                self.service_app = sp_apps.get_commcell_identity_apps
            if not self.identity_app:
                self.identity_app = idp_apps.get_local_identity_app

            if isinstance(self.service_app, list):
                for app in self.service_app:
                    sp_apps.delete_identity_app(app.app_name)
            else:
                sp_apps.delete_identity_app(self.service_app.app_name)

            idp_apps.delete_identity_app(self.identity_app.app_name)
        except Exception as exp:
            self.log.error('Cleanup failed due to - {0}'.format(str(exp)))

    def saml_config(self, app_display_name, user_dict=None):
        """This method performs the initial configuration on SP and IDP commcells,
           checks for multicommcell login
            Args:
                app_display_name    (str)   -   display name for the commcell app

                user_dict    (dict)  -   dict containing SP & IDP users details
                                         {
                                             'SPUser': {
                                                 'userName': 'user1_name',
                                                 'password': 'user1_password'
                                             },
                                             'IDPUser': {
                                                 'userName': 'user2_name',
                                                 'password': 'user2_password'
                                             }
                                         }

            Returns:
                tuple   -   Tuple containing service provider's commcell
                            object and ID provider's commcell object

        """
        idp_username = user_dict['IDPUser']['userName']
        sp_username = user_dict['SPUser']['userName']

        if not self.idp_commcell.users.has_user(idp_username):
            self.log.info("\tAttempting to create IDP user on {0}".format(
                self.idp_commcell.commserv_name
                )
            )
            idpuser_obj = self.idp_commcell.users.add(
                idp_username,
                "IDP User",
                "test@commvault.com",
                password=user_dict['IDPUser']['password'],
                local_usergroups=['master']
            )

        else:
            idpuser_obj = self.idp_commcell.users.get(idp_username)

        if not self.sp_commcell.users.has_user(sp_username):
            self.log.info("\tAttempting to create SP user on {0}".format(
                self.sp_commcell.commserv_name
                )
            )
            spuser_obj = self.sp_commcell.users.add(
                sp_username,
                "SP User",
                "test@commvault.com",
                password=user_dict['IDPUser']['password'],
                local_usergroups=['master']
            )
        else:
            spuser_obj = self.sp_commcell.users.get(sp_username)

        self.log.info("\n\tUNREGISTERING THE APPLICATION AT IDP IF IT WAS ALREADY REGISTERED")
        idp_apps_object = identity_management.IdentityManagementApps(self.idp_commcell)
        idp_app = idp_apps_object.get_local_identity_app
        if idp_app:
            idp_apps_object.delete_identity_app(idp_app.app_name)

        self.log.info("\n\tCREATING CERTIFICATE AT IDP")
        self.identity_app = idp_apps_object.configure_local_identity_app(
            [idpuser_obj.user_name]
        )

        if self.identity_app:
            self.log.info("IDP Certificate created successfully")
            local_identity_props = self.identity_app.get_app_props()

        self.log.info("\n\tUNREGISTERING THE APPLICATION AT SP IF IT WAS ALREADY REGISTERED")
        sp_apps_object = identity_management.IdentityManagementApps(self.sp_commcell)
        commcell_app_list = sp_apps_object.get_commcell_identity_apps
        if commcell_app_list:
            for app in commcell_app_list:
                sp_apps_object.delete_identity_app(app.app_name)

        self.log.info("\n\tREGISTER APPLICATION AT SP")
        self.service_app = sp_apps_object.configure_commcell_app(
            local_identity_props,
            self.idp_commcell.commserv_guid,
            app_display_name,
            user_assoc_list=[spuser_obj.user_name],
            user_mappings={
                idpuser_obj.user_name: spuser_obj.user_name
            }
        )
        if self.service_app:
            self.log.info("SP app created successfully")

        self.log.info("\n\tMULTICOMMCELL LOGIN TO SP")
        idp_user_commcell = Commcell(
            self.idp_commcell.commserv_hostname,
            idpuser_obj.user_name,
            user_dict['IDPUser']['password']
        )

        if idp_user_commcell:
            saml_token = idp_user_commcell.get_saml_token()

        if saml_token:
            sp_user_commcell = Commcell(
                self.sp_commcell.commserv_hostname,
                authtoken=saml_token
            )

        return (idp_user_commcell, sp_user_commcell)

    def token_validity_check(self, idp_commcell, sp_commcell, validity_mins=2):
        """Validates that the commcell login fails if token is used
           after expiry

            Args:
                idp_commcell    (obj)   -   identity provider commcell

                sp_commcell     (obj)   -   service provider commcell

            Returns:
                bool    -   True if validated. False otherwise
        """
        self.log.info("Requesting for a {0} minute valid token".format(validity_mins))
        idp_token = idp_commcell.get_saml_token(validity=validity_mins)

        timeout = time.time() + 60 * validity_mins
        expiry_failure = None
        try:
            while True:
                if time.time() > timeout:
                    self.log.info(
                        "Attempting login after {0} minute timeout".format(
                            validity_mins
                        )
                    )
                    Commcell(
                        sp_commcell.commserv_hostname,
                        authtoken=idp_token
                    )
                    expiry_failure = True
                    raise Exception(
                        'Login succeeded even after token expiry'
                    )
                else:
                    self.log.info(
                        "Timeout of {0} minutes not reached. Attempting login".format(
                            validity_mins
                        )
                    )
                    Commcell(
                        sp_commcell.commserv_hostname,
                        authtoken=idp_token
                    )
                    self.log.info("Login successful")
                    time.sleep(30)
        except Exception as exp:
            if expiry_failure:
                self.log.error(
                    "Login succeeded after token expiry - {0}".format(exp)
                )
            else:
                if exp.exception_id is '106':
                    self.log.info("Token validity passed - {0}".format(exp))
                else:
                    self.log.error("General exception - {0}".format(exp))

    def tampered_validity_check(self, idp_commcell, sp_commcell, validity_mins=2):
        """Validates that the commcell login fails if token's time is tampered

            Args:
                idp_commcell    (obj)   -   identity provider commcell

                sp_commcell     (obj)   -   service provider commcell

                validity_mins   (obj)   -   time duration in minutes for the token validity
        """
        self.log.info("Requesting for a {0} minute valid token".format(validity_mins))
        idp_token = idp_commcell.get_saml_token(validity=validity_mins)
        decoded_token = base64.b64decode(idp_token).decode('utf-8')[3:]
        tampered_token = re.sub(
            '(?<=NotOnOrAfter=").*Z',
            time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime(time.time())),
            decoded_token
        )
        encoded_token = "SAML {0}".format(
            base64.b64encode(tampered_token.encode('utf-8')).decode('utf-8')
        )
        try:
            Commcell(sp_commcell.commserv_hostname, authtoken=encoded_token)
            self.log.error("Login succeeded with tampered token. Please check webserver.log")
        except Exception as exp:
            self.log.info("Login with tampered token prevented with error - {0}".format(
                exp
            ))

    def job_validate(self, job_type, sp_commcell, client_name=None):
        """Validates the jobs run by multicommcell user on SP commcell

            Args:
                job_type    (str)   -   type of the job to be run, input in upper case

                sp_commcell (obj)   -   service provider commcell where job is to be run

                client_name (str)   -   client that is to be backed up
        """
        utils_obj = CommonUtils(sp_commcell)
        subclient_obj = utils_obj.get_subclient(client_name)

        job_obj = utils_obj.subclient_backup(
            subclient_obj,
            backup_type=job_type
        )
        if job_obj.summary['status'] == 'Completed':
            self.log.info('Job operation successful')
        else:
            raise Exception('Job triggerd by SP user has not completed')
