# -*- coding: utf-8 -*-
# pylint: disable=W1202

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for performing job management settings validations.

JobManagementValidator is the only class defined in this file

JobManagementValidator:
    __init__(test_object)                                --          initialize instance of the JobManagementValidator
                                                                     class.

    _entities_setup()                                    --          creates disklibrary and storage policy
                                                                     entities.

    _create_pre_scan_script()                            --          creates pre scan script on the client

    create_subclient()                                   --          creates required number of subclients

    validate_job_preemption()                            --          validates job preemption feature.

    _job_via_schedule()                                  --          triggers a job via schedule.

    _job_via_schedule_policy()                           --          triggers a job via schedule policy.

    validate()                                           --          entry point for validations

    validate_job_max_restarts()                          --          validates job max restarts feature.

    validate_job_total_running_time_pending_scenario()   --          validates total running time scenario 1
                                                                     (pending -> failed)

    validate_job_total_running_time_killed_scenario()    --          validates total running time scenario 2
                                                                    (kill job after total time expires)

    validate_high_watermark_level()                      --          validates high water mark level feature

    validate_queue_jobs_if_conflicting_job_active()      --          validates queue jobs if conflicting job
                                                                     active feature.

    validate_queue_job_if_activity_is_disable()          --          validates queue job if activity is disable

    validate_queue_schedule_jobs()                       --          validates queue schedule jobs feature

"""

import time
from datetime import datetime

from cvpysdk.client import Client
from cvpysdk.agent import Agent
from cvpysdk.subclient import Subclient, Subclients
from cvpysdk.commcell import Commcell
from cvpysdk.policies import schedule_policies
from cvpysdk import schedules
from cvpysdk import activitycontrol
from AutomationUtils.constants import FAILED
from AutomationUtils.constants import WINDOWS_TMP_DIR
from AutomationUtils.constants import UNIX_TMP_DIR
from AutomationUtils import logger, options_selector
from AutomationUtils.idautils import CommonUtils
from AutomationUtils.machine import Machine
from Server.ActivityControl import activitycontrolhelper
from Server.Scheduler import schedulerhelper
from Server.JobManager.jobmanagement_helper import JobManagementHelper


class JobManagementValidator(object):
    """JobManagementValidator helper class to perform job management validations"""

    def __init__(self, test_object):
        """Initialize instance of the JobManagementValidator class."""
        self.log = logger.get_log()
        self.test_object = test_object
        self.common_utils = CommonUtils(test_object.commcell)
        self.commcell = test_object.commcell
        self.job_manager = self.common_utils.job_manager
        self.client_machine_obj = Machine(test_object.client)
        self.schedule_policies = schedule_policies.SchedulePolicies(self.commcell)
        self._utility = options_selector.OptionsSelector(self.commcell)
        self.pre_scan_script = None
        self._cs_db = test_object.csdb
        self.management = JobManagementHelper(self.commcell)

    def _entities_setup(self):
        """
        Creates disklibrary and storage policy entities for automation.

        Returns:
            (obj)              --       subclient object(s) with the required entities

        """
        if not self.test_object.client:
            raise Exception('Client name must be passed in the input json')
        self._entities = options_selector.CVEntities(self.commcell)
        ready_ma = self._utility.get_ma()
        self.current_time = datetime.strftime(datetime.now(), '%Y-%m-%d_%H-%M-%S')
        entity_inputs = {
            'target':
                {
                    'client': self.test_object.client.client_name,
                    'agent': "File system",
                    'instance': "defaultinstancename",
                    'backupset': "defaultBackupSet",
                    'mediaagent': ready_ma
                },
            'disklibrary': {
                'name': "disklibrary_" + self.current_time,
                'mount_path': self._entities.get_mount_path(ready_ma),
                'cleanup_mount_path': True,
                'force': False,
                },
            'storagepolicy':
                {
                    'name': "storagepolicy_" + self.current_time,
                    'dedup_path': None,
                    'incremental_sp': None,
                    'retention_period': 3,
                    'force': False,
                }
        }
        self._entities.create(entity_inputs)

    def _create_pre_scan_script(self, content=''):
        """creates pre scan script required by subclient pre/post processing
           with content passed.

        Args:
            content           (str)   --  content that is to be written to file

        Returns:
            None

        Raises:
            Exception(Exception_Code, Exception_Message):
                if failed to create file

        """
        if self.client_machine_obj.os_info.lower() == 'windows':
            self.pre_scan_script = self.client_machine_obj.join_path(
                WINDOWS_TMP_DIR,
                'temp' + self.current_time + ".bat")
        else:
            self.pre_scan_script = self.client_machine_obj.join_path(
                UNIX_TMP_DIR,
                'temp' + self.current_time + ".sh")
        if self.client_machine_obj.check_file_exists(self.pre_scan_script):
            self.log.info("deleting existing pre scan script {0}".format(self.pre_scan_script))
            self.client_machine_obj.delete_file(self.pre_scan_script)
        self.log.info("creating pre scan command file {0}".format(self.pre_scan_script))
        self.client_machine_obj.create_file(self.pre_scan_script, content)
        if self.client_machine_obj.os_info.lower() == "unix":
            self.client_machine_obj.change_file_permissions(self.pre_scan_script, '777')

    def create_subclient(self, no_subclients=1):
        """
        Creates number of subclients based on the parameter no_subclients

            Args:
                 no_subclients  (int)           --          number of subclients to be created

            Returns:
                (list)                          --          properties of a created subclients
        """
        subclient_pattern = {
            'subclient':
                {
                    'name': "",
                    'client': self.test_object.client.client_name,
                    'agent': "File system",
                    'instance': "defaultinstancename",
                    'storagepolicy': "storagepolicy_" + self.current_time,
                    'backupset': 'defaultBackupSet',
                    'level': 1,
                    'size': 20,
                    'pre_scan_cmd': self.pre_scan_script,
                    'force': False,
                }
        }

        subclient_objects = []
        for value in range(no_subclients):
            current_time = datetime.strftime(datetime.now(), '%Y-%m-%d_%H-%M-%S')
            subclient_pattern['subclient']['name'] = str(self.test_object.id) + 'subclient' + current_time
            subclient_objects.append(self._entities.create(subclient_pattern))
        return subclient_objects

    def validate(self, features):
        """
        Entry point for feature validations

            Args:
                features        (list)      --      list of features to be validated.

        Returns :
            None
        """
        try:
            subclient_properties = None
            self.log.info('Creating DiskLibrary and Storage policy')
            self._entities_setup()
            self.log.info('Successfully created DiskLibrary and Storage policy')
            for feature in features:
                method_name = feature.lower()
                if method_name == 'queue_job_if_activity_is_disable':
                    subclient_properties = self.create_subclient()
                    entity_objects = [self.test_object.commcell, self.test_object.client,
                                      subclient_properties[0]['subclient']['agent'],
                                      subclient_properties[0]['subclient']['object']]
                    for entity in entity_objects:
                        self.validate_queue_job_if_activity_is_disable(entity, subclient_properties)
                elif method_name == 'job_total_running_time_commcell':
                    trigger = ['schedule_policy', 'schedule', 'ondemand']
                    self._create_pre_scan_script(content='exit 1')
                    subclient_properties = self.create_subclient()
                    for way in trigger:
                        self.validate_job_total_running_time_killed_scenario(subclient_properties, 'commcell', way)
                        self.validate_job_total_running_time_pending_scenario(subclient_properties, 'commcell', way)
                elif method_name == 'job_total_running_time_subclient':
                    trigger = ['schedule', 'ondemand']
                    self._create_pre_scan_script(content='exit 1')
                    subclient_properties = self.create_subclient()
                    for way in trigger:
                        self.validate_job_total_running_time_killed_scenario(subclient_properties, 'subclient', way)
                        self.validate_job_total_running_time_pending_scenario(subclient_properties, 'subclient', way)
                elif method_name == 'job_max_restarts_subclient':
                    self._create_pre_scan_script(content='exit 1')
                    subclient_properties = self.create_subclient()
                    trigger = ['ondemand', 'schedule']
                    for way in trigger:
                        self.validate_job_max_restarts(subclient_properties, 'subclient', way)
                elif method_name == 'job_max_restarts_commcell':
                    self._create_pre_scan_script(content='exit 1')
                    subclient_properties = self.create_subclient()
                    trigger = ['ondemand', 'schedule_policy', 'schedule']
                    for way in trigger:
                        self.validate_job_max_restarts(subclient_properties, 'commcell', way)
                elif method_name == 'high_water_mark_level':
                    subclient_properties = self.create_subclient(no_subclients=3)
                    self.validate_high_watermark_level(subclient_properties)
                elif method_name == 'job_preemption' or method_name == 'queue_jobs_if_conflicting_job_active' or \
                        method_name == 'queue_schedule_jobs':
                    if not subclient_properties:
                        subclient_properties = self.create_subclient()
                    getattr(self, 'validate_' + method_name)(subclient_properties, True)
                    getattr(self, 'validate_' + method_name)(subclient_properties, False)
                else:
                    raise Exception('pass the valid feature name {0}'.format(method_name))

        finally:
            time.sleep(30)
            # sometimes entity cleanup fails saying that jobs are still running,
            # JM marks job as completed but sometimes it will get stuck in index update.
            self._entities.cleanup()

    def validate_job_preemption(self, subclient_properties, action):
        """
        Validates job preemption feature at commcell level

            Args:
                subclient_properties    (dict)      --      dict obtained from CVEntities.create()

                action                  (bool)      --       enable/disable of feature to be validated

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails.

        Validates below cases:
            Disable preemption : Job should not be suspended.
            Enable preemption : Job should get suspended

        """
        self.log.info('Validating the job preemption feature at commcell level')
        subclient_object = subclient_properties[0]['subclient']['object']
        jobtype = "File System and Indexing Based (Data Protection)"
        self.log.info('obtaining the restart settings of jobtype {0}'.format(jobtype))
        initial_setting = self.management.get_restart_setting(jobtype)
        copy_settings = initial_setting.copy()
        self.log.info('initial settings are as follows {0}'.format(initial_setting))
        try:
            initial_setting['preemptable'] = action
            self.log.info('Preemptable property of jobtype is set to {0}= {1}'.format(jobtype, action))
            self.management.modify_restart_settings([initial_setting])
            self.log.info('modified settings are as follows {0}'.format(self.management.get_restart_setting(jobtype)))
            self.job_manager.job = self.common_utils.subclient_backup(subclient_object, "full", False)
            self.log.info('Full backup is triggered on subclient {0} with job id : {1}'.
                          format(subclient_object, self.job_manager.job.job_id))
            self.job_manager.wait_for_state('running', time_limit=2)
            self.job_manager.modify_job('suspend', hardcheck=False, wait_for_state=False)
            self.log.info('suspend command issued')

            if action:
                self.job_manager.wait_for_state(expected_state='suspended')
                if self.job_manager.job.status != 'Suspended':
                    raise Exception("Job is not suspended, it should be as preemptable is enabled")
                self.job_manager.modify_job('kill')
                self.log.info('job preemption enable feature validated successfully')
            else:
                self.job_manager.wait_for_state('completed', time_limit=10)
                self.log.info('job preemption disable feature validated successfully')
        except Exception as exp:
            if self.job_manager.job and self.job_manager.job not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.management.modify_restart_settings([copy_settings])

    def _job_via_schedule(self, subclient_object, options=None):
        """
        Triggers job via schedule

            Args:
                subclient_object        (obj)   --      instance of the subclient

                options                 (dict)  --      advanced options to be included

            Returns:
                job object

        """
        date, rtime = schedulerhelper.ScheduleCreationHelper.add_minutes_to_datetime(time=datetime.today())
        pattern = {"freq_type": 'one_time',
                   "active_start_date": str(date),
                   "active_start_time": str(rtime)
                   }
        schedule_object = self.common_utils.subclient_backup(subclient_object, "full", wait=False,
                                                             common_backup_options=options,
                                                             schedule_pattern=pattern)
        job_list = schedulerhelper.SchedulerHelper(schedule_object,
                                                   self.commcell).check_job_for_taskid(retry_count=5,
                                                                                       retry_interval=30)
        job_object = job_list[0]
        return job_object

    def _job_via_schedule_policy(self, subclient_object):
        """
        Triggers job via schedule Policy

            Args:
                subclient_object        (obj)   --      instance of the subclient

            Returns:
                job object

        """
        assocs = [{
            "clientName": self.test_object.client.client_name,
            "backupsetName": 'defaultBackupSet',
            "subclientName": subclient_object.name,
            "instanceName": "DefaultInstanceName",
            "appName": "File System"
        }]

        date, rtime = schedulerhelper.ScheduleCreationHelper.add_minutes_to_datetime(time=datetime.today())
        current_time = datetime.strftime(datetime.now(), '%Y-%m-%d_%H-%M-%S')
        pattern = [{'name': 'Auto' + current_time,
                    'pattern': {"freq_type":
                                'one_time',
                                "active_start_date": str(date),
                                "active_start_time": str(rtime)
                                }}]
        types = [{"appGroupName": "Protected Files"}]
        schedule_policy = self.schedule_policies.add(name='auto' + current_time,
                                                     policy_type='Data Protection', associations=assocs,
                                                     schedules=pattern, agent_type=types)
        schedules_list = schedule_policy.all_schedules
        schedule_object = schedules.Schedule(self.test_object.commcell,
                                             schedule_id=schedules_list[0]['schedule_id'])
        job_list = schedulerhelper.SchedulerHelper(schedule_object,
                                                   self.commcell).check_job_for_taskid(retry_count=5,
                                                                                       retry_interval=30)
        job_object = job_list[0]
        return job_object

    def validate_job_max_restarts(self, subclient_properties, entitiy_level, trigger):
        """
        Validates job max restarts feature

            Description of the feature:
            say max restarts = 2
            job should be resumed or restarted as per max restarts
            after exhaustive attempts or max restarts job should be failed

            Args:
                subclient_properties    (dict)  --      dict obtained from CVEntities.create()

                entitiy_level           (str)   --      at which level feature need to be validated

                    Values:

                        "commcell"
                        "subclient"

                trigger                 (str)   --      in what way job must should be triggered

                    Values:

                        "ondemand"
                        "schedule"
                        "schedule_policy"

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails
        """
        job_object = None
        max_restarts = 2
        restart_interval = 1
        jobtype = "File System and Indexing Based (Data Protection)"
        self.log.info('Validating job max restarts at {0} level, job is triggered through = {1}'.format(
            entitiy_level, trigger))
        subclient_object = subclient_properties[0]['subclient']['object']
        self.log.info('obtaining the restart settings of jobtype {0}'.format(jobtype))
        initial_setting = self.management.get_restart_setting(jobtype)
        default_settings = initial_setting.copy()
        try:
            if entitiy_level.lower() == 'commcell':
                self.log.info('initial settings are as follows {0}'.format(initial_setting))

                initial_setting['restartable'] = True
                initial_setting['maxRestarts'] = max_restarts

                initial_setting['restartIntervalInMinutes'] = restart_interval
                initial_setting['enableTotalRunningTime'] = False
                self.management.modify_restart_settings([initial_setting])
                self.log.info('modified settings are as follows {0}'.format(
                    self.management.get_restart_setting(jobtype)))
                if trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object)
                elif trigger == 'schedule_policy':
                    job_object = self._job_via_schedule_policy(subclient_object)
                elif trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, "full", wait=False)
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            elif entitiy_level.lower() == 'subclient':
                options = {
                    'enable_number_of_retries': True,
                    'number_of_retries': max_restarts
                }
                initial_setting['restartIntervalInMinutes'] = restart_interval
                initial_setting['enableTotalRunningTime'] = False

                self.management.modify_restart_settings([initial_setting])
                self.log.info('modified settings are as follows {0}'.format(
                    self.management.get_restart_setting(jobtype)))
                if trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object, options)
                elif trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, 'full', wait=False,
                                                                    common_backup_options=options)
                elif trigger == 'schedule_policy':
                    pass    # Changes need to be done in schedule policy creation(sdk) to change JM settings
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            self.job_manager.job = job_object
            self.log.info('Full backup is triggered on subclient {0} with job id : {1}'.
                          format(subclient_object, self.job_manager.job.job_id))

            self.job_manager.wait_for_state(expected_state='failed', time_limit=5)
            # if 'Number of restarts has exceeded the maximum number allowed' not in self.job_manager.job.delay_reason:
            #    raise Exception("delay reason didn't match, obtained = {0}".format(self.job_manager.job.delay_reason))
            # self.log.info('Validated job delay reason')
            self.log.info('Delay reason of the job {0} is {1}'.format(self.job_manager.job.job_id,
                                                                      self.job_manager.job.delay_reason))
            self._utility.exec_commserv_query("select count(phase) from JMBkpAtmptStats where "
                                              "jobid = {0} and phase = {1} group by "
                                              "phase".format(self.job_manager.job.job_id, 3))
            if int(self._cs_db.rows[0][0]) != max_restarts + 1:
                raise Exception('given number of attempts are not performed,'
                                ' failed at {0} {1}'.format(entitiy_level, trigger))
            else:
                self.log.info('level = {0}, job triggered via {1} successfully validated'
                              ' max restarts'.format(entitiy_level, trigger))
        except Exception as exp:
            if job_object and job_object.status not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.log.info('Applying default settings as follows {0}'.format(default_settings))
            self.management.modify_restart_settings([default_settings])

    def validate_job_total_running_time_pending_scenario(self, subclient_properties, entitiy_level, trigger):
        """
        Validates job total running time feature(scenario 1)

            Description of the scenario 1:

            say total running time = 5 minutes
            scenario 1: after 5 minutes if the job is in pending or waiting state then the job should be failed

            Args:
                subclient_properties    (dict)  --      dict obtained from CVEntities.create()

                entitiy_level           (str)   --      at which level feature need to be validated

                    Values:

                        "commcell"
                        "subclient"

                trigger                 (str)   --      in what way job must be triggered

                    Values:

                        "ondemand"
                        "schedule"
                        "schedule_policy"

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails
        """
        job_object = None
        total_run_rime = 120  # secs
        restart_interval = 30  # mins
        jobtype = "File System and Indexing Based (Data Protection)"
        self.log.info('entity level = {0} job triggered via = {1} ,'
                      'Validating, pending -> failed, when job is'
                      ' in pending state after reaches time interval.'.format(entitiy_level, trigger))
        subclient_object = subclient_properties[0]['subclient']['object']
        self.log.info('obtaining the restart settings of jobtype {0}'.format(jobtype))
        initial_setting = self.management.get_restart_setting(jobtype)
        default_settings = initial_setting.copy()
        try:
            # ---------------------------- scenario #1 start----------------------------------------------------- #
            if entitiy_level == 'commcell':
                self.log.info('initial settings are as follows {0}'.format(initial_setting))
                initial_setting['enableTotalRunningTime'] = True
                initial_setting['restartIntervalInMinutes'] = restart_interval
                initial_setting['totalRunningTime'] = total_run_rime
                initial_setting['killRunningJobWhenTotalRunningTimeExpires'] = False
                self.log.info('enabling the total running time property with {0} minutes for jobtype'
                              ' {1}'.format(3, jobtype))
                self.management.modify_restart_settings([initial_setting], validation=False)
                self.log.info('modified settings are as follows {0}'.format(
                    self.management.get_restart_setting(jobtype)))
                if trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, "full", False)
                elif trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object)
                elif trigger == 'schedule_policy':
                    job_object = self._job_via_schedule_policy(subclient_object)
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            elif entitiy_level == 'subclient':
                options = {
                    'enable_total_running_time': True,
                    'total_running_time': total_run_rime
                }
                if trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, 'full', wait=False,
                                                                    common_backup_options=options)
                elif trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object, options)
                elif trigger == 'schedule_policy':
                    pass    # Changes need to be done in schedule policy creation(sdk) to change JM settings
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            self.job_manager.job = job_object
            self.log.info('Full backup is triggered on subclient {0} with job id : {1}'.
                          format(subclient_object, self.job_manager.job.job_id))
            self.job_manager.wait_for_state(expected_state='pending', time_limit=2)
            fail_status = self.job_manager.wait_for_state(expected_state=['failed', 'killed'],
                                                          hardcheck=False, time_limit=5)
            if 'The job has exceeded the total running time' not in self.job_manager.job.delay_reason:
                raise Exception("delay reason didn't match, obtained = {0}".format(self.job_manager.job.delay_reason))
            self.log.info('Validated job delay reason')
            self.log.info('Delay reason of the job {0} is {1}'.format(self.job_manager.job.job_id,
                                                                      self.job_manager.job.delay_reason))
            if fail_status:
                self.log.info('job is in failed state')
                self.log.info('Successfully Validated at level {0} job'
                              ' triggered via = {1} ,pending -> failed,'
                              ' when job is in pending state'
                              ' after reaches time interval'.format(entitiy_level, trigger))
            else:
                raise Exception('job is not in failed state')
        except Exception as exp:
            if job_object and job_object not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.log.info('Applying default settings as follows {0}'.format(default_settings))
            default_settings['enableTotalRunningTime'] = True
            self.management.modify_restart_settings([default_settings], validation=False)
            default_settings['enableTotalRunningTime'] = False
            self.management.modify_restart_settings([default_settings], validation=False)

    def validate_job_total_running_time_killed_scenario(self, subclient_properties, entitiy_level, trigger):
        """
        Validates job total running time feature(scenario 2)

             Description of the scenario 2:

            say total running time = 5 minutes
            if killRunningJobWhenTotalRunningTimeExpires is enabled : running job should be killed after time expires.

            Args:
                subclient_properties    (dict)  --      dict obtained from CVEntities.create()

                entitiy_level           (str)   --      at which level feature need to be validated

                    Values:

                        "commcell"
                        "subclient"

                trigger                 (str)   --      in what way job must should be triggered

                    Values:

                        "ondemand"
                        "schedule"
                        "schedule_policy"

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails
        """
        job_object = None
        total_runtime = 60  # secs
        acceptable_delay = 600  # secs
        restart_interval = 30  # mins
        jobtype = "File System and Indexing Based (Data Protection)"
        self.log.info('entity level = {0}, job triggered via = {1},'
                      ' Validating, killRunningJobWhenTotalRunningTimeExpires'.format(entitiy_level, trigger))
        backupset_object = subclient_properties[0]['subclient']['backupset']
        # Picking default subclient from DefaultBackupSet for this testcase, Because backup must run for certain time
        # to validate this case, so instead of generating dummy data on client, i am using default subclient.
        subclient_object = Subclients(backupset_object).get("default")
        if not subclient_object.properties.get(
                'commonProperties').get('storageDevice').get('dataBackupStoragePolicy').get('storagePolicyName', {}):
            raise Exception('Please attach a storage policy to default subclient of DefaultBackupSet')
        self.log.info('obtaining the restart settings of jobtype {0}'.format(jobtype))
        initial_setting = self.management.get_restart_setting(jobtype)
        default_settings = initial_setting.copy()
        self.job_manager.modify_all_jobs(operation_type='kill')
        time.sleep(20)
        try:
            if entitiy_level == 'commcell':
                initial_setting['enableTotalRunningTime'] = True
                initial_setting['restartIntervalInMinutes'] = restart_interval
                initial_setting['totalRunningTime'] = total_runtime
                initial_setting['killRunningJobWhenTotalRunningTimeExpires'] = True
                self.log.info('enabling killRunningJobWhenTotalRunningTimeExpires property for jobtype '
                              '{0}'.format(jobtype))
                self.management.modify_restart_settings([initial_setting], validation=False)
                self.log.info('modified settings are as follows {0}'.format(
                    self.management.get_restart_setting(jobtype)))
                if trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, "full", False)
                elif trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object)
                elif trigger == 'schedule_policy':
                    job_object = self._job_via_schedule_policy(subclient_object)
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            elif entitiy_level == 'subclient':
                options = {
                    'enable_total_running_time': True,
                    'total_running_time': total_runtime,
                    'kill_running_job_when_total_running_time_expires': True
                }
                if trigger == 'ondemand':
                    job_object = self.common_utils.subclient_backup(subclient_object, 'full', wait=False,
                                                                    common_backup_options=options)
                elif trigger == 'schedule':
                    job_object = self._job_via_schedule(subclient_object, options)
                elif trigger == 'schedule_policy':
                    pass    # Changes need to be done in schedule policy creation(sdk) to change JM settings
                else:
                    raise Exception('Unexpected value passed for parameter trigger')
            self.job_manager.job = job_object
            self.log.info('Full backup is triggered on subclient')
            self.job_manager.wait_for_state(expected_state='running')
            kill_status = self.job_manager.wait_for_state(expected_state=['failed', 'killed'],
                                                          hardcheck=False,
                                                          time_limit=
                                                          ((total_runtime + acceptable_delay)//total_runtime))
            if 'The job has exceeded the total running time' not in self.job_manager.job.delay_reason:
                raise Exception("delay reason didn't match, obtained = {0}".format(self.job_manager.job.delay_reason))
            self.log.info('Validated job delay reason')
            self.log.info('Delay reason of the job {0} is {1}'.format(self.job_manager.job.job_id,
                                                                      self.job_manager.job.delay_reason))
            if kill_status:
                self.log.info('Successfully Validated at level = {0}, job triggered via = {1}'
                              ' killRunningJobWhenTotalRunningTimeExpires'.format(entitiy_level, trigger))
            else:
                raise Exception('job with id: {0} not killed successfully'.format(self.job_manager.job.job_id))
        except Exception as exp:
            if job_object and job_object not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.log.info('Applying default settings as follows {0}'.format(default_settings))
            default_settings['enableTotalRunningTime'] = True
            self.management.modify_restart_settings([default_settings], validation=False)
            default_settings['enableTotalRunningTime'] = False
            self.management.modify_restart_settings([default_settings], validation=False)

    def validate_high_watermark_level(self, subclient_properties):
        """
        Validates high water mark level feature

            Args:
                subclient_properties        (dict)      --      dict obtained from CVEntities.create()

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails

        Description of the faeture:

            say high watermark level = 6
            maximum 5 jobs should be running, 6th job should in waiting state
        """
        job_objects = []
        count = 0
        self.log.info('Validating High water mark level feature at commcell level')
        high_water_mark_level = 2
        default_value = self.management.job_stream_high_water_mark_level
        try:
            self.management.job_stream_high_water_mark_level = high_water_mark_level
            self.log.info('high water mark level is set to {0}'.format(
                self.management.job_stream_high_water_mark_level))

            for element in subclient_properties:
                subclient_object = element['subclient']['object']
                job_object = self.common_utils.subclient_backup(subclient_object, "full", False)
                job_objects.append(job_object)
            jobs_status = list()
            time_limit = 30  # mins
            total_time = time.time() + time_limit * 60
            while jobs_status.count('Completed') != 3:
                jobs_status.clear()
                jobs_status.append(job_objects[0].status)
                jobs_status.append(job_objects[1].status)
                jobs_status.append(job_objects[2].status)
                self.log.info('Status of the jobs: {0}, {1}, {2} = {3}'.format(
                                     job_objects[0].job_id, job_objects[1].job_id, job_objects[2].job_id, jobs_status))
                if jobs_status.count('Running') > high_water_mark_level:
                    count += 1
                    if count > 4:
                        raise Exception('High Water mark level failed, Three jobs are running instead of two.')
                if 'Pending' in jobs_status or 'Failed' in jobs_status:
                    raise Exception('One of the job went into unexpected state,'
                                    ' status of three jobs as follows {0}, {1}, {2}= {3}'.format(
                                     job_objects[0].job_id, job_objects[1].job_id, job_objects[2].job_id, jobs_status))
                if time.time() >= total_time:
                    raise Exception('Total time of {0} mins elapsed, check logs for more info'.format(time_limit))
                time.sleep(10)
            self.log.info('High water mark level validated succesfully')
        except Exception as exp:
            if job_objects:
                self.job_manager.modify_all_jobs('kill')
                time.sleep(60)
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.management.job_stream_high_water_mark_level = default_value

    def validate_queue_jobs_if_conflicting_job_active(self, subclient_properties, action):
        """
        Validates Queue jobs if conflicting job active feature at commcell level

            Args:

                subclient_properties        (dict)      --      dict obtained from CVEntities.create()

                action                  (bool)      --       enable/disable of feature to be validated

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails

         Description of the feature:

            run two backup jobs on same subclient, second backup job should be in queued state
        """
        self.log.info('Validating queue job if conflicting job active feature at commcell levels')
        job_object1 = None
        default_state = self.management.queue_jobs_if_conflicting_jobs_active
        try:
            self.management.queue_jobs_if_conflicting_jobs_active = action
            self.log.info('enabling the property "queue jobs if conflicting job active" '
                          '{0}'.format(self.management.queue_jobs_if_conflicting_jobs_active))

            subclient_object = subclient_properties[0]['subclient']['object']
            job_object1 = self.common_utils.subclient_backup(subclient_object, "full", False)
            self.log.info('Full backup is triggered on subclient with job id %s', job_object1.job_id)
            job_object2 = self.common_utils.subclient_backup(subclient_object, "full", False)
            self.log.info("status = {0}, Delay Reason = {1}".format(job_object2.status, job_object2.delay_reason))
            time.sleep(20)
            if action:
                if job_object2.status == "Queued":
                    self.log.info('Delay reason of the second job {0} is {1}'.format(job_object2.job_id,
                                                                                     job_object2.delay_reason))
                    self.log.info('second backup job is in queue, Validated')
                    self.job_manager.job = job_object1
                    complete_status = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                    if complete_status:
                        self.log.info('first backup job is successfully completed')
                        self.job_manager.job = job_object2
                        self.job_manager.modify_job('resume')
                        status2 = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                        if status2:
                            self.log.info('second backup job is successfully completed')
                            self.log.info('queue jobs if conflicting job active feature is Validated successfully')
                        else:
                            raise Exception('Second backup job with id : {0} not completed'
                                            'successfully'.format(self.job_manager.job.job_id))
                    else:
                        raise Exception('First backup job with id : {0} not completed'
                                        'successfully'.format(self.job_manager.job.job_id))
                else:
                    raise Exception('Second backup job with id : {0} not queued'
                                    'successfully'.format(self.job_manager.job.job_id))
            else:
                if 'Another backup is running for client' in job_object2.delay_reason:
                    self.job_manager.job = job_object1
                    self.job_manager.modify_job('kill')
                    self.log.info('queue jobs if conflicting job active feature = {0} is'
                                  ' Validated successfully'.format(action))
        except Exception as e:
            self.job_manager.modify_all_jobs('kill')
            time.sleep(10)
            self.log.info(e)
        finally:
            self.management.queue_jobs_if_conflicting_jobs_active = default_state

    def _backup_activity_switch(self, entity_object, switch):
        """
        Enables/disables backup activity at various levels

            Args:
                  entity_object     (obj)       --      Instance of the any level

                    values:
                        Commcell
                        Client
                        Agent
                        Subclient

                    switch          (bool)      --      True/False

            Returns:
                None
        """
        if isinstance(entity_object, (Client, Agent, Subclient)):
            activity_helper = activitycontrolhelper.ActivityControlHelper(self.test_object)
            self.log.info('{0} the backup activity on {1} level '.format(switch, entity_object))
            activity_helper.modify_activity(activity_type='backup',
                                            entity_object=entity_object,
                                            action=switch)
        elif isinstance(entity_object, Commcell):
            activity_controller = activitycontrol.ActivityControl(entity_object)
            self.log.info('{0} the data management activity on {1} level '.format(switch, entity_object))
            activity_controller.set(activity_type='DATA MANAGEMENT', action=switch)

    def _restore_activity_switch(self, entity_object, switch):
        """
        Enables/disables restore activity at various levels

            Args:
                  entity_object     (obj)       --      Instance of the any level

                    values:
                        Commcell
                        Client
                        Agent
                        Subclient

                    switch          (bool)      --      True/False

            Returns:
                None
        """
        if isinstance(entity_object, (Client, Agent)):
            activity_helper = activitycontrolhelper.ActivityControlHelper(self.test_object)
            self.log.info('{0} the restore activity on {1} level '.format(switch, entity_object))
            activity_helper.modify_activity(activity_type='restore',
                                            entity_object=entity_object,
                                            action=switch)
        elif isinstance(entity_object, Commcell):
            activity_controller = activitycontrol.ActivityControl(entity_object)
            self.log.info('{0} the restore activity on {1} level '.format(switch, entity_object))
            activity_controller.set(activity_type='DATA RECOVERY', action=switch)

    def validate_queue_job_if_activity_is_disable(self, entity_object, subclient_properties):
        """
        Validates queue job if activity is disable at various levels

            Args:

                entity_object       (obj)       --      Instance of the any level

                    values:
                        Commcell
                        Client
                        Agent
                        Subclient

                subclient_properties        (dict)      --      dict obtained from CVEntities.create()

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails
        Description of the feature:

            entity_object : commcell, client, agent, subclient
            Based on the entity object, backup and restore will be disabled correspondingly and
            feature will be validated.
        """
        self.log.info('Validating queue job if activity disabled feature at level {0}'.format(entity_object))
        default_setting = self.management.queue_jobs_if_activity_disabled
        try:
            self.management.queue_jobs_if_activity_disabled = True
            self.log.info('queue jobs if activity is disable {0}'.format(
                self.management.queue_jobs_if_activity_disabled))
            subclient_object = subclient_properties[0]['subclient']['object']
            self.test_object.subclient = subclient_object
            self._backup_activity_switch(entity_object, switch='Disable')
            backup_job_obj = self.common_utils.subclient_backup(subclient_object, "full", False)
            if backup_job_obj.status == 'Queued':
                self._backup_activity_switch(entity_object, switch='Enable')
                self.job_manager.job = backup_job_obj
                self.log.info('Delay reason of the job {0} is {1}'.format(self.job_manager.job.job_id,
                                                                          self.job_manager.job.delay_reason))
                backup_job_status = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                if backup_job_status:
                    self.log.info('backup job with id : {0} is completed successfuly'.format(backup_job_obj.job_id))
                    if not isinstance(entity_object, Subclient):
                        self._restore_activity_switch(entity_object, switch='Disable')
                        restore_job_obj = self.common_utils.subclient_restore_in_place(
                            subclient_properties[0]['subclient']['content'],
                            subclient=subclient_object,
                            wait=False)
                        if restore_job_obj.status == 'Queued':
                            # -------------------------------- Add delay ------------------------------------------
                            self._restore_activity_switch(entity_object, switch='Enable')
                            self.job_manager.job = restore_job_obj
                            self.log.info('Delay reason of the job '
                                          '{0} is {1}'.format(self.job_manager.job.job_id,
                                                              self.job_manager.job.delay_reason))
                            restore_job_status = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                            if restore_job_status:
                                self.log.info('restore job with id : {0} is completed successfuly'.format(
                                    restore_job_obj.job_id))
                            else:
                                raise Exception('Restore job with job id: {0} not completed successfully'.
                                                format(restore_job_obj.job_id))
                        else:
                            raise Exception(
                                'Restore job with job id: {0} not queued successfully'.format(restore_job_obj.job_id))
                else:

                    raise Exception('Backup job with id: {0} not completed successfully'
                                    .format(self.job_manager.job.job_id))
            else:
                raise Exception('backup job with id: {0} not queued successfully'.format(backup_job_obj.job_id))
            self.log.info('queue jobs if activity disabled is Validated at {0} level'.format(entity_object))
        except Exception as exp:
            if self.job_manager.job and self.job_manager.job.status not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
                self.log.info('Enabling the activities at level {0}'.format(entity_object))
            self._backup_activity_switch(entity_object, switch='Enable')
            self._restore_activity_switch(entity_object, switch='Enable')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.management.queue_jobs_if_activity_disabled = default_setting

    def validate_queue_schedule_jobs(self, subclient_properties, action):
        """
        Validates queue schedule jobs

            Args:

                subclient_properties        (dict)     --       dict obtained from CVEntities.create()

                action                  (bool)      --       enable/disable of feature to be validated

            Returns:
                None

            Raises:
                Exception:
                    when validation of the feature fails.

        Description of the feature:
            validates whether the scheduled job is getting queued or not.
        """
        job_obj = None
        reason = 'The scheduled job is queued because the Queue Scheduled Jobs option is on'
        self.log.info('Validating queue schedule jobs feature at commcell level')
        default_state = self.management.queue_scheduled_jobs
        try:
            self.management.queue_scheduled_jobs = action
            self.log.info('queue schedule jobs is enabled {0}'.format(self.management.queue_scheduled_jobs))
            subclient_object = subclient_properties[0]['subclient']['object']
            date, start_time = schedulerhelper.ScheduleCreationHelper.add_minutes_to_datetime(time=datetime.today())
            pattern = {"freq_type": 'one_time',
                       "active_start_date": str(date),
                       "active_start_time": str(start_time)
                       }
            schedule_obj = self.common_utils.subclient_backup(subclient_object, "full",
                                                              wait=False, schedule_pattern=pattern)
            job_list = schedulerhelper.SchedulerHelper(schedule_obj,
                                                       self.commcell).check_job_for_taskid(retry_count=5,
                                                                                           retry_interval=30)
            job_obj = job_list[0]
            self.log.info('Backup with id : {0}'.format(job_obj.job_id))
            self.log.info('status of the job {0}'.format(job_obj.status))
            self.job_manager.job = job_obj
            if action:
                queue_status = self.job_manager.wait_for_state(expected_state='queued', hardcheck=False, time_limit=5)
                if queue_status:
                    self.log.info('scheduled job is successfully Queued')
                    self.job_manager.job = job_obj
                    if reason not in self.job_manager.job.delay_reason:
                        raise Exception("delay reason didn't match, obtained = {0}".format(
                            self.job_manager.job.delay_reason))
                    self.log.info('Validated job delay reason')
                    self.log.info('Delay reason of the job {0} is {1}'.format(self.job_manager.job.job_id,
                                                                              self.job_manager.job.delay_reason))
                    self.job_manager.modify_job('resume')
                    backup_status = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                    if backup_status:
                        self.log.info('queue schedule jobs(Enable) is successfully Validated')
                    else:
                        raise Exception('Backup job with id: {0} not completed successfully'
                                        .format(self.job_manager.job.job_id))
                else:
                    raise Exception('backup job with id: {0} not queued successfully'.format(job_obj.job_id))
            else:
                complete_status = self.job_manager.wait_for_state(hardcheck=False, time_limit=10)
                if complete_status:
                    self.log.info('queue schedule jobs(Disable) is successfully Validated')
                else:
                    raise Exception('Backup job with id: {0} not completed successfully'
                                    .format(self.job_manager.job.job_id))
        except Exception as exp:
            if job_obj and job_obj.status not in ('Completed', 'Failed', 'Killed'):
                self.job_manager.modify_job('kill')
            self.log.error(exp)
            self.test_object.result_string = str(exp)
            self.test_object.status = FAILED
        finally:
            self.management.queue_scheduled_jobs = default_state
