# -*- coding: utf-8 -*-
# pylint: disable=W1202
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for performing Commcell -> Control Panel -> DR Management related operations.

DRManagementHelper: Class for performing DR Management operations

DRCommvaultCloudManagement: Class for downloading dr dumps from commvault cloud.


DRManagementHelper:
==========================

    __init__()                          --   initializes DisasterRecoveryManagement class object

    refresh()                           --   retrives the latest dr settings

    set_local_dr_path                   --   sets the local dr path and validates the setting.

    set_network_dr_path                 --   sets the unc path and validates the setting.

    upload_metdata_to_commvault_cloud   --   sets ths account to be used for commvault cloud backup.

    upload_metdata_to_cloud_library     --   sets the libarary to be used for cloud backup.

    impersonate_user                    --   account to be used for execution of pre/post scripts

    use_impersonate_user                --  gets the setting use_impersonate_user

    number_of_metadata                  --  sets number of metadata folders to be retained
                                            and validates the setting..

    number_of_metadata                  --  gets the value of the setting, number of metadata to be retained.

    use_vss                             --  sets the property and validates the setting.

    use_vss                             --  gets the property

    wild_card_settings                  --  sets log file names to be backed up for selected
                                            clients and validates the setting..

    wild_card_settings                  --  gets the log file names.

    backup_metadata_folder              --  gets the metadata destination location or path.

    upload_backup_metadata_to_cloud     --  gets the property.

    upload_backup_metadata_to_cloud_lib --  sets the property and validates the setting..

    dr_storage_policy                   --  gets the associated dr storage policy

    dr_storage_policy                   --  sets the dr storage policy and validates the setting.

    pre_scan_process                    --  sets the property and validates the setting.

    pre_scan_process                    --  gets the property

    post_scan_process                   --  sets the property and validates the setting.

    post_scan_process                   --  gets the property

    pre_backup_process                  --  sets the property and validates the setting.

    pre_backup_process                  --  gets the property

    post_backup_process                 --  sets the property and validates the setting.

    post_backup_process                 --  gets the property

    run_post_scan_process               --  sets the property and validates the setting.

    run_post_scan_process               --  gets the property

    run_post_backup_process             --  sets the property and validates the setting.

    run_post_backup_process             --  gets the property

DRCommvaultCloudManagement
===============================

    __init__()                          --   initializes DRCommvaultCloudManagement class object.

    refresh()                           --   refreshes the company details.

    _get_info()                         --   executes get api to fetch the company details.

    get_company_name()                  --   gets the available company names.

    get_commcell_ids_of_company         --   gets the available commcell id's(hexadecimal) for a given company.

    get_number_of_set_folders_of_a_commcell --  gets the count of set folders available for a given commcell.

    get_available_set_folders_of_a_commcell --  gets the available set folder names for a given commcell.

    get_file_names_of_a_set_folder          --  gets the file names for a given set folder name.

    get_set_id_of_a_set_folder              --  gets the set id for a given set folder name.

    get_status                              --  gets the status of last dr backup

    get_file                                --  gets the file data in binary format.

"""

from cvpysdk.disasterrecovery import DisasterRecoveryManagement
from cvpysdk.policies.storage_policies import StoragePolicy
from AutomationUtils import logger


class DRManagementHelper(object):
    """Class to perform all the disaster recovery management operations on commcell"""

    def __init__(self, commcell):
        """Initializes DisasterRecoveryManagement object

            Args:
            commcell    (object)    --  instance of commcell

        """
        self.log = logger.get_log()
        self.management = DisasterRecoveryManagement(commcell)
        self._commcell = commcell

    def refresh(self):
        """
        refreshes the dr settings associated with commcell.

        Returns:
            None
        """
        self.management.refresh()
        self.log.info('Settings have been refreshed successfully')

    def set_local_dr_path(self, path, validate=True):
        """
        Sets local DR path

            Args:
                 path       (str)       --         local path.

                 validate   (bool)      --         based on the bool value passed validation of setting
                                                   will be handled.

            Returns:
                None
        """
        self.management.set_local_dr_path(path=path)
        if validate:
            if self.management.backup_metadata_folder != path:
                raise Exception('{0} path has not been set successfully'.format(path))
            self.log.info('{0} path has been set successfully'.format(path))

    def set_network_dr_path(self, path, username, password, validate=True):
        """
        Sets network DR path

            Args:
                 path       (str)       --      UNC path.

                 username   (str)       --      username with admin privileges of the remote machine.

                 password   (str)       --      password.

                 validate   (bool)      --      whether to validate the setting has been updated successfully or not

            Returns:
                None
        """
        self.management.set_network_dr_path(path, username, password)
        if validate:
            if self.management.backup_metadata_folder != path:
                raise Exception('{0} path has not been set successfully'.format(path))
            self.log.info('{0} path has been set successfully'.format(path))

    def upload_metdata_to_commvault_cloud(self, flag, username=None, password=None, validate=True):
        """
        Enable/Disable upload metadata to commvault cloud setting.

            Args:
                 flag       (bool)      --      True/False.

                 username   (str)       --      username of the commvault cloud.

                 password   (str)       --      password of the commvault cloud.

                 validate   (bool)      --      whether to validate the setting has been updated successfully or not

            Returns:
                 None
        """
        self.management.upload_metdata_to_commvault_cloud(flag, username, password)
        if validate:
            if self.management.upload_backup_metadata_to_cloud != flag:
                raise Exception('upload metadata to commvault cloud is not set'
                                ' successfully with value {0}'.format(flag))
            self.log.info('upload metadata to commvault cloud = {0}'.format(flag))

    def upload_metdata_to_cloud_library(self, flag, libraryname=None, validate=True):
        """
        Enable/Disable upload metadata to cloud library

            Args:
                 flag       (bool)      --      True/False.

                 libraryname   (str/object)    --      Third party cloud library name or disklibrary object.

                 validate   (bool)      --      whether to validate the setting has been updated successfully or not

            Returns:
                None
        """
        self.management.upload_metdata_to_cloud_library(flag, libraryname)
        if validate:
            if self.management.upload_backup_metadata_to_cloud_lib != flag:
                raise Exception('upload metadata to cloud library is not set'
                                ' successfully with value {0}'.format(flag))
            self.log.info('upload metadata to cloud library = {0}'.format(flag))

    def impersonate_user(self, flag, username, password, validate=True):
        """
        Enable/Disable Impersonate user option for pre/post scripts.

            Args:
                flag        (bool)      --  True/False.

                username    (str)       --  username with admin privileges.

                password    (str)       --  password for the account.

                validate   (bool)      --      whether to validate the setting has been updated successfully or not

            Returns:
                None
        """
        self.management.impersonate_user(flag, username, password)
        if validate:
            if self.management.use_impersonate_user != flag:
                raise Exception('impersonate user is not set to value {0}'.format(flag))
            self.log.info('impersonate user = {0}'.format(flag))

    @property
    def use_impersonate_user(self):
        """
        gets the impersonate user(True/False)

            Returns:
                  True/False
        """
        return self.management.use_impersonate_user

    @property
    def number_of_metadata(self):
        """
         gets the value, Number of metadata folders to be retained.

            Returns:
                number of metadata     (int)
        """
        return self.management.number_of_metadata

    @number_of_metadata.setter
    def number_of_metadata(self, value):
        """
        Sets the value, Number of metadata folders to be retained.

            Args:
                value       (int)       --      number of metadata folders to be retained.

            Returns:
                None
        """
        self.management.number_of_metadata = value
        if self.management.number_of_metadata != value:
            raise Exception('number of metadata is not successfully with value {0}'.format(value))
        self.log.info('number of metadata = {0}'.format(value))

    @property
    def use_vss(self):
        """
        gets the value, use vss()

            Returns:
                True/False
        """
        return self.management.use_vss

    @use_vss.setter
    def use_vss(self, flag):
        """
        sets the value, use vss

            Args:
                 flag   (bool)      --      True/Flase

            Returns:
                None
        """
        self.management.use_vss = flag
        if self.management.use_vss != flag:
            raise Exception('use vss is not set successfully with value {0}'.format(flag))
        self.log.info('use vss = {0}'.format(flag))

    @property
    def wild_card_settings(self):
        """
        gets the wild card settings

            Returns:
                (list)       --     client logs that are to be backed up
        """
        return self.management.wild_card_settings.split(';')

    @wild_card_settings.setter
    def wild_card_settings(self, logs):
        """
        sets the wild card setting

            Args:
                 logs    (list)      --      log file names

            Returns:
                  None
        """
        self.management.wild_card_settings = logs
        log_names = self.management.wild_card_settings.lower()
        for log_name in logs:
            if log_name.lower() not in log_names:
                raise Exception('{0} log names has not set succesfully'.format(logs))
        self.log.info('{0} log names got updated successfully')

    @property
    def backup_metadata_folder(self):
        """
        gets the backup metadata folder

            Returns:
                 (str)      --      Backup metadata folder
        """
        return self.management.backup_metadata_folder

    @property
    def upload_backup_metadata_to_cloud(self):
        """
        gets the upload backup metadata to cloud setting

            Returns:
                 True/False
        """
        return self.management.upload_backup_metadata_to_cloud

    @property
    def upload_backup_metadata_to_cloud_lib(self):
        """
        gets the upload metadata to cloud lib

            Returns:
                (str)       --      Third party library name
        """
        return self.management.upload_backup_metadata_to_cloud_lib

    @property
    def dr_storage_policy(self):
        """
        gets the storage policy name, that is being used for DR backups

            Returns:
                (object)       --      storage policy object.
         """
        return StoragePolicy(self._commcell, self.management.dr_storage_policy)

    @dr_storage_policy.setter
    def dr_storage_policy(self, storage_policy_object):
        """
        sets the storage policy for DR jobs

            Args:
                storage_policy_object       (object)        --      object of the storage policy

            Returns:
                None
        """
        self.management.dr_storage_policy = storage_policy_object
        if self.management.dr_storage_policy != storage_policy_object.name:
            raise Exception('storage policy {0} has not been set successfully'.format(storage_policy_object.name))
        self.log.info('DR Storage policy = {0}'.format(storage_policy_object.name))

    @property
    def pre_scan_process(self):
        """
        gets the script path of the pre scan process

            Returns:
                (str)       --      script path
        """
        return self.management.pre_scan_process

    @pre_scan_process.setter
    def pre_scan_process(self, path):
        """
        sets the pre scan process.

            Args:
                 path   (str)      --   path of the pre scan script

            Returns:
                None
        """
        self.management.pre_scan_process = path
        if self.management.pre_scan_process != path:
            raise Exception('pre scan process script path = {0} has not updated successfully'.format(path))
        self.log.info('pre scan process = {0}'.format(path))

    @property
    def post_scan_process(self):
        """
        gets the script path of the post scan process

            Returns:
                (str)       --      script path
        """
        return self.management.post_scan_process

    @post_scan_process.setter
    def post_scan_process(self, path):
        """
         sets the post scan process.

            Args:
                 path   (str)      --   path of the post scan script

            Returns:
                None
        """
        self.management.post_scan_process = path
        if self.management.post_scan_process != path:
            raise Exception('post scan process script path = {0} has not updated successfully'.format(path))
        self.log.info('post scan process = {0}'.format(path))

    @property
    def pre_backup_process(self):
        """
        gets the script path of the pre backup process

            Returns:
                (str)       --      script path
        """
        return self.management.pre_backup_process

    @pre_backup_process.setter
    def pre_backup_process(self, path):
        """
         sets the pre backup process.

            Args:
                 path   (str)      --   path of the pre backup script

            Returns:
                None
        """
        self.management.pre_backup_process = path
        if self.management.pre_backup_process != path:
            raise Exception('pre backup process script path = {0} has not updated successfully'.format(path))
        self.log.info('pre backup process = {0}'.format(path))

    @property
    def post_backup_process(self):
        """
        gets the script path of the post backup process

            Returns:
                (str)       --      script path
        """
        return self.management.post_backup_process

    @post_backup_process.setter
    def post_backup_process(self, path):
        """
         sets the post backup process.

            Args:
                 path   (str)      --   path of the post backup script

            Returns:
                None
        """
        self.management.post_backup_process = path
        if self.management.post_backup_process != path:
            raise Exception('post backup process script path = {0} has not updated successfully'.format(path))
        self.log.info('post backup process = {0}'.format(path))

    @property
    def run_post_scan_process(self):
        """
        gets the value, run post scan process

            Returns:
                 True/False
        """
        return self.management.run_post_scan_process

    @run_post_scan_process.setter
    def run_post_scan_process(self, flag):
        """
        sets the value, run post scan process

            Args:
                 flag      (bool)   --      True/False

            Returns:
                None
        """
        self.management.run_post_scan_process = flag
        if self.management.run_post_scan_process != flag:
            raise Exception('run post scan process setting is not updated successfully')
        self.log.info('run post sacn process = {0}'.format(flag))

    @property
    def run_post_backup_process(self):
        """
         gets the value, run post backup process

            Returns:
                 True/False
        """
        return self.management.run_post_backup_process

    @run_post_backup_process.setter
    def run_post_backup_process(self, flag):
        """
        sets the value, run post backup process

            Args:
                 flag      (bool)   --      True/False

            Returns:
                None
        """
        self.management.run_post_backup_process = flag
        if self.management.run_post_backup_process != flag:
            raise Exception('run post backup process setting is not updated successfully')
        self.log.info('run post backup process = {0}'.format(flag))


class DRCommvaultCloudManagement(object):

    def __init__(self, commcell_object, company_name=None, source_commcell_id=None):
        """Initializes DRCommvaultCloudManagement object

            Args:
                commcell_object    (object)    --  instance of commcell

        """
        self.log = logger.get_log()
        self._commcell = commcell_object
        self._status_service = self._commcell._services['CVDRBACKUP_STATUS']
        self._info_service = self._commcell._services['CVDRBACKUP_INFO']
        self._download_service = self._commcell._services['CVDRBACKUP_DOWNLOAD']
        self._cvpysdk_object = self._commcell._cvpysdk_object
        self._company_name = company_name
        self._commcell_id = source_commcell_id
        self._companies = None
        self._company_names = []
        self.refresh()

    def refresh(self):
        """
        gets the latest set folder details from cvcloud

            Returns:
                  None
        """
        self._get_info()

    def _get_info(self):
        """
        gets the information of companies associated to cvcloud.

            Returns:
                 None

            Raises:
                Exception:
                    if failed to get the required details from cvcloud
        """
        if self._company_name and self._commcell_id:
            self._info_service = self._info_service + '?companyname=' + self._company_name \
                                 + '&commcellid=' + self._commcell_id
        elif self._company_name:
            self._info_service = self._info_service + '?companyname=' + self._company_name
        elif self._commcell_id:
            self._info_service = self._info_service + '?commcellid=' + self._commcell_id

        flag, response = self._cvpysdk_object.make_request(method='GET',
                                                           url=self._info_service)
        if flag:
            if response and response.json():
                self._companies = response.json().get('companies')
            else:
                raise Exception('Response received is empty')
        else:
            response_string = self._commcell._update_response_(response.text)
            raise Exception('Response was not success, error = {0}'.format(response_string))

    def get_company_names(self):
        """
        gets the company names.

            Returns:
                 (list)     --          list of company names
        """
        if self._companies:
            for company in self._companies:
                self._company_names.append(company.get('company_name'))
            return self._company_names
        else:
            raise Exception('Company details are not available')

    def get_commcell_ids_of_company(self, hardcheck=True):
        """
        gets the available commcell id's for a given company name.

            Args:
                company_name        (str)   --      name of the company

                hardcheck           (bool)  --      if company name not found and hardcheck = True
                    default:True                                    raises exception
                                                    if company name not found and hardcheck = False
                                                                    logs the error

            Returns:
                  (list)        --      list of commcell id's

            Raises:
                  Exception:
                    if company not found and hardcheck = True, raises exception

                    if company not found and hardcheck = False, logs the error

            NOTE: commcell ids will be in hexadecimal format(eg: XXXXXXX)

            NOTE: use setter company_name to set the company name
        """
        commcell_ids = []
        company_found = False

        if self._companies:
            if self._commcell_id:
                for company in self._companies:
                    if company.get('company_name') == self._company_name:
                        company_found = True
                        for commcell in company.get('commcells'):
                            commcell_ids.append(commcell.get('commcell_id'))
                        return commcell_ids
                if not company_found and hardcheck:
                    raise Exception('{0} company not found'.format(self._company_name))
                else:
                    self.log.info('{0} company not found'.format(self._company_name))
            else:
                raise Exception('Data type of the input(s) is not valid')
        else:
            raise Exception('Company details are not available')

    def get_number_of_set_folders_of_a_commcell(self, hardcheck=True):
        """
        gets the count of set folders assocaited to a commcell.

            Args:
                hardcheck           (bool)  --      if company name not found and hardcheck = True
                    default:True                                    raises exception
                                                    if company name not found and hardcheck = False
                                                                    logs the error

            Returns:
                (int)           --      count of the available set folders for a given commcell

            Raises:
                  Exception:
                    if company not found and hardcheck = True, raises exception

                    if company not found and hardcheck = False, logs the error

            NOTE: use setters company_name and source_commcell_id to set the company name and source commcell id
        """
        company_found = False
        commcel_id_found = False
        if self._companies:
            if self._company_name and self._commcell_id:
                for company in self._companies:
                    if company.get('company_name') == self._company_name:
                        company_found = True
                        for commcell in company.get('commcells'):
                            if commcell.get('commcell_id') == self._commcell_id:
                                commcel_id_found = True
                                return commcell.get('num_sets')
                if (not company_found or not commcel_id_found) and hardcheck:
                    raise Exception('one of the value is not found, {0} = {1}, {2} = {3}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found))
                else:
                    self.log.info('one of the value is not found, {0} = {1}, {2} = {3}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found))
            else:
                raise Exception('Data type of the input(s) is not valid')
        else:
            raise Exception('Company details are not available')

    def get_available_set_folders_of_a_commcell(self, hardcheck=True):
        """
        gets the names of available set folders for a given commcell_id.

            Args:
                hardcheck           (bool)  --      if company name not found and hardcheck = True
                    default:True                                    raises exception
                                                    if company name not found and hardcheck = False
                                                                    logs the error

            Returns:
                (int)           --      count of the available set folders for a given commcell

            Raises:
                  Exception:
                    if company not found and hardcheck = True, raises exception

                    if company not found and hardcheck = False, logs the error

            NOTE: use setters company_name and source_commcell_id to set the company name and source commcell id
        """
        set_folder_names = []
        company_found = False
        commcel_id_found = False
        if self._companies:
            if self._company_name and self._commcell_id:
                for company in self._companies:
                    if company.get('company_name') == self._company_name:
                        for commcell in company.get('commcells'):
                            if commcell.get('commcell_id') == self._commcell_id:
                                for folder in commcell.get('sets'):
                                    set_folder_names.append(folder.get('set_name'))
                                return set_folder_names
                if (not company_found or not commcel_id_found) and hardcheck:
                    raise Exception('one of the value is not found, {0} = {1}, {2} = {3}'.format(
                        self._company_name, company_found, self._company_name, commcel_id_found))
                else:
                    self.log.info('one of the value is not found, {0} = {1}, {2} = {3}'.format(
                        self._company_name, company_found, self._company_name, commcel_id_found))
            else:
                raise Exception('Data type of the input(s) is not valid')
        else:
            raise Exception('Company details are not available')

    def get_file_names_of_a_set_folder(self, set_folder_name, hardcheck=True):
        """
        gets file names of a set folder for a given commcell and company.

            Args:
                set_folder_name    (str)    --      name of the set folder

                hardcheck           (bool)  --      if company name not found and hardcheck = True
                    default:True                                    raises exception
                                                    if company name not found and hardcheck = False
                                                                    logs the error

            Returns:
                (list)           --      list of file names for a given set folder

            Raises:
                  Exception:
                    if company not found and hardcheck = True, raises exception

                    if company not found and hardcheck = False, logs the error

            NOTE: commcell id will be in hexadecimal format(eg: XXXXXXX)

            NOTE: use setters company_name and source_commcell_id to set the company name and source commcell id

            NOTE: set folder name format(eg: SET_123)
        """
        file_names = []
        company_found = False
        commcel_id_found = False
        set_folder_found = False
        if self._companies:
            if self._company_name and self._commcell_id and isinstance(set_folder_name, str):
                for company in self._companies:
                    if company.get('company_name') == self._company_name:
                        company_found = True
                        for commcell in company.get('commcells'):
                            if commcell.get('commcell_id') == self._commcell_id:
                                commcel_id_found = True
                                for folder in commcell.get('sets'):
                                    if folder.get('set_name') == set_folder_name:
                                        set_folder_found = True
                                        for file in folder.get('files'):
                                            file_names.append(file.get('file_name'))
                                        return file_names
                if (not company_found or not commcel_id_found or not set_folder_found) and hardcheck:
                    raise Exception('one of the value is not found, {0} = {1}, {2} = {3}, {4} = {5}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found, set_folder_name,
                        set_folder_found))
                else:
                    self.log.info('one of the value is not found, {0} = {1}, {2} = {3}, {4} = {5}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found, set_folder_name,
                        set_folder_found))
            else:
                raise Exception('Data type of the input(s) is not valid')
        else:
            raise Exception('Company details are not available')

    def get_set_id_of_a_set_folder(self, set_folder_name, hardcheck=True):
        """
        gets set id of a set folder for a given commcell and company.

            Args:
                company_name       (str)    --      name of the company

                commcell_id        (str)    --      hexadecimal id of a commcell

                set_folder_name    (str)    --      name of the set folder

                hardcheck           (bool)  --      if company not found and hardcheck = True
                    default:True                                    raises exception
                                                    if company not found and hardcheck = False
                                                                    logs the error

            Returns:
                (int)           --      set id of a set folder.

            Raises:
                  Exception:
                    if company not found and hardcheck = True, raises exception

                    if company not found and hardcheck = False, logs the error

            NOTE: use setters company_name and source_commcell_id to set the company name and source commcell id

            NOTE: set folder name format(eg: SET_123)
        """
        company_found = False
        commcel_id_found = False
        set_folder_found = False
        if self._companies:
            if self._company_name and self._commcell_id and isinstance(set_folder_name, str):
                for company in self._companies:
                    if company.get('company_name') == self._company_name:
                        company_found = True
                        for commcell in company.get('commcells'):
                            if commcell.get('commcell_id') == self._commcell_id:
                                commcel_id_found = True
                                for folder in commcell.get('sets'):
                                    if folder.get('set_name') == set_folder_name:
                                        set_folder_found = True
                                        return folder.get('set_id')
                if (not company_found or not commcel_id_found or not set_folder_found) and hardcheck:
                    raise Exception('one of the value is not found, {0} = {1}, {2} = {3}, {4} = {5}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found, set_folder_name,
                        set_folder_found))
                else:
                    self.log.info('one of the value is not found, {0} = {1}, {2} = {3}, {4} = {5}'.format(
                        self._company_name, company_found, self._commcell_id, commcel_id_found, set_folder_name,
                        set_folder_found))
            else:
                raise Exception('Data type of the input(s) is not valid')
        else:
            raise Exception('Company details are not available')

    @property
    def get_status_of_last_dr_backup_job(self):
        """
        returns the status of last dr backup

        Raises:
            Exception:
                when failed to get the required response

        Returns:
            (dict)  -   format as shown below
            {
                "DRbackupURL": "",
                "UploadStatus": int,
                "DRLastUploadedTime": ""
            }
        """
        if self._commcell_id:
            flag, response = self._cvpysdk_object.make_request(method='GET',
                                                               url=self._status_service % self._commcell_id)
        else:
            raise Exception('Please set the source commcell id using setter source_commcell_id')
        if flag:
            if response and response.json():
                return response.json()
            else:
                raise Exception('Response received is empty')
        else:
            response_string = self._commcell._update_response_(response.text)
            raise Exception('Response was not success, error = {0}'.format(response_string))

    def get_dump_data(self, set_id, file_name):
        """
        gets the data in binary format which can be written to a file.

            Args:
                set_id      (int)   -       set_id associated to set folder in cvcloud

                file_name   (str)   -       file to be downloaded from cvcloud

            Returns:
                response    (bytes) -       downloaded data

            Raises:
                Exception:
                    if failed to get the required response.

            NOTE: set id can be obtained with method get_set_id_of_a_set_folder()

            use download_from_cvcloud() in drhelper module for downloading SET folders to destination path.
        """
        url = self._download_service + '?setId={0}&fileName={1}'.format(set_id, file_name)
        flag, response = self._cvpysdk_object.make_request(method='GET', url=url)
        if flag:
            if response and response.json():
                file_url = response.json().get('fileURL', None)
                sas_token = response.json().get('SASToken', None)
                if not file_url or not sas_token:
                    raise Exception('Failed to get the required response')
                flag, response = self._cvpysdk_object.make_request(method='GET', url=file_url + '?' + sas_token,
                                                                   headers=[], stream=True)
                if flag:
                    if response:
                        return response
                    raise Exception('Response received is empty')
                else:
                    response_string = self._commcell._update_response_(response.text)
                    raise Exception('Response was not success, error = {0}'.format(response_string))
            else:
                raise Exception('Response received is empty')
        else:
            response_string = self._commcell._update_response_(response.text)
            raise Exception('Response was not success, error = {0}'.format(response_string))

    @property
    def company_name(self):
        """
        gets the name of the company

            Returns:
                (str)       --      name of the company
        """
        return self._company_name

    @company_name.setter
    def company_name(self, company_name):
        """
        sets the name of the company

            Args:
                company_name       (str)  --        name of the company
        """
        if isinstance(company_name, str):
            self._company_name = company_name
            self.refresh()
        else:
            raise Exception('company_name must be of type string')

    @property
    def source_commcell_id(self):
        """
        gets the hexadecimal id of the source commcell

            Returns:
                (str)       --      hexadecimal id of the commcell
        """
        return self._commcell_id

    @source_commcell_id.setter
    def source_commcell_id(self, commcell_id):
        """
         sets the hexadecimal id of the source commcell

            Args:
                commcell_id       (str)  --        hexadecimal id of the commcell
        """
        if isinstance(commcell_id, str):
            self._commcell_id = commcell_id
            self.refresh()
        else:
            raise Exception('commcell_id must of type string')

