# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing Commcell Migration operations

CCMHelper: Helper class to perform Commcell Migration operations

CCMHelper:
    __init__()                  --  initializes CCM helper object

    create_destination_commcell()  --  creates the destination commcell object

    get_active_mediaagent()     --  returns the online available media agent

    run_ccm_export()            --  start a CCM Export job

    create_entities()           --  Creates all the required entities for test case execution

    clean_entities()            --  Cleans up the created entities as part of this run.

    run_ccm_import()            --  start a CCM Import job

    get_latest_subclient()      --  Returns the latest subclient object for provided client

    get_jobs_for_subclient()    --  Returns list of jobs available for a provided subclient.

    set_libary_mapping()        --  Sets all the required library mapping settings post Migration

    get_barcode_list()          --  Returns barcode list for provided subclient

    restore_by_job()            --  Submits as restore by job with provided jobs of a backupset.

"""
from cvpysdk.commcell import Commcell
from AutomationUtils.options_selector import OptionsSelector, CVEntities
from AutomationUtils.database_helper import CommServDatabase, MSSQL
from Server.serverhelper import ServerTestCases


class CCMHelper(object):
    """Helper class to perform Commcell Migration related operations"""

    def __init__(self, testcase):
        """Initialize instance of the CCMHelper class."""
        self.testcase = testcase
        self._destination_cs = None
        self.server = ServerTestCases(self.testcase)
        self.log = testcase.log
        self.entities = CVEntities(self.testcase)
        self._csdb = None
        self._mssql = None
        self._option_selector = None

    def create_destination_commcell(self, destination_cs_name,
                                    cs_user_name, cs_password):
        """Returns the destination commcell object

        Args:

            destination_cs_name     (str)   --      destination commcell host name

            cs_user_name            (str)   --      destination commcell user name

            cs_password             (str)   --      destaintion commcell password

        """

        try:

            self._destination_cs = Commcell(destination_cs_name,
                                            cs_user_name,
                                            cs_password)
        except Exception as excp:

            self.log.error('destination commcell object creation failed : %s',
                           str(excp))
            raise Exception('destination commcell object creation failed : %s',
                            str(excp))

    @property
    def destination_cs(self):
        """Returns destination commcell object"""
        if self._destination_cs is None:
            raise Exception("Destination Commcell object is not initialized.")
        return self._destination_cs

    @property
    def option_selector(self):
        """Returns the option selector instance"""
        if self._option_selector is None:
            self._option_selector = OptionsSelector(self.testcase.commcell)
        return self._option_selector

    def get_active_mediaagent(self):
        """"Returns the active MediaAgent Instance"""

        utility = self.option_selector
        return utility.get_ma('windows')

    def run_ccm_export(self, export_location, client_name):
        """Return the CCM Ecport job instance

        Args:
            export_location     ( str )     -       location to perform CCM Export

            client_name             ( str )    -       list of clients to be exported

         Returns:
             CCM Export job instance

        """

        ccmobj = self.testcase.commcell.commcell_migration
        opt_dic = {'pathtype': 'Local'}

        return ccmobj.commcell_export(export_location,
                                      [client_name], opt_dic)

    def create_entities(self, mountpath_location=None, mountpath_user_name=None, mountpath_user_password=None,
                        tape_library=None):

        """Creates all the required entities for a client

        Args:
            mountpath_location          ( str )     -   UNC Location for the mount path

            mountpath_user_name         ( str )     -   UNC path user name

            mountpath_user_password     ( str )     -   UNC path password

            tape_library                ( str)      -   Tape library name

        Returns:

            returns all the entities created.

        """

        try:
            available_ma = self.get_active_mediaagent()

            all_inputs = {
                'target':
                    {
                        'client': self.testcase.client.client_name,
                        'agent': "File system",
                        'instance': "defaultinstancename",
                        'mediaagent': available_ma,
                        'force': True
                    },
                'backupset': None,
                'subclient': {'content': [self.testcase.client.log_directory], },
                'storagepolicy': None
            }

            if mountpath_location is not None:
                disk_lib = {'disklibrary': {
                    'mount_path': mountpath_location,
                    'username': mountpath_user_name,
                    'password': mountpath_user_password,
                    'cleanup_mount_path': True, }}

                all_inputs.update(disk_lib)

            elif tape_library is not None:
                query = ("select top 1 DrivePoolName from MMdrivepool where MasterPoolId"
                         " in(select MasterPoolId from MMMasterPool where libraryid in"
                         " (select LibraryId from MMLibrary where AliasName"
                         " ='{0}'".format(tape_library) + "))")

                result = self.option_selector.exec_commserv_query(query)
                drivepool_name = str(result[1][0][0])

                tape_lib = {'storagepolicy': {
                    'library': tape_library,
                    'mediaagent_name': self.testcase.commcell.commserv_name,
                    'drivepool': drivepool_name,
                    'scratchpool': 'Default Scratch',
                    'istape': True
                }}

                all_inputs.update(tape_lib)
            else:
                raise Exception("Unable to find required library")

            created_entities = self.entities.create(all_inputs)
            backupset = created_entities["subclient"]["backupset"]
            subclient_name = created_entities["subclient"]["name"]
            subclient = backupset.subclients.get(subclient_name)

            job = subclient.backup()
            self.log.info("started backup job for subclient: {%s"
                          "with job id: %s", subclient_name,
                          str(job.job_id))
            if job.wait_for_completion():
                self.log.info("Backup Job id: %s "
                              "completed successfully", str(job.job_id))
            else:
                self.log.error("Backup Job id: %s failed/ killed",
                               str(job.job_id))

        except Exception as excp:
            self.log.error('Entities creation failed : %s',
                           str(excp))
            raise Exception('Entities creation failed : %s',
                            str(excp))

    def clean_entities(self):
        """Clean the entities created in this run"""

        self.entities.cleanup()

    def run_ccm_import(self, import_location):
        """Return the CCM Import job instance

        Args:
            import_location     ( str )     -       location to perform CCM Import

         Returns:
             CCM Import job instance

        """

        opt_dic = {'pathtype': 'Local'}

        return self.destination_cs.commcell_migration.commcell_import(import_location,
                                                                      opt_dic)

    @property
    def csdb_destination(self):
        """Returns the commcell cs DB object"""
        try:
            if self._csdb is None:
                self._csdb = CommServDatabase(self.destination_cs)

            return self._csdb
        except Exception as excp:
            self.log.error(excp)
            raise Exception("csdb object creation failed",
                            str(excp))

    def get_latest_subclient(self, client_name=None, destination_commcell=False):
        """Return the latest subclient instance for a client

        Returns:
            latest subclient object for client

        """

        if destination_commcell:

            query = ("select top(1) app.subclientName,bs.name, c.name from APP_Application app,"
                     " APP_BackupSetName bs, APP_Client c where bs.id = app.backupSet"
                     " and app.clientId =")
            if client_name is None:
                query += "c.id order by app.refTime desc"
            else:
                client_id = self.destination_cs.clients[client_name]['id']
                query += client_id + "order by app.refTime desc"

            self.csdb_destination.execute(query)
            subclient_name = self.csdb_destination.fetch_all_rows()[0][0]
            backupset_name = self.csdb_destination.fetch_all_rows()[0][1]
            c_name = self.csdb_destination.fetch_all_rows()[0][2]
            agent = self.destination_cs.clients.get(c_name).agents.get("file system")
            backupset = agent.backupsets.get(backupset_name)
            return backupset.subclients.get(subclient_name)
        else:
            query = ("select top(1) app.subclientName,bs.name,"
                     " c.name from APP_Application app, APP_BackupSetName bs,"
                     " APP_Client c where bs.id = app.backupSet and app.clientId=c.id"
                     " and app.clientId =")
            if client_name is None:
                query += "c.id order by app.refTime desc"
            else:
                client_id = self.testcase.commcell.clients[client_name]['id']
                query += client_id + " order by app.refTime desc"

            result = self.option_selector.exec_commserv_query(query)
            subclient_name = str(result[1][0][0])
            backupset_name = str(result[1][0][1])
            c_name = str(result[1][0][2])
            agent = self.testcase.commcell.clients.get(c_name).agents.get("file system")
            backupset = agent.backupsets.get(backupset_name)
            return backupset.subclients.get(subclient_name), backupset

    def get_jobs_for_subclient(self, sub_client):
        """returns the jobid list for a provided subclient"""

        query = ("select distinct jobId from JMjobdatastats where dataType=2 and appid="
                 " {0}".format(sub_client.subclient_id))

        result = self.option_selector.exec_commserv_query(query)
        jobs_list = result[1][0]
        for i in range(0, len(jobs_list)):
            jobs_list[i] = int(jobs_list[i])
        return jobs_list

    @property
    def mssql_destination(self):
        """Returns MSSQL instance for a destination cs"""
        if self._mssql is None:
            sql_server =""
            if self.destination_cs.is_linux_commserv:
                sql_server = self.testcase.tcinputs["DestinationCSHostName"]
            else:
                sql_server = self.testcase.tcinputs["DestinationCSHostName"] + "\\commvault"
            self._mssql = MSSQL(sql_server,
                                self.testcase.tcinputs["SQLUserName"],
                                self.testcase.tcinputs["SQLPassword"],
                                "CommServ",
                                as_dict=False)
        return self._mssql

    def set_libary_mapping(self, source_commcell_name):
        """Sets all the required library mappings post CCM

        Args:

            source_commcell_name    (str)   --      sourcce commcell name

        """
        query = "Select id from App_Client where ID = (Select top 1 ClientID from MMHost where OfflineReason = 0)"
        self.csdb_destination.execute(query)
        ma_id = self.csdb_destination.fetch_all_rows()[0][0]
        query = "update CCMLibraryAnswers set TargetMediaAgentId={0} where SourceCommCellName='{1}'".format(
            ma_id, source_commcell_name)
        self.mssql_destination.execute(query)
        self.mssql_destination.execute_storedprocedure("CCMSetLibrariesAccessible", None)
        query = "UPDATE MMConfigs SET value=2 WHERE name= 'MMS2_CONFIG_STRING_MAGNETIC_CONFIG_UPDATE_INTERVAL_MIN';"
        self.mssql_destination.execute(query)
        dest_commcell_client = self.destination_cs.clients.get(self.destination_cs.commserv_name)
        dest_commcell_client.restart_service("GXMLM(" + dest_commcell_client.instance + ")")

    def get_barcode_list(self, subclient_id):
        """Returns barcode list for provided subclient

        Args:

            subclient_id            (int)   --      Subclient id of a subcient

        Returns:

            list of barcode's for a provided subclient id

        """

        query = ("select Mediaid from MMMedia where MediaId in ( select MediaId"
                 " from MMVolume where VolumeId in ( select volumeId from archChunk where id"
                 " in (select archChunkId from archChunkMapping where jobId in"
                 " (select jobId from JMJobDataStats where"
                 " appId = " + subclient_id + " )))) and MediaTypeId =8990")

        result = self.option_selector.exec_commserv_query(query)
        return result[1][0]

    def tape_import(self, media_list):
        """Run the tape import with the provided bar codes

        Args:

            media_list              (list)  --      list of barcodes needed to be imported

        """
        try:
            query = ("select DP.DrivePoolId, MP.LibraryId from MMDrivePool DP, MMMasterPool MP"
                     " where DP.MasterPoolId = MP.MasterPoolId and MP.LibraryId in ( select LibraryId from"
                     " MMMedia where MediaId = {0} )".format(media_list[0]))

            result = self.option_selector.exec_commserv_query(query)
            drive_pool_id = int(result[1][0][0])
            library_id = int(result[1][0][1])

            ccmobj = self.testcase.commcell.commcell_migration
            job = ccmobj.tape_import(library_id, media_list, drive_pool_id)

            self.log.info("started tape import job "
                          "with job id: %s", job.job_id)
            if job.wait_for_completion():
                if job.status != "Completed":
                    raise Exception("Tape import job not completed successfully")
                self.log.info("Tape Import Job id: %s "
                              "completed successfully", str(job.job_id))
            else:
                self.log.error("Tape Import Job id: %s failed/ killed",
                               str(job.job_id))

        except Exception as excp:
            self.log.error('Failed to start tape import with error : %s',
                           str(excp))
            raise Exception('Failed to start tape import with error : %s',
                            str(excp))

    def restore_by_job(self, backupset, jobs_list):
        """Run the restore by job for the provided backupset with specified jobs

        Args:
            backupset               (instance)      --      instance of the backupset class.

            jobs_list               (list)          --      list of the job available for restore.

        """
        try:
            restore_job = backupset.restore_out_of_place(client=self.testcase.tcinputs["ClientName"],
                                                         destination_path="c:\\Tape_Restore",
                                                         paths=[], fs_options={"index_free_restore": True},
                                                         restore_jobs=jobs_list)

            self.log.info("started Restore by job "
                          "with job id: %s", restore_job.job_id)
            if restore_job.wait_for_completion():
                self.log.info("Restore by Job id: %s "
                              "completed successfully", str(restore_job.job_id))
            else:
                self.log.error("Restore by Job id: %s failed/ killed",
                               str(restore_job.job_id))

        except Exception as excp:
            self.log.error('Failed to start Restore with error : %s',
                           str(excp))
            raise Exception('Failed to start Restore with error : %s',
                            str(excp))
