"""Utilities for the TestCase"""

from AutomationUtils import logger
from AutomationUtils.mail_box import MailBox
from AutomationUtils.mail_box import EmailSearchFilter
from cvpysdk.license import LicenseDetails
import os
from AutomationUtils.wrapper7z import Wrapper7Z
from Web.Common.exceptions import CVTestStepFailure
from AutomationUtils.config import get_config

_CONFIG = get_config()


class SendLogUtils:
    """Utilities for the TestCase"""

    def __init__(self, testcase, machine_object):
        """

        Args:
            testcase:
            machine_object: Object of machine where send logs bundle has to accessed
        """
        self.testcase = testcase
        self.machine_object = machine_object
        self.commcell = testcase.commcell
        licence = LicenseDetails(self.commcell)
        self.commcell_id = licence.commcell_id_hex
        self.commcell_name = self.commcell.commserv_name
        self.mail = None
        self.sysyteminfo_path = None
        self.log = logger.get_log()

    def verify_email(self, download_directory, subject=None):
        """
        Verifying email for sendLog job
        Args:
            download_directory: Email will get downloaded in given directory

        Returns:

        """
        try:
            self.mail = MailBox()
            self.mail.connect()
            if subject:
                email_filter = EmailSearchFilter(subject)
            else:
                email_filter = EmailSearchFilter(
                "CommCell ID " + self.commcell_id + " Logs for machines: " + self.commcell_name)
            if email_filter:
                self.mail.download_mails(search_filter=email_filter,
                                         download_dir=download_directory)
        except Exception as exp:
            raise CVTestStepFailure(
                f"Email verification get failed  [{exp}]"
            )
        finally:
            self.mail.disconnect()

    def is_send_log_bundle_exist(self, file_path, job_id):
        """
        verifies Content present at local drive location or not
        Args:
            file_path (str): Where send log bundle exist
            job_id  (str)  : for that you want to check send log bundle

        Returns:

        """
        self.log.info(f"Checking bundle exist in path {file_path}")
        file = self.machine_object.get_files_in_path(folder_path=file_path)
        temp = job_id + ".7z"
        for sendLog in file:
            if temp in sendLog:
                self.log.info("Send Log file present at " + sendLog)
                return sendLog
        raise CVTestStepFailure(
            f"Send log file not present for [{job_id}] at [{file_path}]"
        )


    def is_index_file_exists(self, file_path, request_id, client_name, backupset_guid):
        """
        verifies Content present at local drive location or not
        Args:
            file_path (str): Where send log bundle exist
            request_id  (str)  : requst id
            client_name (str) : client name
            backupset_guid (str) : backup set guid

        Returns:

        """

        file_path = os.path.join(file_path, str(request_id), client_name, "Index Cache",
                                 "CvIdxDB", backupset_guid, backupset_guid)
        self.log.info("Index file path  : " + file_path)
        files = self.machine_object.get_files_in_path(folder_path=file_path)
        temp = "ImageTable.dat"
        for file_name in files:
            if temp in file_name:
                self.log.info("Index is present : " + file_name)
                return file_name
        raise CVTestStepFailure(
            f"Index  not present for [{request_id}] at [{file_path}]"
        )

    def log_file_verification_with_full_name(self, file_path, file_names):
        """
        verifying in comm server log existence of cvd,evmgr,Job manager log,.........
        Args:
            file_path   (str): full path for file want to check
            file_names (list): Full Name of file that want to check

        Returns:

        """
        for file_name in file_names:
            if self.machine_object.check_file_exists(os.path.join(file_path, file_name)):
                self.log.info(f"[{file_name}] present in send log file in {file_path}")
            else:
                raise CVTestStepFailure(
                    f"[{file_name}] file not present at location:[{file_path}]"
                )

    def log_file_verification_with_partial_name(self, file_path, file_name):
        """
        In this case we don't have exact file name due to name change in every job
        Args:
            file_path (str): full path for file want to check
            file_name (str): File name  either full or partial name

        Returns:

        """

        file = self.machine_object.get_files_in_path(folder_path=file_path)
        for file_var in file:
            if file_name in file_var:
                self.log.info(f"[{file_name}] present at location [{file_path}]")
                return file_var

        raise CVTestStepFailure(
            f"{file_name} not present in SendLog files "
        )

    def verify_machine_logs_and_os_logs(self, client_object, file_path, full_file_name, partial_file_name):
        """
        Unzip and verify content by full name of file or given partial name of file
        Args:
            client_object           : Client object for machine where file exist
            file_path (str)         : full path for file that want to check
            full_file_name (list)   : full name of file
            partial_file_name (list): full or partial name of file

        Returns:

        """
        wrapper = Wrapper7Z(commcell=self.commcell, client=client_object, log=self.log, zipfilepath=file_path)
        wrapper.extract()
        file_path = file_path[:-3]
        self.log_file_verification_with_full_name(file_path, full_file_name)
        for file_name in partial_file_name:
            self.log_file_verification_with_partial_name(file_path, file_name)

    def unzip_send_log_file(self, client_obj, file_path=''):
        """

        Args:
            client_obj      : Client of Machine there send log bundle exist
            file_path (str) : Where send log bundle exist

        Returns:

        """
        self.log.info("Send Log content is wrapping .....")
        wrapper = Wrapper7Z(commcell=self.commcell, client=client_obj, log=self.log, zipfilepath=file_path)
        wrapper.extract()
        temp = file_path.find('.7z') - len(file_path)
        file_path = file_path[:int(temp)]
        return file_path

    def create_directory_for_given_path(self, directory_path):
        """

        Args:
            directory_path: folder name, don't give directory name like (C:\\)

        Returns:

        """
        if self.machine_object.create_directory(directory_name=os.path.join("C:\\", directory_path), force_create=True):
            return os.path.join("C:\\", directory_path)
        raise CVTestStepFailure(
            f" Failed to create folder name C:\\ {directory_path}"
        )
