# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file that executes the  Intellisnap basic acceptance test cases for nas client

Intellisnap BasicAcceptance is the only class defined in this file

This class include below cases:
•	Full snap backup with skip catalog
•	Add data and run Inc snap backup with skip catalog
•	Add data and run Diff snap backup with skip catalog
•	Add data and run Full snap backup with skip catalog
•	Add data and run Inc snap backup with skip catalog
•	Add data and run Diff snap backup with skip catalog
•	Mount snap & validate
•	Unmount snap & validate
•	Restore to unix proxy and validate
•	Inplace restore and validate
•	Backup copy
•	Restore to unix proxy from backupcopy and validate
•	Inplace restore from backupcopy and validate
•	Deferred cataloging on Storage policy
•	Restore to unix proxy from catalog and validate
•	Revert snap & validate
•	Delete snap & validate



BasicAcceptance:
     run()                   --  runs the basic acceptance test case
"""

from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from NAS.NASUtils.snapbasicacceptance import SnapBasicAcceptance



class NFSSnapBasicAcceptance(SnapBasicAcceptance):
    """Helper class to run Intellisnap basic acceptance test case for nas client"""

    def run(self):
        """Executes Intellisnap basic acceptance test case"""
        self._log.info(
            "Will run below test case on: %s subclient", format(str(self._inputs['SubclientName']))
        )
        self._log.info("Number of data readers: " + str(self._subclient.data_readers))
        if self._subclient.data_readers != 3:
            self._log.info("Setting the data readers count to 3")
            self._subclient.data_readers = 3
        self._log.info("Get NAS Client object")
        self.nas_client = self._nas_helper.get_nas_client(self._client, self._agent,
                                                          is_cluster=self._is_cluster)
        self.proxy = Machine(self._inputs['ProxyClient'], self._commcell)
        self.sccontent = self._inputs['SubclientContent'].split(",")
        ignore_files = self._nas_helper.ignore_files_list
        ignore_folder = self._nas_helper.ignore_files_list

        job = self._run_backup("FULL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        job = self._run_backup("INCREMENTAL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        job = self._run_backup("DIFFERENTIAL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        job = self._run_backup("FULL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        job = self._run_backup("INCREMENTAL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        job = self._run_backup("DIFFERENTIAL")
        self.snapjob = job
        if self._inputs.get('mount_path'):
            self.mount_path = self._inputs['mount_path']

        self.mount_snap(job.job_id, self.proxy.machine_name)
        self.mount_validation(job.job_id, self.proxy)
        self.unmount_snap(job.job_id)
        self.unmount_validation(job.job_id, self.proxy)

        self._log.info("*" * 10 + " Run out of place restore to Linux Client" + "*" * 10)
        options_selector = OptionsSelector(self._commcell)
        linux_restore_client, linux_restore_location = \
            options_selector.get_linux_restore_client()
        job = self._subclient.restore_out_of_place(
            linux_restore_client.machine_name,
            linux_restore_location,
            self.sccontent,
            restore_data_and_acl=False
            )
        self._log.info(
            "Started restore out of place to linux client job with Job ID: " + str(job.job_id)
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: " + str(job.delay_reason)
            )

        self._log.info("Successfully finished Restore out of place to linux client")
        destination_path = linux_restore_location +self.sccontent[0]
        out = linux_restore_client.compare_folders(linux_restore_client,
                                                   self.sccontent[0],
                                                   destination_path,
                                                   ignore_files,
                                                   ignore_folder)
        if out:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self._log.info("Successfully validated restored content")

        self._log.info("*" * 10 + " Run Restore in place " + "*" * 10)
        job = self._subclient.restore_in_place(self.sccontent,
                                               proxy_client=self.proxy.machine_name)

        self._log.info("Started restore in place job with Job ID: %s", format(str(job.job_id)))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore in place job with error: {0}".format(job.delay_reason)
            )
        self._log.info("Successfully finished restore in place job")
        out = linux_restore_client.compare_folders(linux_restore_client,
                                                   destination_path,
                                                   self.sccontent[0],
                                                   ignore_files,
                                                   ignore_folder)
        if out:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self._log.info("Successfully validated restored content")
        storage_policy_copy = "Primary"

        self._log.info("*" * 10 + "Running backup copy now" + "*" * 10)

        job = self._storage_policy.run_backup_copy()
        self._log.info("Backup copy workflow job id is : %s", format(job.job_id))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run backup copy job with error: " + str(job.delay_reason)
            )
        self._log.info("Successfully finished backup copy workflow Job :%s", format(job.job_id))

        if job.status != 'Completed':
            raise Exception(
                "job: {0} for Backup copy operation is completed with errors, \
                    Reason: {1}".format(job.job_id, job.delay_reason)
            )

        self._log.info("*" * 10 + "Run out of place restore from backupcopy" + "*" * 10)
        copy_precedence = self._get_copy_precedence(
            self._subclient.storage_policy, storage_policy_copy
        )
        linux_restore_client, linux_restore_location = \
            options_selector.get_linux_restore_client()
        job = self._subclient.restore_out_of_place(linux_restore_client.machine_name,
                                                   linux_restore_location,
                                                   self.sccontent,
                                                   restore_data_and_acl=False,
                                                   copy_precedence=int(copy_precedence))

        self._log.info(
            "Started Restore out of place job with Job ID: %d", job.job_id
        )
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: {0}".format(job.delay_reason)
            )
        self._log.info("Successfully finished Restore out of place from backup copy")
        destination_path = linux_restore_location +self.sccontent[0]
        out = linux_restore_client.compare_folders(linux_restore_client,
                                                   self.sccontent[0],
                                                   destination_path,
                                                   ignore_files,
                                                   ignore_folder)
        if out:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self._log.info("Successfully validated restored content")

        self._log.info("*" * 10 + " Run Restore in place restore from backup copy " + "*" * 10)
        job = self._subclient.restore_in_place(self.sccontent,
                                               copy_precedence=int(copy_precedence),
                                               proxy_client=self.proxy.machine_name)

        self._log.info("Started restore in place job with Job ID: %s", format(str(job.job_id)))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore in place job with error: {0}".format(job.delay_reason)
            )
        self._log.info("Successfully finished restore in place job from backup copy")
        out = linux_restore_client.compare_folders(linux_restore_client,
                                                   destination_path,
                                                   self.sccontent[0],
                                                   ignore_files,
                                                   ignore_folder)
        if out:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self._log.info("Successfully validated restored content")

        self._log.info("*" * 10 + " Run deferred cataloging on the storage policy  " + "*" * 10)
        self.snapshot_cataloging()
        self._log.info("*" * 10 + "Run out of place restore to proxy from deferred catalog" +
                       "*" * 10)
        storage_policy_copy = "Primary Snap"
        copy_precedence = self._get_copy_precedence(
            self._subclient.storage_policy, storage_policy_copy
        )
        linux_restore_client, linux_restore_location = \
            options_selector.get_linux_restore_client()
        job = self._subclient.restore_out_of_place(linux_restore_client.machine_name,
                                                   linux_restore_location,
                                                   self.sccontent,
                                                   restore_data_and_acl=False,
                                                   copy_precedence=int(copy_precedence))
        self._log.info(
            "Started Restore out of place job with Job ID: %d", job.job_id
        )
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error:{0}".format(job.delay_reason)
            )
        self._log.info("Successfully finished Restore out of place from deferred catalog")
        destination_path = linux_restore_location +self.sccontent[0]
        out = linux_restore_client.compare_folders(linux_restore_client,
                                                   self.sccontent[0],
                                                   destination_path,
                                                   ignore_files,
                                                   ignore_folder)
        if out:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )
        self._log.info("Successfully validated restored content")

        job = self._run_backup("FULL")
        for content in self.sccontent:
            self._nas_helper.copy_test_data_to_proxy(self.proxy, content)
        if(self._nas_helper.nas_vendor(self._client).upper())not in ['ISILON', 'HUAWEI']:
            self.revert_snap(job.job_id)
            self.revert_validation(job.job_id, self.proxy)
        self.delete_snap(self.snapjob.job_id)
        self.delete_validation(self.snapjob.job_id)
