#  -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing nas operations

NASHelper is the only class defined in this file

NASHeper: Helper class to perform nas operations

NASHeper:
    __init__()                                --  initializes nas helper object

    _get_client_credentials()                 --  returns the client credentials

    _get_cluster_client_info()                --  returns the cluster client info

    _get_test_data_path()                     --  returns test data path on controller machine

    ignore_files_list()                       --  returns the ignore files/patterns list

    copy_test_data()                          --  copies test data to destination client

    get_attribute_value_from_xml()            --  returns the attribute value

    get_snap_name_from_job()                  --  returns the snap name for specified job

    validate_if_smtape_backup()               --  validates if specified job was smtape backup

    validate_if_smtape_restore()              --  validates if specified job was smtape restore

    get_nas_client()                          --  returns NASClient object for specified filer

    validate_volume_status()                  --  validates if volume status is as specified

    validate_windows_restored_content()       -- Validates windows restored content with that of
    on filer

    validate_filer_restored_content()         -- Validates restored content on filer with that
    of on windows

    validate_filer_to_filer_restored_content()-- Validates restored content on filer with that of
    on another filer
"""

import shutil
import random
import string

from AutomationUtils import logger, cvhelper
from AutomationUtils.database_helper import get_csdb
from .nasclient import NetAPPClient, NetAPPClusterClient, IsilonClient, HuaweiClient
from .nasclient import UnityVnxCelerraClient, HNASClient


class NASHelper(object):
    """Helper class to perform nas operations"""

    def __init__(self, csdb=None):
        """Initializes nashelper object and gets the commserv database object if not specified

            Args:
                csdb    (object)    --  commserv database object using which we can
                                            query commserv db
                    default: None

        """
        self.log = logger.get_log()
        if csdb:
            self._csdb = csdb
        else:
            self._csdb = get_csdb()

        self._ignore_files_list = [
            '~snapshot', 'rstab*', 'restore_symbol*', 'RST_SH*', 'rst*', '.etc', 'lost+found',
            '$__NDMP__', '$__CFN__', '.snapshot', '.ckpt*'
        ]

    def _get_client_credentials(self, client_id, filer_type):
        """Returns the client credentials

            Args:
                client_id   (int)   --  id of the client for which the credentials
                                            are to be read from database

            Returns:
                (str, str)  -   username and password of the client

            Raises:
                Exception:
                    if failed to get client credentials

        """
        cur = []
        if filer_type != "":
            # Check if client credentials can be found in SMControlHost
            query = ("SELECT SMHostUserName, SMHostPassword, SMHostName FROM SMControlHost "
                     "WHERE ClientId = '{0}'").format(client_id)

            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            # validate credentials specified or not
            if len(cur) >= 3:
                if cur[1] != '3':
                    return cur[0], cur[1], cur[2]
            else:
                self.log.info("Client credentials not found in SMControlHost")
        elif len(cur) >= 1:
            # Check if client credentials can be found in MMNDMPHostInfo
            query = ("SELECT Login,Password FROM MMNDMPHostInfo WHERE "
                     "ClientId = '{0}'").format(str(client_id))
            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            query1 = "select net_hostname from APP_Client where id = '{0}'".format(str(client_id))
            self._csdb.execute(query1)
            cur1 = self._csdb.fetch_one_row()
            # validate credentials specified or not
            if len(cur) >= 2:
                if cur[1] != '3':
                    return cur[0], cur[1], cur1[0]
        else:
            raise Exception(
                "Client credentials are not specified. Please specify them from commcell GUI.")

    def _get_cluster_client_info(self, client_obj):
        """Returns cluster client name and id

            Args:
                client_obj      (object)    --  python sdk client object whose credentials
                                                    are to be read

            Returns:
                (str, str)  -   username and password of the client

            Raises:
                Exception:
                    if no cluster is selected for this client

                    if specified client is not cluster client

        """
        if 'nasClusterProperties' in client_obj._properties['pseudoClientInfo']:
            nas_cluster_prop = client_obj._properties['pseudoClientInfo']['nasClusterProperties']

            self.log.info("Check if the specified client is cluster/ vserver")
            if 'vServerName' not in nas_cluster_prop:
                self.log.info("This might be a cluster machine")
                return client_obj.client_name, client_obj.client_id

            self.log.info("This might be a vserver")
            if 'selectedCluster' in nas_cluster_prop:
                cluster_client_name = nas_cluster_prop['selectedCluster']['name']
                cluster_client_id = nas_cluster_prop['selectedCluster']['id']
                self.log.info("Selected cluster client name: %s", cluster_client_name)
                self.log.info("Selected cluster client id: %s", cluster_client_id)

                return cluster_client_name, cluster_client_id
            else:
                raise Exception("No cluster is selected for this client")
        else:
            raise Exception("Make sure the specified client is cluster client")

    def _get_test_data_path(self, local_machine, size):
        """Returns the test data path on this machine

            Args:
                local_machine   (object)    --  controller machine object which can be used to
                                                    perform operation on controller machine

                size            (int)       --  size of test data that is to be generated

            Returns:
                str     -   path where test data is generated

            Raises:
                Exception:
                    if failed to generate test data on controller machine
        """
        drives_dict = local_machine.get_storage_details()
        if local_machine.os_info == "WINDOWS":
            for drive in drives_dict.keys():
                if not isinstance(drives_dict[drive], dict):
                    continue

                if float(drives_dict[drive]['available']) >= size:
                    return drive + ":\\" +''.join(random.choices(string.ascii_uppercase, k=7))
        elif local_machine.os_info == "UNIX":
            if float(drives_dict['available']) >= size:
                return "/"+''.join(random.choices(string.ascii_uppercase, k=7))

        raise Exception("Failed to get test data path")

    @property
    def ignore_files_list(self):
        """Treats the ignore files list as read-only property"""
        return self._ignore_files_list

    def copy_test_data(self, nas_client, path):
        """Copies test data to specified path on nas client

            Args:
                nas_client  (object)    --  object for nas filer where test data is to be copied

                path        (str)       --  path where test data is to be copied

        """
        test_data_size = 10

        test_data_path = self._get_test_data_path(nas_client._local_machine, test_data_size)
        nas_client._local_machine.generate_test_data(test_data_path, file_size=test_data_size)

        self.log.info("Generated Test Data at path: " + test_data_path)

        nas_client.copy_folder(test_data_path, path)
        self.log.info("Copying test data to: " + path)

        shutil.rmtree(test_data_path)

    def copy_test_data_to_proxy(self, proxy, path):
        """Copies test data to specified path on unix client

            Args:
                Proxy  (object)    --  object for unix proxy client where test data is to be copied

                path   (str)       --  path where test data is to be copied

        """
        test_data_size = 10
        test_data_path = self._get_test_data_path(proxy, test_data_size)
        proxy.generate_test_data(
            test_data_path, file_size=test_data_size, hlinks=False, slinks=False)
        self.log.info("Generated Test Data at path: " + test_data_path)
        self.log.info("Folder %s will be copied to %s", test_data_path, path)
        proxy.copy_folder(test_data_path, path, '-f')
        self.log.info("Copying test data to: " + path)
        proxy.remove_directory(test_data_path)

    def get_snap_name_from_job(self, job_id):
        """Returns snap name for specified job

            Args:
                job_id  (int)   --  id of the job of which the snap name is to be determined

            Returns:
                str     -   name of the snap created during the specified job id

            Raises:
                Exception:
                    if failed to get snap name for the specified job id

        """
        query = "SELECT UniqueIdentifier FROM SMSnap WHERE SMSnapId = (SELECT SMSnapId FROM \
            SMVolSnapMap WHERE SMVolumeId = (SELECT SMVolumeId FROM SMVolume WHERE \
            JobId='{0}'))".format(job_id)

        self._csdb.execute(query)
        cur = self._csdb.fetch_one_row()
        if cur:
            return str(cur[0])
        else:
            raise Exception("Failed to get snap name for job id: {0}".format(job_id))

    def validate_if_smtape_backup(self, job_id):
        """Validates if performed job was smtape backup job

            Args:
                job_id      (int)   --  id of the backup job which is to be
                                            verified if was ran as smtape

            Raises:
                Exception:
                    if failed to validate if specified job was ran as smtape

        """
        self.log.info("Validating if %d job was SMTAPE backup job", job_id)
        query = "SELECT name,objName1 FROM archFile WHERE jobId = {0}".format(job_id)

        try:
            self._csdb.execute(query)
            rows = self._csdb.fetch_all_rows()
            self.log.info(str(rows))
            for row in rows:
                if not ("CV_BUTYPE=smtape" in str(row[1]) and "TYPE=smtape" in str(row[1])):
                    if str(row[1]) != "N/A":
                        raise Exception("This is not a SMTape job")

            self.log.info("Successfully validated SMTAPE backup job")
        except Exception as exp:
            raise Exception("Failed to validate if smtape job with error: " + str(exp))

    def validate_if_smtape_restore(self, job_id):
        """Validates if performed job was smtape restore

            Args:
                job_id      (int)   --  id of the restore job which is to be
                                            verified if was ran as smtape

            Raises:
                Exception:
                    if failed to validate if specified job was ran as smtape

        """
        self.log.info("Validating if %d job was SMTAPE restore", job_id)
        query = "select Message from EvLocaleMsgs where LocaleID = 0 and MessageID IN \
        (select messageId from evMsg where jobId_l = {0})".format(job_id)

        try:
            self._csdb.execute(query)
            rows = self._csdb.fetch_all_rows()
            self.log.info("Validating if SMTAPE Restore")
            self.log.info(str(rows))
            smtape_restore = False
            for row in rows:
                if "SMTAPE restore" in row[0]:
                    smtape_restore = True
                    break
            if not smtape_restore:
                raise Exception("This is not a SMTape  restorejob")

            self.log.info("Successfully validated SMTAPE restore job")
        except Exception as exp:
            raise Exception("Failed to validate if smtape restore job with error: " + str(exp))

    def get_nas_client(self, client_obj, agent_obj, filer_type=None, is_cluster=False):
        """Returns the NAS client object

            Args:
                client_obj  (object)    --  python sdk client object

                filer_type  (str)       --  type of the filer for which object is to be created

                is_cluster  (bool)      --  returns cluster client object if specified true

            Returns:
                object   -  nas client object for the specified client object

        """
        filer_type = self.nas_vendor(client_obj)
        agent_type = agent_obj.agent_name
        if filer_type.upper() == "NETAPP":
            if is_cluster:
                cluster_client_name, cluster_client_id = self._get_cluster_client_info(client_obj)
                user, password, _ = self._get_client_credentials(int(cluster_client_id),
                                                                 filer_type)
                return NetAPPClusterClient(
                    cluster_client_name, client_obj._commcell_object, agent_obj, user,
                    cvhelper.format_string(client_obj._commcell_object, password)
                )
            else:
                user, password, _ = (self._get_client_credentials(int(client_obj.client_id),
                                                                  filer_type))
                return NetAPPClient(
                    client_obj.client_name, client_obj._commcell_object, agent_obj, user,
                    cvhelper.format_string(client_obj._commcell_object, password)
                )
        elif filer_type.upper() == "ISILON":
            self.log.info("filer type is :%s", filer_type)
            user, password, controlhost = (self._get_client_credentials(int(client_obj.client_id),
                                                                        filer_type))
            return IsilonClient(client_obj.client_name, client_obj._commcell_object, agent_obj,
                                user, cvhelper.format_string(client_obj._commcell_object, \
                                                             password), controlhost)
        elif filer_type.upper() == "HUAWEI":
            user, password, controlhost = (self._get_client_credentials(int(client_obj.client_id),
                                                                        filer_type))
            return HuaweiClient(client_obj.client_name, client_obj._commcell_object, agent_obj,
                                user, cvhelper.format_string(client_obj._commcell_object, \
                                                             password), controlhost)
        elif filer_type.upper() == "DELL EMC VNX/CELERRA" or filer_type.upper() == "DELL EMC UNITY":
            user, password, controlhost = self._get_client_credentials(int(client_obj.client_id),
                                                                       filer_type)
            return UnityVnxCelerraClient(client_obj.client_name, client_obj._commcell_object,
                                         agent_obj, user, cvhelper.format_string
                                         (client_obj._commcell_object, password), controlhost)
        elif filer_type.upper() == "HNAS":
            self.log.info("filer type is :%s", filer_type)
            user, password, controlhost = self._get_client_credentials(int(client_obj.client_id),
                                                                       filer_type)
            return HNASClient(client_obj.client_name, client_obj._commcell_object, agent_obj,
                              user, cvhelper.format_string(client_obj._commcell_object, password),
                              controlhost
                              )



    def nas_vendor(self, client_obj):
        """Returns the vendor type of Client

            Args:
                client_obj  (object)    --  python sdk client object

            Returns:
                str   -  nas vendor type string for the specified client object

        """
        cur = []
        cur1 = []
        filer_type = None
        query = "SELECT OSType from MMNDMPHostInfo where \
            clientId ={0}".format(int(client_obj.client_id))
        self._csdb.execute(query)
        cur = self._csdb.fetch_one_row()
        query1 = "SELECT simOperatingSystemId from app_client \
           where id={0}".format(int(client_obj.client_id))
        self._csdb.execute(query1)
        cur1 = self._csdb.fetch_one_row()
        if cur[0]:
            if int(cur[0]) == 18:
                filer_type = "Isilon"
            elif int(cur[0]) == 2:
                filer_type = "Netapp"
            elif int(cur[0]) == 32:
                filer_type = "Huawei"
            elif int(cur[0]) == 1:
                filer_type = "Dell EMC VNX/Celerra"
            elif int(cur[0]) == 33:
                filer_type = "Dell EMC Unity"
            elif int(cur[0]) == 9:
                filer_type = "HNAS"
        elif cur1[0] and not cur[0]:
            if int(cur1[0]) == 16:
                filer_type = "Netapp"
        return filer_type

    def validate_volume_status(self, volume_obj, required_status):
        """Validates the volume status for specified volume

            Args:
                volume_obj      (object)    --  volume class object of the volume whose
                                                    status is to be verified

                required_status (str)       --  expected status of the specified volume object

            Raises:
                Exception:
                    if volume status is not as expected for specified volume object

        """
        self.log.info(
            "Validate if %s volume status is %s", volume_obj.name, required_status
        )

        if volume_obj.status.upper() != required_status.upper():
            raise Exception(
                "{0} volume status is not {1}".format(volume_obj.name, required_status)
            )

        self.log.info("Successfully validated volume status")

    def validate_windows_restored_content(self,
                                          nas_client,
                                          windows_restore_client,
                                          windows_restore_location,
                                          subclient_content,
                                          files=None):
        """Validates the windows restored content

            Args:
                nas_client                  (object)    --  nas client object on which the restored
                                                                content is to be verified

                windows_restore_client      (object)    --  machine class object for windows client
                                                                where the content was restored

                windows_restore_location    (str)       --  path on windows machine where content
                                                                was restored

                subclient_content           (list)      --  subclient content obtained from the
                                                                python sdk subclient object

            Raises:
                Exception:
                    if failed to validate restored content

        """
        self.log.info("Validate Restored content")
        self.log.info("ignoring list:" + str(self.ignore_files_list))
        diff = []
        for content in subclient_content:
            _, volume_name = nas_client.get_path_from_content(content, files=files)
            if files is None:
                restore_path = windows_restore_location + "\\" + volume_name
                diff += nas_client.compare_folders(windows_restore_client,
                                                   content,
                                                   restore_path,
                                                   ignore_files=self.ignore_files_list)
            else:
                restore_path = windows_restore_location + "\\" + volume_name[1]
                diff += nas_client.compare_files(windows_restore_client,
                                                 content,
                                                 restore_path,
                                                 files=files)

        if diff != []:
            self.log.error(
                "Restore validation failed. List of different files \n%s", diff
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self.log.info("Successfully validated restored content")

    def validate_filer_restored_content(self,
                                        nas_client,
                                        windows_restore_client,
                                        windows_restore_location,
                                        subclient_content,
                                        filer_restore_location=None,
                                        files=None):
        """Validates the restored content on different filer

            Args:
                nas_client                  (object)    --  nas client object on which the restored
                                                                content is to be verified

                windows_restore_client      (object)    --  machine class object for windows client
                                                                where the content was restored

                windows_restore_location    (str)       --  path on windows machine where content
                                                                was restored

                subclient_content           (list)      --  subclient content obtained from the
                                                                python sdk subclient object

                filer_restore_location      (str)       --  path on the filer where content is
                                                                restored
                        default: None

            Raises:
                Exception:
                    if failed to validate restored content

        """
        self.log.info("Validate Restored content")
        diff = []
        for content in subclient_content:
            _, volume_name = nas_client.get_path_from_content(content, files=files)
            if files is None:
                if filer_restore_location:
                    filer_destination_path = filer_restore_location + "/" + volume_name
                else:
                    filer_destination_path = content
                windows_restore_path = windows_restore_location + "\\" + volume_name
                diff += windows_restore_client.compare_folders(
                    nas_client, windows_restore_path, filer_destination_path,
                    ignore_files=self.ignore_files_list)
            else:
                if filer_restore_location:
                    filer_destination_path = filer_restore_location + "/" + volume_name[1]
                else:
                    filer_destination_path = content
                windows_restore_path = windows_restore_location + "\\" + volume_name[1]
                res = nas_client.compare_files(windows_restore_client,
                                               filer_destination_path,
                                               windows_restore_path,
                                               files=files)
                if res is False:
                    diff += [content]

        if diff != []:
            self.log.error(
                "Restore validation failed. List of different files \n%s", diff
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self.log.info("Successfully validated restored content")

    def validate_filer_to_filer_restored_content(self,
                                                 nas_client,
                                                 subclient_content,
                                                 filer_restore_location=None):
        """Validates the restored content on destination filer with that of source filer

            Args:
                nas_client                  (object)    --  nas client object on which the restored
                                                                content is to be verified

                subclient_content           (list)      --  subclient content obtained from the
                                                                python sdk subclient object

                filer_restore_location      (str)       --  path on the filer where content is
                                                                restored
                        default: None

            Raises:
                Exception:
                    if failed to validate restored content

        """
        self.log.info("Validate Restored content")
        diff = []
        for content in subclient_content:
            _, volume_name = nas_client.get_path_from_content(str(content))
            if filer_restore_location != '' and nas_client._agent.upper() == 'NDMP':
                filer_destination_path = "{0}/{1}".format(filer_restore_location, volume_name)
            elif filer_restore_location != '' and nas_client._agent.upper() == 'FILE SYSTEM':
                filer_destination_path = "{0}\{1}".format(filer_restore_location, volume_name)
            else:
                filer_destination_path = content
            diff += nas_client.compare_folders(nas_client,
                                               content,
                                               filer_destination_path,
                                               ignore_files=self.ignore_files_list)
        if diff != []:
            self.log.error(
                "Restore validation failed. List of different files \n%s", diff
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self.log.info("Successfully validated restored content")
