# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------


"""This file contains classes named DedupeHelper, CloudLibrary and MMHelper which consists of
 methods required for MM testacse assistance like DB queries, log parsing, etc.

Class DedupeHelper:
    parse_log()                     --  parsing log files in search of a particular pattern

    change_path()                   --  modifies given path to network path

    get_primary_objects()           --  Gets primary objects on a SIDB

    get_secondary_objects()         --  Gets secondary objects on a SIDB

    get_primary_objects_sec()       --  Gets primary objects of a secondary copy

    get_secondary_objects_sec()     --  Gets secondary objects on secondary copy

    get_sidb_ids()                  --  get SIDB store and substore id

    get_network_transfer_bytes()    --  gets bytes transfered for a job

    submit_backup_memdb_recon()     --  Launches full backup, kills SIDB to initiate a
                                        full DDB Recon

    poll_ddb_reconstruction()       --  Polls CSDB for recon job

    get_reconstruction_type()	    --  returns if Delta recon or Regular recon or Full recon

    execute_sidb_command()          --  executes given sidb2 command and returns back the
                                        output and status

    get_ddb_partition_ma()          --  Returns dictionary with partition number as key and
                                        client object of DDB MA as value

    get_sidb_dump()                 --  Dumps DDB table and returns the content of the dump as
                                        a string

    get_ddb_recon_details()         --  gets non-memDB reconstruction job details

    is_ddb_online()                 --  Checks if DDB status is online or offline

    get_db_sidb_ids()           	-- get SIDB store and substore ids for DB agent

    get_vm_sidb_ids()           	-- get SIDB store and substore ids for VSA agent

    get_running_sidb_processes()    -- Find out SIDB2 proecesses running on given DDB MA and return details
                                       about Engine Id, Partition ID, Process ID, Job ID

    is_sidb_running()               -- Check if sidb process for given engine id and partition id (optional)
                                       is running on DDB MA

    wait_till_sidb_down()           -- Periodically checks for SIDB process to shut down within given timeout period
                                       and returns when either SIDB process goes down or when timeout expires.

    setup_dedupe_environment()      -- get testcase object and setup library,dedupe storage policy,
                                        backupset and subclient

    configure_dedupe_storage_policy()   -- creates a new dedupe storage policy if not exists

    configure_global_dedupe_storage_policy()    --  creates a new global dedupe storage policy if not exists

    configure_dedupe_secondary_copy()   -- creates Synchronous copy for the storage policy

    get_primary_recs_count()        -- get latest count of total primary records on the store

    get_secondary_recs_count()      -- get latest count of total secondary records on the store

Class MMHelper:
    get_drive_pool_id()         -- get drivepool id

    get_spare_pool_id()         -- get spare pool id

    get_copy_id()               -- get storage policy copy id

    get_mount_path_id()         -- get mountPath id from mountpath name

    get_device_controller_id()  -- get device controller id from mountpath id and media agent id

    get_device_id()             -- get the device id for the given mountpath id

    get_media_location()        -- get media location id

    get_device_path()           -- get device path from mountpath id

    get_device_access_type()     -- get the device access type for the given mountpath id and mediaagent name

    remove_content_subclient()  -- Deletes subclient data

    get_archive_file_size()     -- Gets sum of archive file size on a copy/for a job

    set_opt_lan()               -- Modifies Optimize for concurrent LAN backups option on the MA

    get_global_param_value()    -- gets param value from gxglobalparam table

    get_jobs_picked_for_aux()   -- gets number of jobs picked by an auxcopy job

    get_to_be_copied_jobs()     -- gets number of jobs in to-be-copied state for a copy

    move_job_start_time()       -- moves job start and end time to number of days behind

    execute_update_query()      -- executes update query on CSDB

    cleanup()                   -- deletes backupset, storage policy

    validate_copy_prune()       -- verifies if a copy exits or deleted

    validate_job_prune()        -- Validates if a job is aged by checking table entries

    validate_jmdatastats()      -- Validate if a job id exists in table jmdatastats

    validate_archfile()         -- Validate if archfile entry for job id is exists

    validate_archfilecopy()     -- validate if archfilecopy entry for job id exists

    validate_archchunkmapping() -- Validate if archchunkmapping entry for job id exists

    validate_job_retentionflag()-- Validate if extended retention flag is set for a job id

    setup_environment()         -- get testcase object and setup library, non dedupe storage
                                   policy, backupset and subclient

    configure_disk_library()    -- Create a new disk library if not exists

    configure_disk_mount_path() -- Adds a mount path [local/remote] to the disk library

    configure_cloud_library()   -- Adds a new Cloud Library to the Commcell

    configure_cloud_mount_path()-- Adds a mount path to the cloud library

    configure_storage_policy()  -- creates a new storage policy if not exists

    configure_backupset()       -- Creates a new backupset if not exits

    configure_subclient()       -- Created a new subclient and added content

    configure_secondary_copy()  -- Creates a new secondary copy if doesnt exist

    create_uncompressable_data() -- Creates unique uncompressable data

    execute_select_query()      -- Executes CSDB select query

    unload_drive()              -- Unloads the drive on the library given

    remove_autocopy_schedule()  -- Removes association with System Created Automatic Auxcopy schedule on the given copy


CloudLibrary:
    __init__(libraryname, libraryinfo)             --  initialize the cloud library class
     instance for the commcell

    cleanup_entities(commcell, log, entity_config) --   static method to cleanup commcell entities.


Class PowerManagement:
	
	configure_cloud_mediaagent()	 -- Setup and configure cloud MediaAgent
	
    power_off_media_agents()         -- power-off a list of MediaAgents simultaneously

    power_on_media_agents()          -- power-on a list of MediaAgents simultaneously

	validate_powermgmtjobtovmmap_table () -- Validate the entry on MMPowerMgmtJobToVMMap table for the job and MediaAgent


"""

import os
import time
import re
import zipfile, threading
from cvpysdk.job import Job
from cvpysdk.client import Client
from AutomationUtils import logger
from AutomationUtils.machine import Machine
from AutomationUtils.database_helper import MSSQL
from AutomationUtils.idautils import CommonUtils
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents import mediaagentconstants

class DedupeHelper(object):
    """ Base class for deduplication helper functions"""
    def __init__(self, test_case_obj):
        """Initializes BasicOperations object"""
        self.tcinputs = test_case_obj.tcinputs
        self.commcell = test_case_obj.commcell
        self.client = getattr(test_case_obj, '_client')
        self._log = logger.get_log()
        self.comm_untils = CommonUtils(self)
        self.commserver_name = self.commcell.commserv_name
        self.id = test_case_obj.id
        self.log_dir = test_case_obj.log_dir
        self.csdb = test_case_obj.csdb
        self.agent = getattr(test_case_obj, '_agent')
        self.backupset_name = None
        self.subclient_name = None
        self.storage_policy_name = None
        self.library_name = None
        self.client_machine = None
        self.ddb_path = self.tcinputs.get("DedupeStorePath")
        if hasattr(test_case_obj, 'backupset_name'):
            self.backupset_name = test_case_obj.backupset_name
        if hasattr(test_case_obj, 'subclient_name'):
            self.subclient_name = test_case_obj.subclient_name
        if hasattr(test_case_obj, 'storage_policy_name'):
            self.storage_policy_name = test_case_obj.storage_policy_name
        if hasattr(test_case_obj, 'library_name'):
            self.library_name = test_case_obj.library_name
        if hasattr(test_case_obj, 'ddb_path') and self.ddb_path:
            self.ddb_path = test_case_obj.ddb_path

        try:
            self.backupset = test_case_obj.backupset
        except Exception:
            self.backupset = None
        try:
            self.subclient = test_case_obj.subclient
        except Exception:
            self.subclient = None
        self.mmhelper = MMHelper(test_case_obj)
        self.option_selector = OptionsSelector(self.commcell)
        if hasattr(test_case_obj, 'MediaAgentName'):
            self.machine = Machine(str(self.tcinputs["MediaAgentName"]), self.commcell)

    def setup_dedupe_environment(self):
        """
                get testcase object and setup library, dedupe storage policy, backupset and subclient

                Returns:
                    (object)    -- object of disk library
                    (object)    -- object of storage policy
                    (object)    -- object of backupset
                    (object)    -- object of subclient
        """
        disk_library = self.mmhelper.configure_disk_library()
        storage_policy = self.configure_dedupe_storage_policy()
        backupset = self.mmhelper.configure_backupset()
        subclient = self.mmhelper.configure_subclient()
        return disk_library, storage_policy, backupset, subclient

    def configure_dedupe_storage_policy(self,
                                        storage_policy_name=None,
                                        library_name=None,
                                        ma_name=None,
                                        ddb_path=None,
                                        ddb_ma_name=None):
        """
                creates a new dedupe storage policy if not exists
                Agrs:
                    storage_policy_name (str)   -- storage policy name to create

                    library_name (str)          -- library to use for creating storage policy

                    ma_name (str)               -- datapath MA name

                    ddb_ma_name (str)           -- MA name to create DDB store

                    ddb_path (str)              -- path to create DDB store

                Return:
                    (object)    -- storage policy object
        """
        # config storage policy
        if storage_policy_name is None:
            storage_policy_name = self.storage_policy_name
        if library_name is None:
            library_name = self.library_name

        if ma_name is None:
            if ddb_ma_name is not None:
                ma_name = ddb_ma_name
            else:
                ma_name = self.tcinputs.get("MediaAgentName")

        if ddb_ma_name is None:
            ddb_ma_name = self.tcinputs.get("DDBMediaAgentName")

        if ddb_path is None:
            ddb_path = self.ddb_path
        self._log.info("check SP: %s", storage_policy_name)
        if not self.commcell.storage_policies.has_policy(storage_policy_name):
            self._log.info("adding Storage policy...")
            self.commcell.storage_policies.add(storage_policy_name, library_name, ma_name,
                                               ddb_path + self.option_selector.get_custom_str(),
                                               dedup_media_agent=ddb_ma_name)
            self._log.info("Storage policy config done.")
        else:
            self._log.info("Storage policy exists!")
        storage_policy = self.commcell.storage_policies.get(storage_policy_name)
        return storage_policy

    def configure_global_dedupe_storage_policy(self, global_storage_policy_name, library_name, media_agent_name,
                                               ddb_path, ddb_media_agent):
        """
            creates a new global dedupe storage policy if not exists
                Args:
                    global_storage_policy_name (str)   --   global storage policy name to create

                    library_name (str)          --  library to use for creating storage policy

                    media_agent_name (str)   --  media_agent to be assigned

                    ddb_path (str)              -- path to create DDB store

                    ddb_media_agent         (str)   --  media agent name on which deduplication store
                                                    is to be hosted

                Return:
                    (object)    -- storage policy object
        """

        self._log.info("Check for GDSP : %s", global_storage_policy_name)
        if not self.commcell.storage_policies.has_policy(global_storage_policy_name):
            self._log.info("Adding a new GDSP: %s", global_storage_policy_name)
            gdsp = self.commcell.storage_policies.add_global_storage_policy(global_storage_policy_name, library_name,
                                                                            media_agent_name, ddb_path, ddb_media_agent)
            self._log.info("GDSP Configuration Done.")
            return gdsp
        else:
            self._log.info('GDSP already exists')
            return self.commcell.storage_policies.get(global_storage_policy_name)

    def configure_dedupe_secondary_copy(self, storage_policy, copy_name, library_name, media_agent_name, partition_path,
                                        ddb_media_agent, **kwargs):
        """Creates Synchronous copy for this storage policy

            Args:
                storage_policy          (object) --  instance of storage policy to add the copy to

                copy_name               (str)   --  copy name to create

                library_name            (str)   --  library name to be assigned

                media_agent_name        (str)   --  media_agent to be assigned

                partition_path          (str)   --  path where deduplication store is to be hosted

                ddb_media_agent         (str)   --  media agent name on which deduplication store
                                                    is to be hosted

            \*\*kwargs  (dict)  --  Optional arguments

                Available kwargs Options:

                dash_full               (bool)  --  enable DASH full on deduplication store (True/False)

                source_side_disk_cache  (bool)  -- enable source side disk cache (True/False)

                software_compression    (bool)  -- enable software compression (True/False)

            Return:
                    (object)    -- storage policy copy object
        """
        dash_full = kwargs.get('dash_full', None)
        source_side_disk_cache = kwargs.get('source_side_disk_cache', None)
        software_compression = kwargs.get('software_compression', None)
        self._log.info("Check Secondary Copy: %s", copy_name)
        if not storage_policy.has_copy(copy_name):
            self._log.info("Adding a new synchronous secondary copy: %s", copy_name)
            storage_policy.create_dedupe_secondary_copy(copy_name, library_name, media_agent_name,
                                                        partition_path,
                                                        ddb_media_agent, dash_full,
                                                        source_side_disk_cache,
                                                        software_compression)
            self._log.info("Secondary Copy has created")
        else:
            self._log.info("Storage PolicyCopy Exists!")
        return storage_policy.get_copy(copy_name)

    def parse_log(self,
                  client,
                  log_file,
                  regex,
                  jobid=None,
                  escape_regex=True,
                  single_file=False):
        """
        This function parses the log file in the specified location based on
        the given job id and pattern

        Args:
            client  (str)   --  MA/Client Name on which log is to be parsed

            log_file (str)  --  Name of log file to be parsed

            regex   (str)   --  Pattern to be searched for in the log file

            jobid   (str)   --  job id of the job within the pattern to be searched

            escape_regex (bool) -- Add escape characters in regular expression before actual comparison

            single_file (bool) -- to parse only the provided log file instead of all older logs

        Returns:
           (tuple) --  Result of string searched on all log files of a file name

        """
        found = 0
        matched_string = []
        matched_line = []
        self.client_machine = Machine(client, self.commcell)
        log_path = self.client_machine.client_object.log_directory
        if escape_regex:
            self._log.info("Escaping regular expression as escape_regex is True")
            regex = re.escape(regex)
        self._log.info("Log path : {0}".format(str(log_path)))
        if not single_file:
            all_log_files = self.client_machine.get_files_in_path(log_path)
            self._log.info("Got files in path ")
            log_files = [x for x in all_log_files if os.path.splitext(log_file)[0] in x]
        else:
            log_files = [self.client_machine.join_path(log_path, log_file)]

        if self.client_machine.os_info == 'UNIX':
            logfile_index = 0
            for log in log_files:
                if os.path.splitext(log)[1].lower() == '.bz2':
                    command = 'bzip2 -d %s'%(log)
                    self._log.info("decompressing .bz2 file %s", log)
                    exit, response, error = self.client_machine.client_object.execute_command(command)
                    if exit == 0:
                        self._log.info("Successfully decompressed log file %s",  log)
                        log_files[logfile_index] = log.replace(r'.bz2', '')
                    else:
                        self._log.error("Failed to decompress log file %s", log)
                logfile_index += 1

            # get log file versions
        for file in log_files:
            if found == 0:
                lines = []
                if os.path.splitext(file)[1].lower() not in ['.zip', '.bz2']:
                    lines = self.client_machine.read_file(file, search_term=regex).splitlines()
                if os.path.splitext(file)[1].lower() == '.zip':
                    log_dir = self.client_machine.join_path(self.client_machine.client_object.install_directory, 'Log Files')
                    base_dir = self.client_machine.join_path(self.client_machine.client_object.install_directory, 'Base')
                    command = '"%s%sunzip" -o "%s" -d "%s"' % (base_dir, self.client_machine.os_sep, file, log_dir)
                    response = self.client_machine.client_object.execute_command(command)
                    if response[0] == 0:
                        self._log.info('Decompressed log file %s', file)
                        extracted_file = os.path.splitext(file)[0]
                        lines = self.client_machine.read_file(extracted_file, search_term=regex).splitlines()
                    else:
                        self._log.error('Failed to Decompress log file %s', file)
                if not jobid and lines:
                    self._log.info("Searching for [{0} in file {1}]".format(regex, file))
                    for line in lines:
                        line = str(line)
                        regex_check = re.search(regex, line)
                        if regex_check:
                            matched_line.append(line)
                            matched_string.append(regex_check.group(0))
                            found = 1
                elif lines:
                    self._log.info("""Searching for string [{0}] for job [{1}] in file [{2}]
                                   """.format(regex, jobid, file))
                    for line in lines:
                        #required to change a byte stream to string
                        line = str(line)
                        jobid_check = re.search(" {0} ".format(str(jobid)), line)
                        if jobid_check:
                            regex_check = re.search(regex, line)
                            if regex_check:
                                matched_line.append(line)
                                matched_string.append(regex_check.group(0))
                                found = 1

        if found:
            count = len(matched_line)
            self._log.info("Found {0} matching line(s)".format(str(count)))
            return matched_line, matched_string
        self._log.error("Not found!")
        return None, None

    def change_path(self, file_path, machine_name):
        """
        change path to a UNC path

            Args:
                file_path (str)     --Path of file

                machine_name (str)  -- Name of mahine on which the file resides

            Returns:
                Network path of the file

        """
        drive = file_path[:1]
        path = file_path[3:]
        new_path = "\\\\{0}\\{1}$\\{2}".format(str(machine_name), str(drive), str(path))
        return new_path

    def get_primary_objects(self, jobid):
        """
        get primary records

            Args:
                jobid   (str)   --Job Id for which primary records to be checked

            Returns:
                Number of primary records added according to the CSDB

        """
        query = """select sum(primaryObjects) from archFileCopyDedup where archfileid in
                (select id from archFile where fileType =1 and jobId = {0})
                """.format(jobid)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(cur[0]))
        return cur[0]

    def get_secondary_objects(self, jobid):
        """
        get secondary objects

            Args:
                jobid (str)     --Job id for which secondary records to be fetched

            Returns:
                Number of secondary records in the DDB for this particular job

        """
        query = """select sum(secondaryObjects) from archFileCopyDedup where archfileid in
                 (select id from archFile where fileType =1 and jobId = {0})""".format(jobid)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur[0])))
        return cur[0]

    def get_primary_objects_sec(self, bkup_jobid, copy_name):
        """
        get primary records for secondary copy

            Args:
                bkup_jobid  (str)   --Job id of the initial backup

                copy_name   (str)   --Name of copy on which records are to be fetched

            Returns:
                Number of primary records on given copy for given job

        """
        query = ("""
                 select sum(primaryObjects) from archFileCopyDedup where archfileid in (select id
                 from archFile where fileType =1 and jobId = {0}) and archcopyid
                 in (select id from archgroupcopy where name =  '{1}')
                 """.format(str(bkup_jobid), str(copy_name)))
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur[0])))
        return cur[0]

    def get_secondary_objects_sec(self, bkup_jobid, copy_name):
        """
        get secondary records for secondary copy

            Args:
                bkup_jobid  (str)   --Job id of the initial backup

                copy_name   (str)   --Name of copy on which records are to be fetched

            Returns:
                Number of secondary records on given copy for given job

        """
        query = ("""
                 select sum(secondaryObjects) from archFileCopyDedup where archfileid in (select id
                 from archFile where fileType =1 and jobId = {0}) and archcopyid in (select id
                 from archgroupcopy where name = '{1}')
                 """.format(str(bkup_jobid), str(copy_name)))
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur[0])))
        return cur[0]

    def get_sidb_ids(self, sp_id, copy_name):
        """
        get SIDB store and substore ids

        Args:
            sp_id   (str)   --  Storage policy id

            copy_name   (str) -- Storage policy copy name

        Returns:
            Store ID associated with the storage policy

            SubStore ID associated with the storage policy copy

        """
        self._log.info("Getting SIDB ids for copy : {0}".format(copy_name))
        query = """
                select SS.SIDBStoreId, SS.SubStoreId
                from IdxSIDBSubStore SS inner join archgroupcopy AGC
                on SS.SIDBStoreId = AGC.SIDBStoreId
                and AGC.archGroupId = {0} and AGC.name = '{1}'
                """.format(str(sp_id), copy_name)
        self._log.info("QUERY : {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur)))
        return cur

    def get_db_sidb_ids(self, copy_id):
        """
        get SIDB store and substore ids for DB agent

        Args:
            copy_id   (str)   --  Storage policy copy id

        Returns:
            Store ID associated with the storage policy

            SubStore ID associated with the storage policy copy

        """
        self._log.info("Getting SIDB ids for copy id : {0}".format(copy_id))
        query = """
                select S.SIDBStoreId, SS.SubStoreId
                from IdxSIDBStore S,IdxSIDBSubStore Ss ,archCopySIDBStore ACS
                where  ACS.SIDBStoreId = S.SIDBStoreId 
                and S.SIDBStoreId = SS.SIDBStoreId
                and SS.SealedTime = 0
                and S.AppTypeGroupId = 1002
                and ACS.CopyId = {0}
                """.format(copy_id)
        self._log.info("QUERY : {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur)))
        return cur

    def get_vm_sidb_ids(self, copy_id):
        """
        get SIDB store and substore ids for VSA agent

        Args:
            copy_id   (str)   --  Storage policy copy id

        Returns:
            Store ID associated with the storage policy

            SubStore ID associated with the storage policy copy

        """
        self._log.info("Getting SIDB ids for copy id : {0}".format(copy_id))
        query = """
                select S.SIDBStoreId, SS.SubStoreId
                from IdxSIDBStore S,IdxSIDBSubStore Ss ,archCopySIDBStore ACS
                where  ACS.SIDBStoreId = S.SIDBStoreId 
                and S.SIDBStoreId = SS.SIDBStoreId
                and SS.SealedTime = 0
                and S.AppTypeGroupId = 1003
                and ACS.CopyId = {0}
                """.format(copy_id)
        self._log.info("QUERY : {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur)))
        return cur

    def get_network_transfer_bytes(self, jobid):
        """
        gets bytes transfered for a job

        Args:
            jobid (str) -- job ID

        Returns:
            network transferred bytes for a jobid

        """
        query = """select nwTransBytes/1024/1024 from JMBkpStats where jobId = {0}
                """.format(str(jobid))
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(str(cur[0])))
        return cur[0]

    def submit_backup_memdb_recon(self,
                                  subclient_object,
                                  sp_name,
                                  copy_name,
                                  time_out=5):
        """Launches full backup, kills SIDB to initiate a full DDB Recon

        Args:
            subclient_object (object)   -- subclient instance

            sp_name (str)               -- storage policy name

            copy_name (str)             -- storage policy copy name

            time_out (int)              -- Time out period to kill SIDB process
                Default : 5

        Returns:
            backup job ID

            reconstruction job ID

        """
        # get MA log location
        log_file = "SIDBEngine.log"
        bkp_job = subclient_object.backup("FULL")
        self._log.info("Backup job for delta recon: {0}".format(str(bkp_job.job_id)))
        if bkp_job:
            # parse log to find string: "Requested by Job-Id "+str(backupJobId)
            count = 0
            (matched_line, matched_string) = self.parse_log(self.tcinputs['MediaAgentName'],
                                                            log_file,
                                                            """Requested by Job-Id {0}""".format(
                                                                str(bkp_job.job_id)))
            while not matched_string:
                time.sleep(2)
                count += 1
                (matched_line, matched_string) = self.parse_log(
                    self.tcinputs['MediaAgentName'], log_file,
                    """Requested by Job-Id {0}""".format(str(bkp_job.job_id)))
                if count == 300:
                    self._log.error("SIDB2 process did not start in time")
                    return 0
            self._log.info("Going to kill SIDB2 prococess in {0} secs".format(str(time_out)))
            time.sleep(time_out)
            if self.machine.os_info == 'UNIX':
                kill_sidb_cmd = "kill -9 $(ps -e | grep 'sidb2' | awk '{print $1}')"
                exit, response, error = self.machine.client_object.execute_command(kill_sidb_cmd)
                if exit == 0:
                    self._log.info("SIDB Killed Successfully")
                else:
                    self._log.info("SIDB not killed")
            else:

                cmd_to_kill_sidb = """taskkill /F /IM SIDB2.exe /T /S {0} /U {1} /P {2}
                                   """.format(str(self.tcinputs["MediaAgentHostName"]),
                                              str(self.tcinputs["MediaAgentUsername"]),
                                              str(self.tcinputs["MediaAgentPassword"]))
                output = self.machine.execute_command(cmd_to_kill_sidb)
                if output._exit_code:
                    self._log.error("Failed to kill SIDB2 process")
                    raise Exception(
                        "Failed to kill SIDB2 process"
                    )
                else:
                    self._log.info("SIDB2 process killed")
        try:
            time.sleep(30)
            self._log.info("Attempting to resume backup job")
            bkp_job.resume()
        except Exception:
            pass

        recon_job = self.poll_ddb_reconstruction(sp_name, copy_name)

        try:
            time.sleep(30)
            self._log.info("Attempting to resume backup job")
            bkp_job.resume()
        except Exception:
            pass

        if not bkp_job.wait_for_completion():
            raise Exception(
                "Failed to run FULL backup with error:{0}".format(bkp_job.delay_reason)
            )
        self._log.info("Backup job completed.")
        return bkp_job, recon_job

    def poll_ddb_reconstruction(self, sp_name, copy_name, no_wait=False):
        """
        Polls whether DDB reconstruction jobs started and if started waits for
        completion and reports the status.

        Args:
            sp_name (str)   -- storage policy name
            copy_name (str) -- storage policy copy name
            no_wait (boolean)   -- option to wait for recon job to complete

        Returns:
            Reconstruction job ID

        """
        count = 0
        interval = 0
        self._log.info("Poll DDB Recon...")
        query1 = "select id from archGroup where name = '{0}'".format(str(sp_name))
        self.csdb.execute(query1)
        cur = self.csdb.fetch_one_row()
        sp_id = cur[0]

        copyId = self.mmhelper.get_copy_id(str(sp_name), str(copy_name))
        storeId = self.get_sidb_ids(str(sp_id), str(copy_name))[0]

        self._log.info("Check if given copy is a GDSP dependent copy")
        query4 = """SELECT dedupeFlags&134217728
                    FROM archGroupCopy
                    WHERE id = {0}""".format(str(copyId))
        self.csdb.execute(query4)
        # copy_id = copyId
        dedFlag = self.csdb.fetch_one_row()[0]
        if dedFlag == '134217728':
            self._log.info("Detected as a GDSP dependent copy")
            query5 = """
                    SELECT id
                    FROM archGroupCopy
                    WHERE SIDBStoreId = {0} AND dedupeFlags&268435456 = 268435456
                    """.format(str(storeId))
            self.csdb.execute(query5)
            copy_id = self.csdb.fetch_one_row()[0]
        else:
            copy_id = copyId

        while int(count) <= 0 and interval < 1800:
            time.sleep(10)
            interval = interval + 10
            _query = """select count(*) from JMAdminJobInfoTable where opType=80 and
                      archGrpCopyID = {0}""".format(str(copy_id))
            self._log.info("QUERY: {0}".format(_query))
            self.csdb.execute(_query)
            cur = self.csdb.fetch_one_row()
            count = cur[0]
            self._log.info("POLL: DDB reconstruction job: {0}".format(str(count)))
        if int(count) == 1:
            _query = """select jobId from JMAdminJobInfoTable where opType=80 and
                                  archGrpCopyID = {0}""".format(str(copy_id))
            self._log.info("QUERY: {0}".format(_query))
            self.csdb.execute(_query)
            cur = self.csdb.fetch_one_row()
            self._log.info("Recon Job: {0}".format(str(cur[0])))
            job_id = cur[0]

            recon_job = Job(self.commcell, job_id)
            if no_wait == False:
                if not recon_job.wait_for_completion():
                    raise Exception(
                        "Failed to run delta recon: {0}".format(recon_job.delay_reason)
                    )
                self._log.info("Recon job completed.")
            return recon_job
        raise Exception(
            "Reconstruction Job does not start in the 10 minutes wait time"
        )

    def get_reconstruction_type(self, recon_job_id):
        """
        returns if Delta recon or Regular recon or Full recon

        Args:
            recon_job_id (int/str)    -- reconstruction job ID

        Returns:
            Type of reconstruction job

        """
        recon_type = {
            1: 'Delta Reconstruction',
            2: 'Regular Reconstruction',
            4: 'Full Reconstruction'}
        _query = ("""
                  select flags
                  from IdxSIDBRecoveryHistory
                  where AdminJobId = {0}"""
                  .format(str(recon_job_id)))
        self._log.info("QUERY: {0}".format(_query))
        self.csdb.execute(_query)
        cur = self.csdb.fetch_one_row()
        recon_flag = int(cur[0])
        if recon_flag:
            if recon_flag == 1:
                self._log.info("Recon type: Delta Reconstruction")
                return recon_type[1]
            if recon_flag == 2:
                self._log.info("Recon type: Regular Reconstruction")
                return recon_type[2]
            if recon_flag == 4:
                self._log.info("Recon type: Full Reconstruction")
                return recon_type[4]
            raise Exception(
                "recon job type not found"
            )
        raise Exception(
            "no results returned - getReconstructionType"
        )

    def execute_sidb_command(self, operation, engineid, groupnumber, ddbmaobject):
        """Execute sidb2 command like compact or reindex to get output

        Args:
            operation (str)     -- sidb2 command option like compact or reindx or validate

            engineid (int)      -- sidbstore id

            groupnumber (int)   -- sidb partition number ( eg. single partition
                                    ddb has partition0 where as double partition ddb has
                                    partition0 and partition1)

            ddbmaobject (Client or String) -- Client object for DDB MA or Client Name

            instance (String)   --  Simpana instance
                                    Default : Instance001
        Returns:
            Output list - [returnstatus, output]
        """

        if isinstance(ddbmaobject, Client):
            ddbma_clientobj = ddbmaobject
        elif isinstance(ddbmaobject, str):
            ddbma_clientobj = self.commcell.clients.get(ddbmaobject)

        os_sep = '/'
        if ddbma_clientobj.os_info.lower().count('windows') > 0:
            os_sep = "\\"

        basedir = "{0}{1}Base{1}".format(ddbma_clientobj.install_directory, os_sep)
        sidb2cmd = "\"{0}sidb2\"".format(basedir)
        command = ""
        # If WIN MA, enclose in double quotes
        if ddbma_clientobj.os_info.lower().count('windows') > 0:
            command = "{0} -{5} -in {1} -cn {2} -i {3} -split {4}".format(
                sidb2cmd, ddbma_clientobj.instance, ddbma_clientobj.client_hostname,
                engineid, groupnumber, operation)
        if ddbma_clientobj.os_info.lower().count('linux') > 0:
            # If LINUX MA, use stdbuf -o0
            command = "stdbuf -o0 {0} -{5} -in {1} -cn {2} \
                        -i {3} -split {4}".format(sidb2cmd, ddbma_clientobj.instance,
                                                  ddbma_clientobj.client_hostname, engineid,
                                                  groupnumber, operation)

        self._log.info(command)
        output = ddbma_clientobj.execute_command(command)
        return output

    def get_ddb_partition_ma(self, sidbstoreid):
        """
        Return a dictionary having mapping of group number to ddb ma
        Args:
         sidbstoreid: sidb store id

        Return:
            Dictionary containing group number ==> DDB MA client object
        """
        query = """select idxsidbsubstore.GroupNumber,APP_Client.name from
        idxsidbsubstore inner join APP_Client
        on idxsidbsubstore.ClientId = APP_Client.id
        where sidbstoreid = {0}
        order by IdxSIDBSubStore.GroupNumber""".format(sidbstoreid)

        self._log.info("Firing query to find DDB MAs for given SIDB store ==> {0}".format(query))
        self.csdb.execute(query)

        ddbmalist = self.csdb.fetch_all_rows()
        ddbma_partition_dict = {}
        for ddbma in ddbmalist:
            self._log.info("Creating client object for DDB MA ==> {0}".format(ddbma[1]))
            ddbma_partition_dict[ddbma[0]] = self.commcell.clients.get(ddbma[1])
        return ddbma_partition_dict

    def get_sidb_dump(self, client_name, type, store_id, dump_path, split=0):
        """Dumps the SIDB and returns the table as string

        Args:
                client_name --  (str)   --  name of the client on which the ddb is located

                type        --  (str)   --  type of table
                                            (Primary/Secondary)

                store_id    --  (str)   -- DDB Store ID

                dump_path   --  (str)   --  Path where the sidb can be dumped

                split       --  (int)   --  Split number of DDB

        Returns:
                file content of the dump as a string

        """
        machine_obj = Machine(client_name, self.commcell)
        if machine_obj.os_info == 'WINDOWS':
            command = "& '{0}{1}Base{1}SIDB2.exe' -dump {2} -i {3} -split {4} {5}".format(
                machine_obj.client_object.install_directory, machine_obj.os_sep, type, store_id,
                split,
                dump_path)
            output = machine_obj.execute_command(command)
        elif machine_obj.os_info == 'UNIX':
            command = "(cd {0}{1}Base ; ./sidb2 -dump {2} -i {3} -split {4} {5})".format(
                machine_obj.client_object.install_directory, machine_obj.os_sep, type, store_id,
                split,
                dump_path)
            output = machine_obj.client_object.execute_command(command)

        time.sleep(60)

        dump = machine_obj.read_file(dump_path)
        return dump

    def get_ddb_recon_details(self, sp_id, wait_for_complete=True):
        """gets non-memDB reconstruction job details
        Args:
           sp_id                --  (str)   --  Storage policy ID

           wait_for_complete    --  (bool)  -- wait for job to complete
           Default : True

        Returns:
            Reconstruction job id

        """
        query = """select jobid 
                from jmadminjobinfotable 
                where optype = 80 
                and archGrpID = {0}
                """.format(sp_id)
        self.csdb.execute(query)
        recon_job_id = self.csdb.fetch_all_rows()[0][0]
        self._log.info("Started recon job : {0}".format(str(recon_job_id)))
        recon_job = self.commcell.job_controller.get(recon_job_id)
        if wait_for_complete:
            time.sleep(120)
            self._log.info("waiting for recon job to complete")
            if not recon_job.wait_for_completion():
                raise Exception(
                    "Failed to run {0}  with error: {1}".format("recon job", recon_job.delay_reason)
                )
            self._log.info("Recon job completed.")
        return recon_job_id

    def is_ddb_online(self, store_id, sub_store_id):
        """Checks if DDB status is online or offline
        Args:
           store_id         --  (str)   --  deduplication store id

           sub_store_id     --  (str)   --  deduplication substore id

        Returns:
            (int)   --  deduplication store status (0 - online, 1 - offline)

        """
        self._log.info("checking if DDB online")
        query = """SELECT status
                          FROM idxSidbSubStore
                          WHERE SIDBStoreId =  {0}
                          AND SubStoreId = {1}
                          """.format(store_id, sub_store_id)
        db_response = self.mmhelper.execute_select_query(query)
        return int(db_response[0][0])

    def get_running_sidb_processes(self, ddbma_object):
        """
        Find out SIDB2 proecesses running on given DDB MA and return details about each
        Engine ID
        Partition ID ( Group Number )
        Process ID
        Job ID

        Args:
            ddbmaobject     --  (client/str)    -- Client object or Client Name for DDB MA

        Returns:
            Output dictionary - {sidb2engineid:(groupnumber,pid,jobid),sidb2engineid:(groupnumber,pid, jobid)..}
        """
        if isinstance(ddbma_object, Client):
            ddbma_clientobj = ddbma_object
        elif isinstance(ddbma_object, str):
            ddbma_clientobj = self.commcell.clients.get(ddbma_object)
        is_windows = False
        # If its a windows machine, make a powershell command else a linux cli
        if ddbma_clientobj.os_info.lower().count('windows') > 0:
            command = "powershell.exe -command \"{0} {1}|{2}\"".format(
                "Get-WmiObject Win32_Process -Filter", "\\\"name = 'SIDB2.EXE'\\\"",
                " format-table -autosize ProcessId, CommandLine | out-string -width 200")
            is_windows = True
        else:
            command = "ps -eo pid,command | grep sidb2 | grep -v grep"

        self._log.info(command)
        output = ddbma_clientobj.execute_command(command)
        self._log.info(output[1])

        engine_partition_map = {}

        if output[0] != 0 or output[1].lower().count('sidb2') == 0:
            self._log.error("Failed to find any SIDB2 process")
            # return empty dictionary
            return engine_partition_map
        # output from windows & linux is in this format respectively, start parsing it
        """
        ProcessId CommandLine                                                                                                                
        --------- -----------                                                                                                                
        12164 "C:\Program Files\Commvault\ContentStore\Base\SIDB2.exe" -j 1714825 -i 808 -c 615 -in Instance001 
        -cn sbhidesp13cs -group 0
        14252 "C:\Program Files\Commvault\ContentStore\Base\SIDB2.exe" -j 1714829 -i 856 -c 722 -in Instance001
         -cn sbhidesp13cs -group 0

        [root@sbhidesp13lima1 /]# ps -eo pid,command | grep sidb2 | grep -v grep
        15348 /opt/commvault/Base/sidb2 -j 1714828 -i 7 -c 9 -in Instance001 -cn sbhidesp13lima1 -group 0
        15350 /opt/commvault/Base/sidb2 -j 1714828 -i 7 -c 9 -in Instance001 -cn sbhidesp13lima1 -group 1

        """

        # Create a datastructure which will be a dictionary of following type
        # key = engineid [as it will be unique ]
        # value = list of (group number, pid) tuples[ same engine can have multiple partitions running ]
        engine_regex = re.compile('-i \d+')
        groupid_regex = re.compile('-group \d')
        jobid_regex = re.compile('-j \d+')
        if is_windows is True:
            output_lines = output[1].split('\r\n')
        else:
            output_lines = output[1].split('\n')

            # start building dictionary from output
        for line in output_lines:
            line = line.strip()
            if line == '' or line.count('ProcessId') > 0 or line.count('CommandLine') > 0 or line.count('-----') > 0:
                # skip blank lines
                continue
            else:
                # extract process ID and command line
                pid = line.split()[0].strip()
                engine_id = engine_regex.findall(line)[0].split()[1]
                group_id = groupid_regex.findall(line)[0].split()[1]
                job_id = jobid_regex.findall(line)[0].split()[1]
                groupid_pid_tuple = (int(group_id), int(pid), int(job_id))
                if engine_id in engine_partition_map:
                    self._log.info("Appending groupid->partition->jobid {0} to engine id {1} ".format(
                        groupid_pid_tuple, engine_id))
                    engine_partition_map[engine_id].append(groupid_pid_tuple)
                else:
                    engine_partition_map[engine_id] = [groupid_pid_tuple]
                    self._log.info("Adding new engine id - partition pair " \
                                   "to map ==> {0} - {1}".format(engine_id, groupid_pid_tuple))

        return engine_partition_map

    def is_sidb_running(self, engine_id, ddbma_object, partition_number=-1):
        """
        Check if sidb process for given engine id and partition id (optional) is running on DDB MA

        Args:
            engine_id           --  (int)           --  SIDB Store ID
            ddbma_object        --  (client/str)    --  Client object or Client name for DDB MA
            partition_number    --  (int)           --  Group number (0,1 etc.) [ Optional ]
                                                        Default : Skip checking Partition Number and report
                                                        if SIDB for given engine ID is active  on the DDB MA

        Returns:
            List of (groupid,pid,jobid) tuples if SIDB process is running , empty list otherwise
        """
        self._log.info("Checking if sidb process for engine : {0} for partition :{1} is running" \
                       " on DDB MA {2}".format(engine_id, partition_number, ddbma_object.client_name))

        ddb_map = self.get_running_sidb_processes(ddbma_object)
        # If ddb_map is empty, no sidb process is running on MA
        if ddb_map == {}:
            return []
        else:
            # Check if key with given engine id exists
            try:
                group_pid_tuple_list = ddb_map[engine_id]
                self._log.info("Process id corresponding to engine id : {0} ==> {1}".format(engine_id,
                                                                                            group_pid_tuple_list))
                if (partition_number == -1):
                    # Immediately return True as user has not asked to validate a specific group number
                    return group_pid_tuple_list
                else:
                    # Loop over tuples list to see if user provided group number is present
                    for item in group_pid_tuple_list:
                        if item[0] != int(partition_number):
                            continue
                        else:
                            self._log.info("Found partition {0} running with pid {1}".format(partition_number,
                                                                                             item[1]))
                            return [item]

                    return []
            except:
                self._log.info("No process corresponding to engine id : {0} found".format(engine_id))
                return []

    def wait_till_sidb_down(self, engine_id, ddbma_object, partition_number=-1, timeout=600):
        """
        Periodically checks for SIDB process to shut down within given timeout period and returns when
        either SIDB process goes down or when timeout expires.

        Periodicity with which SIDB process status gets checked is 30 seconds.

        Args:
            engine_id                   --  (int)           --  SIDB Store ID
            ddbma_object                --  (client/str)    --  Client object or Client name for DDB MA
            partition_number            --  (int)           --  Group number (0,1 etc.) [ Optional ]
                                                                Default : Skip checking Partition Number and report
                                                                if SIDB for given engine ID is active  on the DDB MA
            timeout                     --  (int)           --  Maximum wait time in seconds [ Optional ]
                                                                Default : 600 seconds

        Returns:
            (Boolean) True if SIDB process stops running , False if it keeps running even after timeout
        """
        time_step = 30
        time_elapsed = 0
        while time_elapsed < timeout:
            is_running = self.is_sidb_running(engine_id, ddbma_object, partition_number)
            if is_running != []:
                self._log.info("SIDB Process still running ==> {0}".format(is_running))
                time.sleep(time_step)
                time_elapsed = time_elapsed + time_step
            else:
                self._log.info("SIDB Process doesn't seem to be running ==> {0}".format(is_running))
                return True

        # As we are here, it means we spent entire timeout period in waiting for SIDB to go down
        # Return False
        self._log.info("SIDB Process did not go down within given timeout period")
        return False

    def get_primary_recs_count(self, store_id, db_password, db_user=None):
        """
        get latest count of total primary records on the store
        Args:
            store_id (int): SIDB store ID or Engine ID to get primary count
            db_password (str): CSDB password to use to run query to get primary count
            db_user (str):  CSDB username to login

        Returns:
            (int) total primary records count on the store id
        """
        query = """with x as (
                select row_number() over (partition by substoreid order by modifiedtime desc) as y,*
                from idxsidbusagehistory where historytype = 0)
                select sum(x.primaryentries)
                from x where y = 1
                    and x.sidbstoreid = {0}
                group by x.sidbstoreid""".format(store_id)
        self._log.info("QUERY: {0}".format(query))
        result = self.mmhelper.execute_update_query(query, db_password=db_password, db_user=db_user)
        self._log.info("RESULT: {0}".format(result.rows[0][0]))
        return result.rows[0][0]

    def get_secondary_recs_count(self, store_id, db_password):
        """
        get latest count of total secondary records on the store
        Args:
            store_id (int): SIDB store ID or Engine ID to get secondary count
            db_password (str): CSDB password to use to run query to get secondary count

        Returns:
            (int) total secondary records count on the store id
        """
        query = """with x as (
                select row_number() over (partition by substoreid order by modifiedtime desc) as y,*
                from idxsidbusagehistory where historytype = 0)
                select sum(x.secondaryentries)
                from x where y = 1
                    and x.sidbstoreid = {0}
                group by x.sidbstoreid""".format(store_id)
        self._log.info("QUERY: {0}".format(query))
        result = self.mmhelper.execute_update_query(query, db_password)
        self._log.info("RESULT: {0}".format(result.rows[0][0]))
        return result.rows[0][0]


class MMHelper(object):
    """ Base clasee for media management helper functions"""
    def __init__(self, test_case_obj):
        """Initializes BasicOperations object

        Args:
            test_case_obj (object)  --  instance of the CVtestcase class

        """
        self.tcinputs = test_case_obj.tcinputs
        self.commcell = test_case_obj.commcell
        self.client = getattr(test_case_obj, '_client', None)

        self._log = logger.get_log()
        self.commserver_name = self.commcell.commserv_name
        self.id = test_case_obj.id
        self.csdb = test_case_obj.csdb
        self.is_windows_cs = True
        self.commserve_instance = "commvault"
        self.instance_sep = "\\"
        self.agent = getattr(test_case_obj, '_agent')

        self.backupset_name = None
        self.subclient_name = None
        self.storage_policy_name = None
        self.library_name = None
        if hasattr(test_case_obj, 'backupset_name'):
            self.backupset_name = test_case_obj.backupset_name
        if hasattr(test_case_obj, 'subclient_name'):
            self.subclient_name = test_case_obj.subclient_name
        if hasattr(test_case_obj, 'storage_policy_name'):
            self.storage_policy_name = test_case_obj.storage_policy_name
        if hasattr(test_case_obj, 'library_name'):
            self.library_name = test_case_obj.library_name

        self.content_path = self.tcinputs.get("ContentPath")

        if hasattr(test_case_obj, 'test_data_path') and self.content_path:
            self.content_path = test_case_obj.test_data_path

        if self.commcell.commserv_client.os_info.lower().count('unix'):
            self.is_windows_cs = False
            self.commserve_instance = ""
            self.instance_sep = ""

        if 'SqlSaPassword' in self.tcinputs:
            dbpassword = self.tcinputs['SqlSaPassword']
            self.dbobject = MSSQL(self.commcell.commserv_hostname + self.instance_sep +
                                  self.commserve_instance,
                                  "sa", dbpassword, "CommServ", False, True)

    def get_drive_pool_id(self, tape_id):
        """
        Get drivepool id

        Args:
            tape_id   (str)   --  Tape library ID

        Returns:
            DrivePool id of the given Tape library

        """
        self.csdb.execute(
            """select *
            from MMDrivePool
            where MasterPoolId =
            (select MasterPoolId
            from MMMasterPool
            where LibraryId = {0})
            """.format(str(tape_id)))
        drive_pool_id = str(self.csdb.fetch_one_row()[0])
        return drive_pool_id

    def get_spare_pool_id(self, tape_id):
        """
        get sparepool id

        Args:
            tape_id   (str)   --  Tape library ID

        Returns:
                SparePool id of the given Tape library

        """
        self.csdb.execute(
            """select *
            from MMSpareGroup
            where LibraryId = {0}
            """.format(str(tape_id)))
        spare_pool_id = str(self.csdb.fetch_one_row()[0])
        return spare_pool_id

    def get_copy_id(self, sp_name, copy_name):
        """
        get copy id

         Args:
            sp_name     (str) --  Storage policy name

            copy_name   (str) -- Storage policy copy name

        Returns:
                Copy id for the given storage policy/copy

        """
        self.csdb.execute(
            """ SELECT AGC.id
                FROM archgroup AG
                    INNER JOIN archgroupcopy AGC ON AG.id = AGC.archgroupid
                WHERE AG.name ='{0}' AND AGC.name = '{1}'
            """.format(sp_name, copy_name))
        copy_id = self.csdb.fetch_one_row()
        return copy_id[0]

    def get_mount_path_id(self, mountpath_name):
        """
        Get MountPathId from MountPathName

        Agrs:
            mountpath_name (str)  --  Mountpath name

        Returns:
            Mountpath Id for the given Mountpath name
        """

        query = """ SELECT	MountPathId
                    FROM	MMMountPath 
                    WHERE	MountPathName = '{0}'""".format(mountpath_name)
        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != ['']:
            value = int(cur[0])
            return value
        self._log.error("No entries present")
        raise Exception("Invalid MountPathName")

    def get_device_controller_id(self, mountpath_id, media_agent_id):
        """
        Get device controller id from mountpath id and media agent id
        Args:
            mountpath_id (int)  --  mountpath Id

            media_agent_id (int) -- media agent Id

        Returns:
            Device Controller id for the given mountpath Id
        """
        query = """ SELECT	MDC.DeviceControllerId
                    FROM	MMDeviceController MDC
                    JOIN	MMMountPathToStorageDevice MPSD
                            ON	MDC.DeviceId = MPSD.DeviceId
                    JOIN	MMMountPath MP
                            ON	MPSD.MountPathId = MP.MountPathId
                    WHERE	MP.MountPathId = {0} AND
                            MDC.ClientId = {1} """.format(mountpath_id, media_agent_id)
        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != ['']:
            return int(cur[0])
        self._log.error("No entries present")
        raise Exception("Invalid MountPathId or ClientId")

    def get_device_id(self, mountpath_id):
        """
        Get the device id for the given mountpath id
            Args:
             mountpath_id(int)   -   Mountpath Id

            Returns:
                int    --  device id
        """

        query = """SELECT	DeviceId
                  FROM	MMMountPathToStorageDevice
                  WHERE	MountPathId = {0}""".format(mountpath_id)
        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != '':
            return int(cur[0])
        raise Exception("Unable to fetch device id")

    def get_media_location(self, media_name):
        """
            Get media location Id
            Args:
                media_name (str) : BarCode of the media
            Returns:
                Slot Id of the Media
        """

        query = f"""SELECT	SlotId
                    FROM	MMSlot MS
                    JOIN	MMMedia MM
                            ON MM.MediaId = MS.MediaId
                    WHERE	MM.BarCode = '{media_name}'"""

        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != '':
            return int(cur[0])
        raise Exception("Unable to fetch media location")

    def get_device_path(self, mountpath_id, client_id):
        """
        Get Device Path from MountpathId
        Agrs:
            mountpath_id (int)  --  Mountpath Id
        Returns:
            Device Path for the given Mountpath Id
        """
        query = """ SELECT	MDC.Folder
                    FROM	MMDeviceController MDC
                    JOIN	MMMountPathToStorageDevice MPSD
                            ON	MDC.DeviceId = MPSD.DeviceId
                    JOIN	MMMountPath MP
                            ON	MPSD.MountPathId = MP.MountPathId
                    WHERE	MP.MountPathId = {0} AND
                            MDC.ClientId = {1} """.format(mountpath_id, client_id)
        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != ['']:
            return cur[0]
        self._log.error("No entries present")
        raise Exception("Invalid MountPathId or ClientId")

    def get_device_access_type(self, mountpath_id, media_agent):
        """
        To get the device access type for the given mountpath id and mediaagent name
            Args:
             mountpath_id(int)   -   Mountpath Id

             media_agent(str)    -   MediaAgent Name

            Returns:
                int    --  device access type
        """

        query = """SELECT	MDC.DeviceAccessType
                    FROM	MMDeviceController MDC
                    JOIN	MMMountPathToStorageDevice MPSD
                            ON	MDC.DeviceId = MPSD.DeviceId
                    JOIN	APP_Client Cli
                            ON	MDC.ClientId = Cli.id
                    WHERE	MPSD.MountPathId = {0}
                            AND	Cli.name = '{1}'""".format(mountpath_id, media_agent)
        self._log.info("QUERY: %s", query)
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: %s", cur[0])
        if cur[0] != '':
            return int(cur[0])
        raise Exception("Unable to fetch device access type")

    def remove_content(self, test_data_path, client_machine, num_files=None):
        """
        Deletes subclient data

        Args:
            test_data_path  (str)   --  path where the subclient backup content is present

            client_machine  (obj)   --  object for the client machine on which subclient is present

            num_files       (int)   --  Number of files to be deleted
            Default : None [all files to be deleted]

        Returns:
            Number of files deleted

        """
        list_of_files = client_machine.get_files_in_path(test_data_path)
        if num_files is None:
            num_files = len(list_of_files)
            self._log.info("Deleting all files")
            client_machine.remove_directory(test_data_path)
        else:
            self._log.info("Deleting {0} files".format(str(num_files)))
            for c in range(0, num_files):
                client_machine.delete_file(list_of_files[c])
        return num_files

    def get_archive_file_size(self, copy_id, job_id=None):
        """
        Gets sum of archive file size on a copy/for a job

        Args:
            copy_id  (str)   --  storage policy copy id

            job_id   (str)   --  job id
            Default : None

        Returns:
            sum of archive file size

        """
        if job_id is None:
            query = """select sum(physicalSize)
                    from archChunkMapping
                    where archCopyId = {0} and jobid in
                        (select distinct jobid
                        from jmjobdatastats
                        where archgrpcopyid = {0})""".format(str(copy_id))
        else:
            query = """select sum(physicalSize)
                                from archChunkMapping
                                where archCopyId = {0} and jobid = {1})""".format(copy_id, job_id)
        self._log.info("Running query : \n{0}".format(query))
        self.csdb.execute(query)
        return int(self.csdb.fetch_one_row()[0])

    def set_opt_lan(self, value, server_name, user, password, client_id):
        """ Modifies Optimize for concurrent LAN backups option on the MA

            Args:
                value       (str)   -- Disable/Enable

                server_name (str)   -- SQL server name

                user        (str)   -- username for the SQL

                password    (str)   -- password for SQL (from registry)

                client_id   (str)   -- client machine ID

                e.g:
                    mmhelper_obj.set_opt_lan("Enable","######",
                    "sa", "######", "2")

            Raises:
                SDKException:
                    if input is incorrect
        """
        if value.lower() == "disable":
            operation = "&~"
        elif value.lower() == "enable":
            operation = "|"
        else:
            raise Exception("Input is not correct")
        query = """ UPDATE MMHost
                SET Attribute = Attribute {0} 32
                WHERE ClientId = {1}""".format(operation, client_id)
        self._log.info("Running query : \n{0}".format(query))
        #As Linux CS does not have any instance name, if user has provided instance name as commvault
        #strip off the instance name and use only server_name
        if not self.is_windows_cs:
            sql_server_name, sql_instance = server_name.split('\\')
            server_name  = sql_server_name

        mssql = MSSQL(server_name, user, password, "CommServ")
        mssql.execute(query)

    def get_global_param_value(self, param_name):
        """ gets param value from gxglobalparam table
        Args:
            param_name   (str)   --  global parameter name

        Returns:
            (int)   param value set in DB
        """

        query = """select value
                    from gxglobalparam
                    where name = '{0}'""".format(param_name)
        self._log.info("Running query : \n{0}".format(query))
        self.csdb.execute(query)
        if not self.csdb.fetch_one_row()[0]:
            return 0
        return int(self.csdb.fetch_one_row()[0])

    def get_jobs_picked_for_aux(self, auxcopy_job_id):
        """ gets number of jobs picked by an auxcopy job
        Args:
            auxcopy_job_id   (str)   --  global parameter name

        Returns:
            (int)   number of the jobs picked by auxcopy job

        """
        query = """SELECT COUNT(DISTINCT BackupJobID)
            FROM ArchChunkToReplicate
            WHERE AdminJobID = {0}""".format(auxcopy_job_id)
        self._log.info("Running query : \n{0}".format(query))
        self.csdb.execute(query)
        return int(self.csdb.fetch_one_row()[0])

    def get_to_be_copied_jobs(self, copy_id):
        """gets number of jobs in to-be-copied state for a copy
        Args:
            copy_id   (str)   --  copy id

        Returns:
            (int)   count of the jobs picked by auxcopy job

        """
        query = """ select count( distinct jobId)
                    from JMJobDataStats
                    where status in (101,102,103)
                    and archGrpCopyId = {0}""".format(copy_id)
        self._log.info("Running query : \n{0}".format(query))
        self.csdb.execute(query)
        return int(self.csdb.fetch_one_row()[0])

    def move_job_start_time(self, job_id, reduce_days=1):
        """
        runs script to change job time based on number of days in arguments
        Args:
            job_id (int) -- Job ID that needs to be moved with time
            reduce_days (int) -- number of days to reduce the job time

        Return:
            (Bool) -- True/False
        """
        self._log.info("Moving job {0} to {1} day(s) behind".format(job_id, reduce_days))
        sql_script = """
        DECLARE @curCommCellId INTEGER
        SET @curCommCellId = 0

        DECLARE @curJobId INTEGER
        SET @curJobId = {0}

        DECLARE @i_days INTEGER
        SET @i_days = {1}

        SELECT @curCommCellId = commcellId
        FROM JMBkpStats
        where jobId = @curJobId

        UPDATE JMBkpStats
        SET servStartDate = servStartDate - (@i_days *24* 3600),
        servEndDate = servEndDate - (@i_days* 24* 3600)
        WHERE jobId = @curJobId
        AND commCellId = @curCommCellId
        """.format(job_id, reduce_days)
        retcode = self.execute_update_query(sql_script)
        if retcode:
            return True
        raise Exception("failed to run the script to move job time")

    def execute_update_query(self, query, db_password=None, db_user=None):
        """
        Executes update query on CSDB
        Args:
            query (str) -- update query that needs to be run on CSDB
            db_password (str)   -- sa password for CSDB login
            db_user (str)   -- username for CSDB login

        Return:
            Response / exception
        """
        if not db_user:
            db_user = "sa"
        try:
            if 'SqlSaPassword' in self.tcinputs and db_password is None:
                db_password = self.tcinputs['SqlSaPassword']

            self.dbobject = MSSQL(self.commcell.commserv_hostname + self.instance_sep +
                                  self.commserve_instance,
                                  db_user, db_password, "CommServ", False, True)
            response = self.dbobject.execute(query)
            if response.rows is None:
                return bool(response.rowcount == 1)
            return response

        except Exception as err:
            raise Exception("failed to execute query {}".format(err))

    def cleanup(self):
        """
        deletes backupset, storage policy
        """
        self._log.info("********* cleaning up BackupSet, StoragePolicy ***********")
        self.agent.backupsets.delete(self.backupset_name)
        self.commcell.storage_policies.delete(self.storage_policy_name)

    def validate_copy_prune(self, copy_name):
        """
        verifies if a copy exits or deleted
        Args:
            copy_name (str) -- copy name to verify if deleted

        Return:
             (Bool) True if copy is deleted
             (Bool) False if copy exists
        """
        self.sp = self.commcell.storage_policies.get(self.storage_policy_name)
        if self.sp.has_copy(copy_name):
            self._log.info("Copy exists! ")
            return False
        self._log.info("Copy doesnt exist!")
        return True

    def validate_job_prune(self, job_id, copy_id):
        """
        Validates if a job is aged by checking table entries
        Args:
            job_id (int) -- job id to check for aged
            copy_id (int) -- copy id needed for the job
                            if copy_id exist use new query else old one.

        Return:
            (Bool) True/False
        """
        if copy_id:
            jmdatastats_exists = self.validate_jmdatastats(job_id, copy_id)
            (archfile_exists, archfile_id) = self.validate_archfile(job_id)
            (archchunkmapping_exists, archchunkmapping_values) = self.validate_archchunkmapping(job_id, copy_id)
        else:
            jmdatastats_exists = self.validate_jmdatastats(job_id)
            (archfile_exists, archfile_id) = self.validate_archfile(job_id)
            (archchunkmapping_exists, archchunkmapping_values) = self.validate_archchunkmapping(job_id)
        if (jmdatastats_exists and archfile_exists and archchunkmapping_exists):
            self._log.info("Job {0} is not aged!".format(job_id))
            return False
        self._log.info("Job {0} is aged!".format(job_id))
        return True

    def validate_jmdatastats(self, job_id, copy_id=None):
        """
        Validate if a job id exists in table jmdatastats
        Args:
            job_id (int) -- backup job id to check in table
            copy_id (Int) -- we can pass copy id to check for particular copy.
                            if copy_id : then use new query else old one

        Return:
            (int) agedTime
            (int) agedBy
            (int) disabled flag for the jobid
        """
        if copy_id:
            query = """select agedTime, agedBy, disabled&256 from JMJobDataStats where jobId = {0} 
            and archGrpCopyId = {1}""".format(job_id, copy_id)
        else:
            query = """select agedTime, agedBy, disabled&256 from JMJobDataStats where jobId = {0}""".format(job_id)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(cur))
        if cur != ['']:
            values = [int(x) for x in cur]
            if values[0] > 0 and values[1] > 0 and values[2] == 256:
                return False
        self._log.info("no entries present")
        return True

    def validate_archfile(self, job_id):
        """
        Validate if archfile entry for job id is exists
        Args:
            job_id (int) -- job id to check in table

        Return:
            (Bool) True with archfileid if exists
            (Bool) False if entry not present
        """
        query = """select id from ArchFile where jobid = {0}""".format(job_id)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_all_rows()
        self._log.info("RESULT: {0}".format(cur[0]))
        if cur[0] != ['']:
            values = [int(x) for x in cur[0]]
            if values:
                return True, values
        self._log.info("no entries present")
        return False, 0

    def validate_archfilecopy(self, archfile_id, copy_id):
        """
        validate if archfilecopy entry for job id exists
        Args:
            archfile_id (int) -- archfile id to check in table
            copy_id (int) -- copy id to check in table

        Return:
            (Bool) True if entries exist
            (Bool) False if entries doesnt exist
        """
        query = """select 1 from archFileCopy where archFileId = {0} and archCopyId = {1}
                """.format(archfile_id, copy_id)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self._log.info("RESULT: {0}".format(cur[0]))
        if cur[0] != ['']:
            value = int(cur[0])
            if value:
                return True
        self._log.info("no entries present")
        return False

    def validate_archchunkmapping(self, job_id, copy_id=None):
        """
        Validate if archchunkmapping entry for job id exists
        Args:
            job_id (int) -- job id to check in table
            copy_id (int) -- copy id of the job to validate for
                            if copy_id : then use new query else old one

        Return:
            (Bool) True if entries exist
            (Bool) False if entries doesnt exist
        """
        if copy_id:
            query = """select archFileId, archChunkId from archchunkmapping where jobId = {0} and archCopyId = {1}
                """.format(job_id, copy_id)
        else:
            query = """select archFileId, archChunkId from archchunkmapping where jobId = {0}""".format(job_id)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_all_rows()
        self._log.info("RESULT: {0}".format(cur))
        if cur != [['']]:
            values = [[int(y) for y in x] for x in cur]
            if values:
                return True, values
        self._log.info("no entries present")
        return False, 0

    def validate_job_retentionflag(self, job_id, retention_flag):
        """
        Validate if extended retention flag is set for a job id
        Args:
            job_id (int) -- Job id to check for retention flags
            retention_flag (str) -- retention flag to check for job id

        Return:
            (Bool) True/False
        """
        mapping = {
            "EXTENDED_ALLFULL": 2,
            "EXTENDED_WEEK": 4,
            "EXTENDED_MONTH": 8,
            "EXTENDED_QUARTER": 16,
            "EXTENDED_HALFYEAR": 32,
            "EXTENDED_YEAR": 64,
            "MANUALLY_PIN": 128,
            "EXTENDED_GRACE_WEEK": 256,
            "EXTENDED_GRACE_MONTH": 512,
            "EXTENDED_GRACE_QUARTER": 1024,
            "EXTENDED_GRACE_HALFYEAR": 2048,
            "EXTENDED_GRACE_YEAR": 4098,
            "EXTENDED_CANDIDATE_WEEK": 8192,
            "EXTENDED_CANDIDATE_MONTH": 16384,
            "EXTENDED_CANDIDATE_QUARTER": 32768,
            "EXTENDED_CANDIDATE_HALFYEAR": 65536,
            "EXTENDED_CANDIDATE_YEAR": 131072,
            "EXTENDED_HOUR": 262144,
            "EXTENDED_DAY": 524288,
            "EXTENDED_CANDIDATE_HOUR": 1048576,
            "EXTENDED_CANDIDATE_DAY": 2097152,
            "EXTENDED_GRACE_HOUR": 4194304,
            "EXTENDED_GRACE_DAY": 8388608,
            "EXTENDED_LAST_JOB": 16777216,
            "EXTENDED_FIRST": 33554432
        }
        value = mapping[retention_flag]
        query = """select retentionFlags&{0} from JMJobDataStats where jobid = {1}
                and dataType = 2""".format(value, job_id)
        self._log.info("QUERY: {0}".format(query))
        self.csdb.execute(query)
        cur = self.csdb.fetch_all_rows()
        self._log.info("RESULT: {0}".format(cur[0]))
        if int(cur[0][0]) == value:
            return True
        self._log.info("flag not present")
        return False

    def setup_environment(self):
        """
        get testcase object and setup library, non dedupe storage policy, backupset and subclient

        Returns:
            (object)    -- disk library object
            (object)    -- storage policy object
            (object)    -- backupset object
            (object)    -- subclient object
        """
        disk_library = self.configure_disk_library()
        storage_policy = self.configure_storage_policy()
        backupset = self.configure_backupset()
        subclient = self.configure_subclient()
        return disk_library, storage_policy, backupset, subclient

    def configure_disk_library(self,
                               library_name=None,
                               ma_name=None,
                               mount_path=None):
        """
        Create a new disk library if not exists
        Args:
            library_name (str)  -- library name to create

            ma_name (str)       -- MA name to use for library

            mount_path (str)    -- path to create mount path for library

        Return:
            (object)    -- disk library object
        """
        # config disk library
        if library_name is None:
            library_name = self.library_name
        if ma_name is None:
            ma_name = self.tcinputs["MediaAgentName"]
        if mount_path is None:
            mount_path = self.tcinputs["MountPath"]
        self._log.info("check library: %s", library_name)
        if not self.commcell.disk_libraries.has_library(library_name):
            self._log.info("adding Library...")
            disk_library = self.commcell.disk_libraries.add(library_name,
                                                            ma_name,
                                                            mount_path)
            self._log.info("Library Config done.")
            return disk_library
        disk_library = self.commcell.disk_libraries.get(library_name)
        self._log.info("Library exists!")
        return disk_library

    def configure_disk_mount_path(self, disk_library, mount_path, media_agent, **kwargs):
        """ Adds a mount path [local/remote] to the disk library

               Args:
                   disk_library  (object) --  instance of disk library to add the mountpath to

                   mount_path  (str)   -- Mount path which needs to be added to disklibrary.
                                         This could be a local or remote mount path on mediaagent

                   media_agent (str)   -- MediaAgent on which mountpath exists

                   \*\*kwargs  (dict)  --  Optional arguments

                    Available kwargs Options:

                       username    (str)   -- Username to access the mount path

                       password    (str)   -- Password to access the mount path

               Returns:
                   None
        """
        username = kwargs.get('username', '')
        password = kwargs.get('password', '')
        self._log.info("Adding MountPath to Disk Library: %s", disk_library.library_name)
        disk_library.add_mount_path(mount_path, media_agent, username, password)
        self._log.info("MountPath Configuration Done.")

    def configure_cloud_library(self, library_name, media_agent, mount_path, username, password='', server_type='',
                                saved_credential_name=''):
        """
        Adds a new Cloud Library to the Commcell.

            Args:
                library_name (str)        --  name of the new cloud library to add

                media_agent  (str/object) --  name or instance of media agent to add the library to

                mount_path   (str)        --  cloud container or bucket

                username     (str)        --  username to access mountpath in the format <ServiceHost>//<AccountName>
                                              Eg: s3.us-west-1.amazonaws.com//MyAccessKeyID

                password     (str)        --  password to access the mount path

                server_type   (str)       --  provide cloud library server type

                saved_credential_name (str) -- name of saved credentials
                                                if saved credentials are username used be given in the format
                                                <ServiceHost>//__CVCRED__
                                                Eg: s3.us-west-1.amazonaws.com//__CVCRED__

            Returns:
                object - instance of the disk library class, if created successfully
        """
        server_type_dict = mediaagentconstants.CLOUD_SERVER_TYPES
        if not server_type.lower() in server_type_dict:
            raise Exception('Invalid server type specified')
        server_type = server_type_dict[server_type.lower()]
        self._log.info("check library: %s", library_name)
        if not self.commcell.disk_libraries.has_library(library_name):
            self._log.info("adding Library...")
            cloud_library = self.commcell.disk_libraries.add(library_name, media_agent, mount_path, username, password,
                                                             server_type, saved_credential_name)
            self._log.info("Library Config done.")
            return cloud_library
        cloud_library = self.commcell.disk_libraries.get(library_name)
        self._log.info("Library exists!")
        return cloud_library

    def configure_cloud_mount_path(self,cloud_library, mount_path, media_agent, username, password, server_type):
        """ Adds a mount path to the cloud library

            Args:
                cloud_library (object)  -- instance of cloud library

                mount_path  (str)   -- cloud container or bucket.

                media_agent (str)   -- MediaAgent on which mount path exists

                username    (str)   -- Username to access the mount path in the format <Service Host>//<Account Name>
                Eg: s3.us-west-1.amazonaws.com//MyAccessKeyID. For more information refer http://documentation.commvault.com/commvault/v11/article?p=97863.htm

                password    (str)   -- Password to access the mount path

                server_type  (str)   -- provide cloud library server type
        """
        server_type_dict = mediaagentconstants.CLOUD_SERVER_TYPES
        if not server_type.lower() in server_type_dict:
            raise Exception('Invalid server type specified')
        server_type = server_type_dict[server_type.lower()]
        self._log.info("Adding MountPath to Cloud Library: %s", cloud_library.library_name)
        cloud_library.add_cloud_mount_path(mount_path, media_agent, username, password, server_type)
        self._log.info("MountPath Configuration Done.")

    def configure_storage_policy(self,
                                 storage_policy_name=None,
                                 library_name=None,
                                 ma_name=None):
        """
        creates a new storage policy if not exists
        Args:
            storage_policy_name (str)   -- storage policy name to create storage policy

            library_name (str)          -- library to use for creating storage policy

            ma_name (str)               -- MA name to use for library

        Return:
            (object)    -- storage policy object
        """
        # config storage policy
        if storage_policy_name is None:
            storage_policy_name = self.storage_policy_name
        if library_name is None:
            library_name = self.library_name
        if ma_name is None:
            ma_name = self.tcinputs["MediaAgentName"]
        self._log.info("check SP: %s", storage_policy_name)
        if not self.commcell.storage_policies.has_policy(storage_policy_name):
            self._log.info("adding Storage policy...")
            storage_policy = self.commcell.storage_policies.add(storage_policy_name,
                                                                library_name,
                                                                ma_name,
                                                                None, None, 1)
            self._log.info("Storage policy config done.")
            return storage_policy
        self._log.info("Storage policy exists!")
        storage_policy = self.commcell.storage_policies.get(storage_policy_name)
        return storage_policy

    def configure_backupset(self,
                            backupset_name=None,
                            agent=None):
        """
        Creates a new backupset if not exits
        Args:
            backupset_name (str)   -- backupset name to create

        Return:
            (object)    -- Backupset object
        """
        # config backupset
        if backupset_name is None:
            backupset_name = self.backupset_name
        if agent is None:
            agent = self.agent
        self._log.info("check BS: %s", backupset_name)
        if not agent.backupsets.has_backupset(backupset_name):
            self._log.info("adding Backupset...")
            backupset = agent.backupsets.add(backupset_name)
            self._log.info("Backupset config done.")
            return backupset
        self._log.info("Backupset exists!")
        backupset = self.agent.backupsets.get(backupset_name)
        return backupset

    def configure_subclient(self,
                            backupset_name=None,
                            subclient_name=None,
                            storage_policy_name=None,
                            content_path=None,
                            agent=None):
        """
        Created a new subclient and added content
        Args:
            storage_policy_name (str)   -- storage policy name to associate with storage policy

            subclient_name (str)        -- subclient name to create

            backupset_name (str)        -- backupset name to create subclient under it

            content_path (str)          -- content to add to subclient

        Return:
             (object)   -- Subclient object
        """
        # config subclient
        if backupset_name is None:
            backupset_name = self.backupset_name
        if subclient_name is None:
            subclient_name = self.subclient_name
        if storage_policy_name is None:
            storage_policy_name = self.storage_policy_name
        if content_path is None:
            content_path = self.content_path
        if agent is None:
            agent = self.agent
        self._log.info("check SC: %s", subclient_name)
        self._log.info("creating backupset object: " + backupset_name)
        self._backupset = agent.backupsets.get(backupset_name)
        if not self._backupset.subclients.has_subclient(subclient_name):
            self._log.info("adding Subclient...")
            self._subclient = self._backupset.subclients.add(subclient_name,
                                                             storage_policy_name)
        else:
            self._log.info("Subclient exists!")
        # add subclient content
        self._log.info("creating subclient object: %s", subclient_name)
        self._subclient = self._backupset.subclients.get(subclient_name)
        self._log.info("setting subclient content to: %s", [content_path])
        self._subclient.content = [content_path]
        self._log.info("Subclient config done.")
        return self._subclient

    def configure_secondary_copy(self,
                                 sec_copy_name,
                                 storage_policy_name=None,
                                 library_name=None,
                                 ma_name=None,
                                 global_policy_name=None):
        """
        Creates a new secondary copy if doesnt exist
        Args:
            sec_copy_name: secondary copy name to create

            storage_policy_name (str)   -- storage policy name to create secondary copy

            library_name (str)          -- library to use for creating secondary copy

            ma_name (str)               -- MA name to use for library

            global_policy_name (str)    -- global storage policy to use for creating secondary copy

        Return:
            (object)    -- secondary copy object
        """
        if storage_policy_name is None:
            storage_policy_name = self.storage_policy_name
        if library_name is None and global_policy_name is None:
            library_name = self.library_name
        if ma_name is None and global_policy_name is None:
            ma_name = self.tcinputs["MediaAgentName"]
        # create secondary copy
        self.sp = self.commcell.storage_policies.get(storage_policy_name)
        self._log.info("check secondary copy: %s", sec_copy_name)
        if not self.sp.has_copy(sec_copy_name):
            self._log.info("adding secondary copy...")
            self.sp.create_secondary_copy(sec_copy_name, library_name, ma_name, global_policy=global_policy_name)
            self._log.info("Secondary copy cofig done.")
            self.sec_copy = self.sp.get_copy(sec_copy_name)
            return self.sec_copy
        self._log.info("secondary copy exists!")
        self.sec_copy = self.sp.get_copy(sec_copy_name)
        return self.sec_copy

    def create_uncompressable_data(self, client, path, size, num_of_folders=1):
        """
        Creates unique uncompressable data

        Args:

            client  --  (str)   -- Client name on which data needs to be created

            path    --  (str)   --  Path where data is to be created

            size    --  (float) --  Data in GB to be created for each folder
                                    (restrict to one decimal point)
        Returns:
              (boolean)
        """
        options_selector = OptionsSelector(self.commcell)
        return options_selector.create_uncompressable_data(client, path, size, num_of_folders)

    def execute_select_query(self, query):
        """ Executes CSDB select query

        Args:
            query (str) -- select query that needs to be run on CSDB

        Return:
            query response
        """
        self._log.info("Running query : \n{0}".format(query))
        self.csdb.execute(query)
        db_response = self.csdb.fetch_all_rows()
        self._log.info("RESULT: {0}".format(db_response))
        return db_response

    def unload_drive(self, library_name, drive_name):
        """
        Unloads the drive on the library given
        Args:
                library_name(str) -- Tape Library Name

                drive_name(str)   -- Drive Name
        """
        self._log.info("Unloading Drive %s on %s ", drive_name, library_name)
        self.commcell.execute_qcommand(f'qdrive unload -l {library_name} -dr {drive_name}')

    def remove_autocopy_schedule(self, storage_policy_name, copy_name):
        """
        Removes association with System Created Automatic Auxcopy schedule on the given copy
        Args:
                storage_policy_name (str)   -- storage policy name

                copy_name           (str)   --  copy name
        """
        if self.commcell.schedule_policies.has_policy('System Created Autocopy schedule'):
            auxcopy_schedule_policy = self.commcell.schedule_policies.get('System Created Autocopy schedule')
            association = [{'storagePolicyName': storage_policy_name, 'copyName': copy_name}]
            auxcopy_schedule_policy.update_associations(association, 'exclude')


class CloudLibrary(object):
    '''Class for representing CloudLibrary'''

    def __init__(self, libraryname, libraryinfo):
        '''Class for representing CloudLibrary
        Args:
                libraryname  (str)         --  name of the cloud library
                libraryinfo  (dictionary)   --  dictionary containing library information

            Returns:
                None

            Raises:
                None
        '''
        self.libraryname = libraryname
        self.loginname = libraryinfo.get("loginName", "")
        self.secret_accesskey = libraryinfo.get("password", "")
        self.servertype = libraryinfo.get("serverType", "")
        self.mountpath = libraryinfo.get("mountPath", "")

    @staticmethod
    def cleanup_entities(commcell, log, entity_config):
        '''Method to cleanup commcell entities

        Args:
            entity_config (dict)   -- Dictionary containing the entity(s) as key,
                                        and value(s) as a list of the entities to delete.
        Returns:
            dictonary with failed entities

        Raises Exception:
            - If failed to delete any entity
        '''

        try:
            log.info("Start deleting entities configured")
            failed_entities = {"storagepolicy": [], "library": []}

            if 'storagepolicy' in entity_config:
                storageobj = commcell.policies.storage_policies
                storagepolies = entity_config['storagepolicy']

                for policy in storagepolies:
                    try:
                        storageobj.delete(policy)
                        log.info("successfully deleted SP {}".format(policy))
                    except Exception as err:
                        log.error(err)
                        failed_entities["storagepolicy"].append(policy)

            if 'library' in entity_config:
                storageobj = commcell.disk_libraries
                libraries = entity_config['library']

                for library in libraries:
                    try:
                        storageobj.delete(library)
                        log.info("successfully deleted library {}".format(library))
                    except Exception as err:
                        log.error(err)
                        failed_entities["library"].append(policy)

            return failed_entities

        except Exception as excp:
            raise Exception("Failed in cleanup_enties function with error {}".format(excp))


class PowerManagement(object):
    """" Class for power management helper functions"""
    def __init__(self):
        self.log = logger.get_log()
	
    def configure_cloud_mediaagent(self, pseudo_client_name, ma_obj):
        """
                Setup and configure cloud MediaAgent

                Args:
                    psudo_client_name -- (str) -- To be used as cloud controller for the MediaAgent
                    ma_obj -- (MediaAgent class object) -- MediaAgent class Object of the MA to configure
        """

        self.log.info("Verifying power management configurations of MediaAgent [%s] ", ma_obj._media_agent_name)
        if ma_obj._is_power_mgmt_supported:
            self.log.info("Power Management supported")
            if ma_obj._is_power_management_enabled:
                self.log.info("Power management is enabled")
                if ma_obj._power_management_controller_name == pseudo_client_name:
                    self.log.info("MA is using correct cloud controller[%s] ", pseudo_client_name)
                else:
                    self.log.info("MA is not using correct cloud controller. Correcting that")
                    ma_obj.enable_power_management(pseudo_client_name)
            else:
                self.log.info("Power management is not enabled. Enabling that")
                ma_obj.enable_power_management(pseudo_client_name)
        else:
            raise Exception('Power management is not supported on MediaAgent ' + ma_obj._media_agent_name)
			

    def power_off_media_agents(self, list_of_media_agent_objects):
        """
        Power-off multiple MediaAgents simultaneously

                    Args:
                        list_of_media_agent_objects (list)  --  List of MediaAgent objects

                    Raises:
                           If number of MediaAgent object passed is less than 2
        """
        if type(list_of_media_agent_objects) != list :
            raise Exception("A List of MediaAgent objects expected. At least 2 MediaAgent objects require. Use ma_obj.power_off() instead")

        thread_pool = []
        for media_agent in list_of_media_agent_objects:
            power_thread = threading.Thread(target=media_agent.power_off)
            self.log.info("Starting power-off thread. MediaAgent : {0}".format(media_agent._media_agent_name))
            power_thread.start()
            thread_pool.append(power_thread)

        self.log.info("Waiting for all MediaAgents to power-off")

        for power_thread in thread_pool:
            power_thread.join()

        self.log.info("All MediaAgents are powered-off successfully")



    def power_on_media_agents(self, list_of_media_agent_objects):
        """
        Power-on multiple MediaAgents simultaneously

                    Args:
                        list_of_media_agent_objects (list)  --  List of MediaAgent objects

                    Raises:
                           If number of MediaAgent object passed is less than 2
        """
        if type(list_of_media_agent_objects) != list :
            raise Exception("A List of MediaAgent objects expected. At least 2 MediaAgent objects require. Use ma_obj.power_on() instead")

        thread_pool = []
        for media_agent in list_of_media_agent_objects:
            power_thread = threading.Thread(target=media_agent.power_on)
            self.log.info("Starting power-on thread. MediaAgent : {0}".format(media_agent._media_agent_name))
            power_thread.start()
            thread_pool.append(power_thread)

        self.log.info("Waiting for all MediaAgents to power-on")

        for power_thread in thread_pool:
            power_thread.join()

        self.log.info("All MediaAgents are powered-on successfully")

    def validate_powermgmtjobtovmmap_table(self,jobid,media_agent_id,csdb):
        """
        Validate the entry on MMPowerMgmtJobToVMMap table for the job and MediaAgent

            Args:
                        jobid --  Job ID that powered-on / powered-off the MediaAgent
                        media_agent_id -- MediaAgent ID that is powered-on / powered-off
                        csdb -- CSDB object to execute the query

        """
        self.log.info("Validating entry on table MMPowerMgmtJobToVMMap for job {0} and host id {1}".format(jobid,media_agent_id))
        query="""select count(*)  
                from MMPowerMgmtJobToVMMap 
                where hostid={0} 
                and entityid={1}
                """.format(media_agent_id,jobid)

        csdb.execute(query)
        count=csdb.fetch_one_row()[0]
        self.log.info("{0} entries found on table MMPowerMgmtJobToVMMap for job {1} and host id {2}".format(count,jobid,media_agent_id))
        if int(count) < 1 :
            raise Exception("Validation Failed. No entry found on table MMPowerMgmtJobToVMMap for job {0} and host id {1}".format(jobid,media_agent_id))

        self.log.info("Successfully validated table MMPowerMgmtJobToVMMap for job {0} and host id {1}".format(jobid,media_agent_id))
