# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

import sys
import time
from operator import itemgetter
from AutomationUtils.machine import Machine
from AutomationUtils.database_helper import MSSQL
from MediaAgents.mediaagentconstants import HYPERSCALE_CONSTANTS
from VirtualServer.VSAUtils import HypervisorHelper


class HyperScaleHelper:
    """
       Hyperscale helper library for DB based and REST based functionalities

       This file contains a class named hyperscale_helper that can be used to carry out
       all hyperscale related operations.

       The instance of this class can be used to perform various operations on a
       Hyperscale Storage Pool like:

       hyperscale_helper
       =================

       __init__                -- initialize object of the class

       execute_update_query    -- Executes update query on CSDB

       get_storage_pool_details -- Gets StoragePool Details from the MMSDSStoragePool table

       get_all_bricks          -- Gets flags, BrickHealth, BlockDeviceHealth and DeviceOsPath
           for all bricks associated with a storage pool for a particular MA

       get_all_bricks_for_hostid -- Gets flags, BrickHealth, BlockDeviceHealth and DeviceOsPath
           for all bricks associated with a particular MA

       get_brick_daemon_for_brick -- Check if brick daemon is present for the hyperscale brick

       get_all_nodes           -- Gets a list of MAs associated with a particular Storage Pool

       check_brick_health_status -- Checks the flag, blockdevicehealth and brickhealth
       values for all bricks for a given MA

       wait_for_completion     -- Check if hyperscale storage pool creation
       or expansion operation has completed or failed

       add_nodes               -- Add 3 more nodes to horizontally expand a setup by 1 block

       create_storage_pool     -- Create a hyperscale storage pool containing 3 nodes

       delete_storage_pool     -- Deletes a hyperscale storage pool

       get_associated_storage_policies -- Get a list of Storage Policies associated
       to a hyperscale Storage pool's GDSP

       check_if_storage_pool_is_present -- Check if s hyperscale storage pool is present or not

       get_associated_mas      -- Get all MAs associated with a hyperscale storage pool

       create_and_associate_plan -- Creates a plan that associates itself with a
       hyperscale storage pool's GDSP

       delete_plan             -- Deletes a hyperscale plan

       clean_up_ma             -- Cleans up all data disks and peer files from an MA

       reassociate_all_associated_subclients -- Reassociates all subclients associated
       with a storage policy to none assigned

       clean_up_storage_pool   -- Cleans up all entries left by a pre-existing storage pool

       check_brick_flag_status -- Check if the health status is proper for a particular
       MA's bricks or not
   """

    def __init__(self, commcell_obj, csdb_obj, log_obj):
        """
            Initializes instance of the Machine class.
            Args:

            commcell_obj    (object)    --  instance of the Commcell
            csdb_obj        (object)    --  instance of the CSDB
            log_obj         (object)    --  instance of the Logger

        """
        self.commcell = commcell_obj
        self.csdb = csdb_obj
        self.log = log_obj
        self.storage_pool_obj = self.commcell.storage_pools
        self.dbobject = None
        self.commserve_instance = "\\commvault"
        if self.commcell.is_linux_commserv:
            self.commserve_instance = ""

    def connect(self, db_user, db_password):
        """
        Connection to db
        """
        try:
            self.dbobject = MSSQL(self.commcell.commserv_hostname + self.commserve_instance,
                                  db_user, db_password, "CommServ", False, True)
        except Exception as err:
            raise Exception("Exception raised {}, failed to connect to the Database with username {}".format(
                err, db_user))

    def execute_update_query(self, query, sql_login, db_password):
        """
        Executes update query on CSDB
        Args:
            query (str) -- update query that needs to be run on CSDB
            sql_login (str) --  Sql login user name
            db_password (str)   -- sa password for CSDB login
        Return:
            Response / exception
        """
        try:
            if self.dbobject is None:
                self.connect(sql_login, db_password)

            response = self.dbobject.execute(query)
            if response.rows is None:
                return bool(response.rowcount == 1)
            return response
        except Exception as err:
            raise Exception("failed to execute query {0}".format(err))

    def get_storage_pool_details(self, name):
        """
            Returns Storage Pool details as a list
            Args:
                name (str) -- Storage Pool Name
            Return:
                storage_pool_details / Exception
        """
        self.commcell.storage_pools.refresh()
        storage_pool_details = self.commcell.storage_pools.get(name)
        self.log.info("Storage Pool Details %s ", storage_pool_details)
        return storage_pool_details

    def get_all_bricks(self, name, host_id):
        """
            Gets flags, BrickHealth, BlockDeviceHealth and DeviceOsPath
            for all bricks associated with a storage pool for a particular MA
            Args:
               name(name) -- Storage Pool name
               host_id(name) -- Media agent id
            Return:
                rows -- List of bricks and their details
        """
        storage_pool_details = self.get_storage_pool_details(name)
        storage_pool_properties = storage_pool_details._storage_pool_properties
        storage_pool_node_detail_list = storage_pool_properties["storagePoolDetails"]["storagePoolNodeDetailsList"]
        result = []
        for brick_details in storage_pool_node_detail_list:
            if brick_details["mediaAgent"]["mediaAgentId"] == host_id:
                result.append(brick_details)
        self.log.info("Bricks for host_id %s are %s", host_id, result)
        return result

    def get_all_bricks_for_hostid(self, hostid):
        """
            Gets flags, BrickHealth, BlockDeviceHealth and DeviceOsPath for all bricks associated with a particular MA
            Args:
                hostid (int/str) -- Host ID of the MA

            Return:
                rows -- List of bricks and their details
        """
        self.csdb.execute(
            "select flags,BrickHealth, BlockDeviceHealth, deviceospath, deviceId from mmdiskhwinfo where "
            "mountpathusagetype = 2 and hostid = " + str(hostid))
        rows = self.csdb.fetch_all_rows()
        return rows

    def get_brick_daemon_for_brick(self, media_agent, disk):
        """
           Check if brick daemon is present for the hyperscale brick
           Args:
               media_agent (str) -- Name of the MA where the daemon is to be checked
               disk (string) -- Name of the hyperscale brick

           Return:
                String -- Console output
        """

        # SSH to MA
        ma_session = Machine(media_agent, self.commcell)
        output = ma_session.execute_command("ps -ef | grep {0}".format(disk))
        self.log.info(output.output)
        return output.output

    def get_all_nodes(self, storage_pool_name):
        """
            Gets a list of MAs associated with a particular Storage Pool
            Args:
                storage_pool_name (str) -- Name of the Storage Pool
            Return:
                rows -- List of all MAs associated with the given Storage Pool
        """
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        storage_pool_properties = storage_pool_details._storage_pool_properties
        storage_pool_node_detail_list = storage_pool_properties['storagePoolDetails']['storagePoolNodeDetailsList']
        node_list = []
        for details in storage_pool_node_detail_list:
            data = [details['mediaAgent']['mediaAgentId'], details['mediaAgent']['mediaAgentName']]
            if data not in node_list:
                node_list.append(data)
        self.log.info("Node detils %s", node_list)
        return node_list

    def check_brick_health_status(self, hostid):
        """
            Checks the flag, blockdevicehealth and brickhealth values for all bricks for a given MA
            Args:
                hostid (int) -- host id for the given MA
            Return:
                True / Exception
        """
        bricks = self.get_all_bricks_for_hostid(hostid)

        for brick in bricks:
            self.log.info(brick[3])
            if int(brick[0]) != 1:
                raise Exception("{0} Brick does not have the flag set to 1".format(brick[3]))
            if int(brick[1]) != 23:
                raise Exception("{0} Brick does not have the BrickHealth set to 23".format(brick[3]))
            if int(brick[2]) != 15:
                raise Exception("{0} Brick does not have the BlockDeviceHealth set to 15".format(brick[3]))
            self.log.info("Flags, brickHealth and BlockDeviceHealth correctly set for {0}".format(brick[3]))

        return True

    def get_host_id(self, ma_name):
        """
            Returns the hostid of the MA
            Args:
                ma_name (str) -- Name of the MA whose hostid is to be returned
            Return:
                hostid (int) -- host id for the given MA / Exception
        """
        ma_client = self.commcell.clients.get(ma_name)
        ma_id = ma_client.client_id
        self.log.info("Host id for ma: %s is %s", ma_name, ma_id)
        return ma_id

    def wait_for_completion(self, storage_pool_name):
        """
            Check if hyperscale storage pool creation or expansion operation has completed or failed

            Args:
                storage_pool_name (str) -- Name of the storage pool

            Return:
                 Boolean -- True or False based on whether the storage pool creation succeeded or failed
        """
        count = 0
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        storage_pool_properties = storage_pool_details._storage_pool_properties
        status = storage_pool_properties["storagePoolDetails"]['statusCode']

        while int(status) is 100 and count <= 20:
            storage_pool_details = self.get_storage_pool_details(storage_pool_name)
            storage_pool_properties = storage_pool_details._storage_pool_properties
            status = storage_pool_properties["storagePoolDetails"]['statusCode']
            self.log.info("status for pool %s is %s", storage_pool_name, status)
            count += 1
            time.sleep(60)
        status = storage_pool_properties["storagePoolDetails"]['statusCode']
        self.log.info(status)
        if int(status) == 300:
            return True
        self.log.info("Storage Pool status %s not 300", status)
        return False

    def add_nodes(self, storage_pool_name, *args):
        """
        Add 3 more nodes to horizontally expand a setup by 1 block

        Args:
            storage_pool_name (str) -- Name of the storage pool
            *args: Media agent names
                    ma1_name (str) -- Name of the first MA
                    ma2_name (str) -- Name of the second MA
                    ma3_name (str) -- Name of the third MA

        Return:
            flag (int) -- Response code received from the Rest API call
            response (str) -- Response received from the Rest API
            status (boolean) -- True or False based on the success or failure of the operation
        """
        if len(args) <= 2:
            self.log.error("insufficient number of inputs %s" % str(args))
            raise Exception("insufficient number of inputs %s" % str(args))

        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        ma_values = [self.commcell.media_agents.get(value) for value in args]

        ma_ids = [self.get_host_id(ma_name) for ma_name in args]
        self.log.info("Check if hosts present and flags = 0 not configured")
        for ma_id in ma_ids:
            if not self.check_brick_available_status(ma_id):
                raise Exception("Brick status not available for creation")
        storage_pool_details.hyperscale_add_nodes(ma_values)

        self.log.info("Waiting CSDB to populate")
        time.sleep(900)

        status = self.wait_for_completion(storage_pool_name)
        response = self.get_storage_pool_details(storage_pool_name)
        self.log.info(response)
        return status, response

    def create_storage_pool(self, storage_pool_name, *args):
        """
            Create a hyperscale storage pool containing 3 nodes

            Args:
                storage_pool_name (str) -- Name of the storage pool
               *args: Media agent names
                    ma1_name (str) -- Name of the first MA
                    ma2_name (str) -- Name of the second MA
                    ma3_name (str) -- Name of the third MA

            Return:
                flag (int) -- Response code received from the Rest API call
                response (str) -- Response received from the Rest API
                status (boolean) -- True or False based on the success or failure of the operation
        """
        setup = "Standard"
        if len(args) <= 2:
            self.log.error("insufficient number of inputs %s" % str(args))
            raise Exception("insufficient number of inputs %s" % str(args))
        ma_values = [self.commcell.media_agents.get(value) for value in args]
        ma_ids = [self.get_host_id(ma_name) for ma_name in args]
        self.log.info("Check if hosts present and flags = 0 not configured")
        for ma_id in ma_ids:
            if not self.check_brick_available_status(ma_id):
                raise Exception("Brick status not available for creation")

        if not self.resiliency(ma_ids, setup):
            self.log.info("Resiliency not Correct")
            raise Exception("Resiliency factor not correct or brick status not available")

        self.commcell.storage_pools.hyperscale_create_storage_pool(storage_pool_name, ma_values)

        status = self.wait_for_completion(storage_pool_name)
        self.log.info("Storage Pool status %s", status)
        response = self.get_storage_pool_details(storage_pool_name)
        self.log.info(response)

        return status, response

    def delete_storage_pool(self, storage_pool_name):
        """
        Deletes a hyperscale storage pool
        Args:
           storage_pool_name (str) -- name of the storage pool

        Return:
            flag (int) -- Response code received from the Rest API call
            response (str) -- Response received from the Rest API call
       """

        self.commcell.storage_pools.delete(storage_pool_name)

    def get_associated_storage_policies(self, storage_pool_name):
        """
           Get a list of Storage Policies associated to a hyperscale Storage pool's GDSP

           Args:
               storage_pool_name (str) -- Name of the storage pool

           Return:
               rows (list) -- List of associated storage policies
        """
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        gdsp = storage_pool_details.storage_pool_id
        query = """select name from archgroup where id in (select archgroupid from archgroupcopy where SIDBStoreId =  
                (select sidbstoreid from archGroupCopy where archgroupid  =  
                (select gdspid from MMSDSStoragePool where GDSPId = '{0}')))  
                and id <> (select gdspid from MMSDSStoragePool where GDSPId = '{0}')""".format(
            gdsp)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        return rows

    def check_if_storage_pool_is_present(self, storage_pool_name):
        """
            Check if s hyperscale storage pool is present or not

            Args:
                storage_pool_name (str) -- Name of the storage pool

            Return:
                 Boolean -- True or False based on whether the storage pool is present or not
                """
        self.commcell.storage_pools.refresh()
        return self.commcell.storage_pools.has_storage_pool(storage_pool_name)

    def get_associated_mas(self, storage_pool_name):
        """
               Get all MAs associated with a hyperscale storage pool

               Args:
                   storage_pool_name (str) -- Name of the storage pool

               Return:
                    rows (list) -- List of associated MAs
               """
        if self.check_if_storage_pool_is_present(storage_pool_name):
            storage_pool_details = self.get_storage_pool_details(storage_pool_name)
            storage_pool_properties = storage_pool_details._storage_pool_properties
            storage_pool_node_detail_list = storage_pool_properties['storagePoolDetails'][
                'storagePoolNodeDetailsList']
            ma_list = []
            for details in storage_pool_node_detail_list:
                data = [details['mediaAgent']['mediaAgentName']]
                if data not in ma_list:
                    ma_list.append(data)
            self.log.info("Node detils %s", ma_list)
            return ma_list
        return []

    def clean_up_ma(self, ma1, storage_pool_name):
        """
            Cleans up all data disks and peer files from an MA
            Args:
                ma1 (str) -- Name of primary MA that is being cleaned up
                storage_pool_name(str) --> Storage Pool name
            Return:
                Exception
        """
        # SSH to MA1
        count = 5
        ma_session = None
        status = False
        while count >= 0:
            try:
                ma_session = Machine(ma1, self.commcell)
                status = True
            except Exception as exp:
                result = str(exp)
                self.log.exception("Exception occurred connecting to host %s ", result)
                count -= 1
                status = False
                self.log.info("Trying again")
                time.sleep(90)
                pass
            if status:
                break

        # Run echo "y" | gluster v stop storage_pool_name
        output = ma_session.execute_command("echo \"y\" | gluster v stop " + storage_pool_name)
        self.log.info(output.output)

        # Run echo "y" | gluster v delete storage_pool_name
        output = ma_session.execute_command("echo \"y\" | gluster v delete " + storage_pool_name)
        self.log.info(output.output)

        # Run for i in `lsblk | grep -i /ws/ | awk '{print $7}'` ; do rm -fr $i/* ; ls -l $i ; done
        self.log.info("Cleaning all /ws/disk drives")
        output = ma_session.execute_command(
            "for i in `lsblk | grep -i /ws/ | awk '{print $7}'` ; do rm -fr $i/* ; ls -l $i ; done")
        self.log.info(output.output)

        # gluster peer detach sds
        self.log.info("Detaching peers")
        peers = ma_session.execute_command("gluster peer status | grep 'Hostname' | awk '{print $2}'")
        peers = peers.output.split("\n")
        peers = peers[:len(peers) - 1]
        for peer in peers:
            self.log.info("Peer %s", peer)
            output = ma_session.execute_command("echo \"y\" | gluster peer detach " + peer)
            self.log.info(output.output)

        # cleaning fstab
        self.log.info("Cleaning fstab entry")
        ma_session = Machine(ma1, self.commcell)
        command = "sed -i '/{0}/d' /etc/fstab".format(storage_pool_name)
        self.log.info(command)
        ma_session.execute_command(command)

        # commvault restart
        self.log.info("Restarting MA services")
        try:
            output = ma_session.execute_command("commvault restart")
        except Exception as exp:
            self.log.info("Exception %s", exp)
            self.log.info("Restarted Services")
            pass
        time.sleep(60)

    def reassociate_all_associated_subclients(self, storage_pool_name, sql_login, sql_sq_password):
        """
            Reassociates all subclients associated with a storage policy to none assigned
            Args:
                storage_pool_name (str) -- Name of the Storage pool which is to be cleaned up
                sql_login       (str) -- SQL login name
                sql_sq_password (str) -- Password to log into the SQL server
            Return:
                Exception
        """

        # Find all associated storage policies with this GDSP
        sstorage_policy_list = self.get_associated_storage_policies(storage_pool_name)
        self.log.info("Associated storage policies: {0}".format(sstorage_policy_list))

        self.log.info(str(len((sstorage_policy_list[0])[0])))

        if not (sstorage_policy_list[0])[0]:
            return

        for storage_policy in sstorage_policy_list[0]:
            storage_policy_obj = self.commcell.storage_policies.get(storage_policy)

            # update archgroupid = 1 for all associated subclients to sp.
            query = """update app_application 
                            set dataArchGrpID = 1, logArchGrpID = 1 
                            where dataArchGrpID = {0}""".format(storage_policy_obj.storage_policy_id)
            # self.log.info("QUERY: %s", query)
            self.execute_update_query(query, sql_login, sql_sq_password)

            # Delete associated storage policies with this GDSP
            try:
                self.commcell.storage_policies.delete(storage_policy)
            except Exception as err:
                self.log.error("failed to delete SP %s" % err)

    def clean_up_storage_pool(self, storage_pool_name, sql_login, sql_sq_password):
        """
            Cleans up all entries left by a pre-existing storage pool
            Args:
                storage_pool_name (str) -- Name of the Storage pool which is to be cleaned up
                sql_login (str)       --  SQL login name
                sql_sq_password (str) -- Password to log into the SQL server
            Return:
                Exception
        """

        status = self.check_if_storage_pool_is_present(storage_pool_name)

        if status is False:
            self.log.info("Storage pool not present, aborting clean up")
            return
        gluster_name = self.gluster_name_for_storage_pool(storage_pool_name)
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        library_name = storage_pool_details._storage_pool_properties['storagePoolDetails'][
            'libraryList'][0]['library']['libraryName']
        mas = self.get_associated_mas(storage_pool_name)

        # Reassociate all associated subclients
        self.reassociate_all_associated_subclients(storage_pool_name, sql_login, sql_sq_password)

        # Delete Storage Pool
        self.delete_storage_pool(storage_pool_name)

        # Delete Storage Target
        try:
            self.commcell._disk_libraries = None
            self.commcell.disk_libraries.delete(library_name)
        except Exception as err:
            self.log.error("Failed to delete library %s" % err)

        # SSH to MA1
        for media_agent in mas:
            ma_session = Machine(media_agent[0], self.commcell)
            # Unmounting glus mount point and removing folder
            command = "df -h | grep '" + str(gluster_name) + "'| awk '{print $6}'"
            self.log.info(command)
            output = ma_session.execute_command(command)
            self.log.info(output.formatted_output)
            command = "umount {0}".format(output.formatted_output)
            self.log.info(command)
            output = ma_session.execute_command(command)
            self.log.info(output)

            # removing /ws/glus_
            self.log.info("Removing glus directory")
            command = "for i in `ls /ws | grep -i glus | awk '{print $1}'` ; do rm -fr /ws/$i ; done"
            self.log.info(command)
            output = ma_session.execute_command(command)
            self.log.info(output.exception)

        for media_agent in mas:
            self.log.info("Cleaning up {0}".format(media_agent[0]))
            self.clean_up_ma(media_agent[0], gluster_name)

    def check_brick_flag_status(self, media_agent):
        """
            Check if the health status is proper for a particular MA's bricks or not
         Args:
            media_agent (str) -- Name of the MA

        Return:
            Boolean -- True if all daemons are active
            Exception if daemons are not active
        """
        hostid = self.get_host_id(media_agent)
        self.log.info(hostid)
        status = self.check_brick_health_status(int(hostid))
        if status is True:
            self.log.info("All brick daemons for {0} are active".format(media_agent))
        else:
            raise Exception("Not all brick daemons for {0} are active".format(media_agent))

    def reconfigure_storage_pool(self, storage_pool_name):
        """
                   Checks device accessibility before reconfigure and trigger reconfigure
                   after completion checks pool status and validates pool and accessibility of devices
                   validation of pool done while checking accessibility
                    Args:
                       storage_pool_name (name) -- storage pool name
                    Return:
                        True / False
                """
        self.log.info("Checking brick accessibility for pool %s ", storage_pool_name)
        if self.brick_accessibility(storage_pool_name) is False:
            return False, "Brick Not Accessible"
        nodes_before = list(self.get_all_nodes_hostids(storage_pool_name))
        nodes_before.sort()
        self.log.info("Nodes use for pool creation %s ", nodes_before)

        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        storage_pool_details.hyperscale_reconfigure_storage_pool(storage_pool_name)
        self.log.info("Waiting")
        time.sleep(1700)

        status = self.wait_for_completion(storage_pool_name)
        response = self.get_storage_pool_details(storage_pool_name)
        self.log.info(response)
        sp_status = self.validate_storage(storage_pool_name)
        nodes_after = list(self.get_all_nodes_hostids(storage_pool_name))
        nodes_after.sort()
        self.log.info("Nodes after reconfigure pool %s are %s ", storage_pool_name
                      , nodes_after)
        nodes_status = False
        if nodes_before == nodes_after:
            nodes_status = True
        device_accessible_status = self.brick_accessibility(storage_pool_name)
        status = status & device_accessible_status & nodes_status & sp_status

        return status, response

    def check_brick_available_status(self, hostid):
        """
            Checks the flag values for all bricks for a given MA
            Args:
                hostid (int) -- host id for the given MA
            Return:
                True / Exception
        """
        time.sleep(10)
        not_configured = 0
        bricks = self.get_all_bricks_for_hostid(hostid)
        for brick in bricks:
            self.log.info("Brick %s status;", brick[3])
            if int(brick[0]) != not_configured:
                self.log.error("{0} Brick does not have the flag set to 0".format(brick[3]))
                return False
            self.log.info("Brick %s is available and flag status %s ", brick[3], brick[0])

        return True

    def false_hosts(self, ma1):
        """
            Update host file with false hosts to break connectivity between
            th host and commserve
            Args:
                ma1(name) --> Media Agent
            Return:
                Nothing
        """
        ma1_session = Machine(ma1, self.commcell)
        self.log.info("Updating hosts, Storage pool failure "
                      "Updating Hosts with wrong details at %s"
                      , ma1)
        ma1_session.execute_command(" mkdir /root/false_hosts")
        ma1_session.execute_command(" mkdir /root/true_hosts")
        ma1_session.execute_command("cp -f /etc/hosts /root/true_hosts/hosts")
        ma1_session.execute_command("cp -f /etc/hosts /root/false_hosts/hosts")
        ma1_session.execute_command("sed -i -e 's/^/#/' /root/false_hosts/hosts")
        # Run cp -f /root/false_hosts/hosts /etc/hosts
        command = r"cp -f /root/false_hosts/hosts /etc/hosts"
        self.log.info(command)
        output = ma1_session.execute_command(command)
        self.log.info("Updated false host file")
        ma1_session.execute_command("rm -rf /root/false_hosts")
        self.log.info(output.output)

    def true_hosts(self, ma1):
        """
           Update host file with true hosts to bring up the connectivity b/w
           host and cs, which was broken by false host
           Args:
               ma1(name) --> Media Agent
           Return:
               Nothing
        """
        ma1_session = Machine(ma1, self.commcell)
        self.log.info("Updating Hosts with correct details at %s"
                      , ma1)

        command = r"cp -f /root/true_hosts/hosts /etc/hosts"
        # Run cp -f /root/true_hosts/hosts /etc/hosts
        self.log.info(command)
        output = ma1_session.execute_command(command)
        self.log.info(output.output)
        self.log.info("Updated correct host file")
        ma1_session.execute_command("rm -rf /root/true_hosts")

    def check_resolution_error(self, hostid):
        """
            Checks the flag, blockdevicehealth and brickhealth values for all bricks for a given MA
            Args:
                hostid (int) -- host id for the given MA
            Return:
                True / False
        """
        self.log.info("Checking Resolution error issue Flags,"
                      " brickHealth and BlockDeviceHealth set to 0 "
                      "after creation failure,resolution error")
        bricks = self.get_all_bricks_for_hostid(hostid)
        not_configured = 0
        for brick in bricks:
            self.log.info("Brick %s status;", brick[3])
            if int(brick[0]) != not_configured and int(brick[1]) != not_configured and int(brick[2]) != not_configured:
                self.log.info("Flags, brickHealth and BlockDeviceHealth are correctly set for {0}".format(brick[3]))
            else:
                self.log.info("For brick: %s  "
                              "Flag: %s, brickHealth: %s and "
                              "BlockDeviceHealth: %s", brick[3], brick[0], brick[1], brick[2])
                self.log.info("Resolution issue, Check hosts file")
                return True
        return False

    def get_device_controller_details(self, name):
        """
            Gets the all device details for the storage pool
            Args:
                name(name) --> storage pool name
            Return:
                devices details
        """
        if self.check_if_storage_pool_is_present(name):
            host_ids = self.get_all_nodes_hostids(name)
            bricks = self.get_all_bricks_for_hostid(host_ids[0])
            device_id = 0
            for brick in bricks:
                if len(brick[4]) != 0:
                    device_id = brick[4]
                    break
            self.log.info("Getting all devices for pool %s ", name)
            query = "select * from MMDeviceController where deviceId = {0}".format(device_id)
            self.log.info(query)
            self.csdb.execute(query)
            rows = self.csdb.fetch_all_rows()
            self.log.info(rows)
            return rows
        raise Exception("No device controller details for storage pool")

    def get_library_details(self, library_id):
        """
           Gets the library details for the given library id
           Args:
               library_id (int) -- library id for given library
           Return:
               library details
        """
        self.log.info("Getting library details for id %s ", library_id)
        query = "select * from MMLibrary as lib where lib.LibraryId = " + str(library_id)
        self.log.info(query)
        self.csdb.execute(query)
        library_details = self.csdb.fetch_one_row()
        if not library_details[0]:
            raise Exception('No library with id %s', library_id)
        self.log.info("Library Details  %s", str(library_details))
        return library_details

    def check_library_mount(self, library_id):
        """
             Gets the library mount details, accessibility and controller
              for the given library id
             Args:
                 library_id (int) -- library id for given library
             Return:
                 library details
                 """
        self.log.info("Getting library mount details, accessibility and controller "
                      "for the given library id %s ", library_id)
        query = """select mmd.DeviceControllerEnabled, mmd.DeviceAccessible, mmd.Folder  
                from MMlibrary as lib, MMMountPath as mp, MMMountPathToStorageDevice as mps, 
                 MMDeviceController as mmd  
                where lib.LibraryId = mp.LibraryId and mp.MountPathId = mps.MountPathId and 
                 mps.DeviceId = mmd.DeviceId and lib.LibraryId = {0}""".format(str(library_id))
        self.log.info(query)
        self.csdb.execute(query)
        mount_details = self.csdb.fetch_all_rows()
        self.log.info("Mount Deatils %s", mount_details)
        return mount_details

    def get_policy_details(self, gdspid):
        """
            Gets the policy details for the given pool
            Args:
                gdspid (int) -- gdspid for the policy to be created
            Return:
                policy details
                    """
        self.log.info("Getting Policy details for gdspid %s ", gdspid)
        query = "select * from archGroup where id = " + str(gdspid)
        self.log.info(query)
        self.csdb.execute(query)
        policy_details = self.csdb.fetch_one_row()
        self.log.info("Policy Details " + str(policy_details))
        return policy_details

    def get_dedup_details(self, name):
        """
            Gets the dedup details for the given pool
                Args:
                    name --> storage pool name
                Return:
                    dedup details
        """
        self.log.info("Getting dedup store details for %s", name)
        details = self.get_storage_pool_details(name)
        storage_pool_details = details._storage_pool_properties['storagePoolDetails']
        storage_pool_dedup = storage_pool_details['dedupDBDetailsList']
        dedup_details = []
        for detail in storage_pool_dedup:
            dedup_details.append(detail['storeName'])

        self.log.info("Dedup Details %s", dedup_details)
        return dedup_details

    def validate_dedup_store(self, gdspid):
        """
           Validate dedup present on ma's or not
                Args:
                    gdspid (int) -- gdspid for the policy to be created
                Return:
                    dedup details
        """
        self.log.info("Checking ddb stores on ma\n")
        query = """select Distinct pt.IdxAccessPathId, pt.ClientId, pt.Path, client.name, pt.OfflineReason
                from APP_Client as client, IdxAccessPath as pt, IdxSIDBStore as idx, archGroup as arc,  
                MMSDSStoragePool as storage, archCopySIDBStore sidb, IdxSIDBSubStore as sub  
                where storage.GDSPId = arc.id and arc.defaultCopy = sidb.CopyId and  
                sidb.SIDBStoreId = sub.SIDBStoreId  and sub.IdxAccessPathId = pt.IdxAccessPathId
                and client.id = pt.ClientId
                and storage.GDSPId = {0}""".format(str(gdspid))
        self.log.info(query)
        self.csdb.execute(query)
        details = self.csdb.fetch_all_rows()
        self.log.info("Dedup Partition details %s ", details)
        ma_dict = {}
        status = True
        for row in details:
            if row[3] not in ma_dict:
                ma_dict[row[3]] = []
            ma_dict[row[3]].append(row)
        for media_agent in ma_dict:
            row = ma_dict[media_agent]
            ma_session = Machine(media_agent, self.commcell)
            # checking /ws/ddb mount on node or not
            mount = r"df -h | grep '/ws/ddb' | awk '{print $1}'"
            mount_output = ma_session.execute_command(mount)
            if not mount_output.output:
                self.log.error("No mount path for /ws/ddb over %s ", media_agent)
                raise Exception("No mount path /ws/ddb")
            self.log.info("/ws/ddb mount path %s ", mount_output.output)
            # Restart services to check ddb copy presence, idx file gets created then,
            # CS sends information and thread creates that
            restart = r"commvault restart"
            try:
                ma_session.execute_command(restart)
            except Exception as exp:
                self.log.info("Exception %s", exp)
                self.log.info("Restarted Services")
                pass
            time.sleep(120)
            for entry in row:
                command = r"cd "
                self.log.info("Checking presence of partition %s on ma %s ", entry[2], entry[3])
                output = ma_session.execute_command(command + str(entry[2]))
                if output.exception:
                    status = False
                    self.log.error(output.exception)
                    # raise Exception("Partition %s not present on ma", entry[2])
                else:
                    self.log.info("Present")
                self.log.info("Checking Offline status")
                if int(entry[4]) != 0:
                    self.log.error("DDB partition %s offline with code %s on node %s", entry[2],
                                   entry[4], entry[3])
                else:
                    self.log.info("Online")
        return status

    def validate_storage(self, name):
        """
                   Validates the pool against library and policy
                   if policy present will have a dedup engine else no
                   Args:
                       name (str) -- Storage pool name
                   Return:
                       True/False
               """
        status = True
        details = self.get_storage_pool_details(name)
        self.log.info("Checking library state for pool %s", name)
        lib_id = details._storage_pool_properties[
            'storagePoolDetails']['libraryList'][0]['library']['libraryId']
        # lib_id = details[2]
        gdspid = details.storage_pool_id
        library_details = self.get_library_details(lib_id)
        if int(library_details[3]) != 1 and int(library_details[34]) != 46003:
            self.log.info("library offline with error %s and creation incomplete for "
                          "pool %s", library_details[34], name)
            status = False
        mount_details = self.check_library_mount(lib_id)
        if not mount_details[0]:
            self.log.info("No Mount path details for library %s", lib_id)
            status = False
        else:
            for mount in mount_details:
                self.log.info("Mount details %s, Accessible %s, DeviceEnabled %s", mount[2], mount[1],
                              mount[0])

        policy_details = self.get_policy_details(gdspid)
        if not policy_details[0]:
            self.log.info("Policy is not present, pool %s creation failed",
                          name)
            status = False
        if policy_details[0] and int(policy_details[13]) & 256 != 256:
            self.log.info("Not Global Policy for pool %s", name)
            status = False
        else:
            self.log.info("policy %s is Global Policy for pool %s", policy_details, name)
        dedup_details = self.get_dedup_details(name)
        if not dedup_details[0]:
            self.log.info("No dedup store present for %s ", name)
            status = False

        if not self.validate_dedup_store(gdspid):
            status = False

        if status is True:
            self.log.info("Library %s and Global policy %s and dedup store %s"
                          "up and present for pool %s ",
                          library_details, policy_details, dedup_details,
                          name)
            self.log.info("Storage pool %s created and up", name)
        return status

    def brick_accessibility(self, name):
        """
                   Check if bricks are accessible for pool creation
                   Args:
                       name (str) -- Storage pool name
                   Return:
                       library details
               """

        devices = self.get_device_controller_details(name)

        for device in devices:
            self.log.info("Node %s with Device id %s ", device[1], device[2])
            if int(device[11]) != 1:
                self.log.info("Node %s not accessible", device[1])
                return False
            self.log.info("Node %s accessible for creation", device[1])
        return True

    def resiliency(self, ma_list, setup):
        """
        Determine resiliency factor for the storage pool is correct or not at the time
        of creation
        Args:
            ma_list(list) --> List of media agent used
            setup(name) --> Type of setup (Standard/ Medium/ High)
        Return:
            True/False
        """
        self.log.info("Determining Resiliency")
        if setup.lower() == "standard":
            scale_out = 1
            self.log.info("Standard configuration, 3 nodes")
        elif setup.lower() == "medium":
            scale_out = 2
            self.log.info("Medium configuration, 6 nodes")
        elif setup.lower() == "high":
            scale_out = 3
            self.log.info("High configuration, 6 nodes")
        else:
            raise Exception("Setup configuration not present")

        count, min_used = 0, sys.maxsize
        node_bricks = {}
        for media_agent in ma_list:
            bricks = self.get_all_bricks_for_hostid(media_agent)
            node_bricks[media_agent] = len(bricks)
            req = int(len(bricks) // 2) * 2
            if req <= min_used:
                min_used = req
        node_bricks_sort = sorted(node_bricks.items(), key=lambda kv: kv[1])
        self.log.info("Bricks on nodes are %s", node_bricks_sort)
        if min_used == 0:
            self.log.error("Disk number not correct for node %s, minimum required are"
                           "%s ", node_bricks_sort[0][0], min_used)
            return False
        if node_bricks_sort[0][1] < min_used:
            self.log.error("Disk number not correct for node %s, minimum required are"
                           "%s ", node_bricks_sort[0][0], min_used)
            return False
        count += len(ma_list) * min_used
        if scale_out == 1 and count % 6 == 0:
            self.log.info("Standard for 3 Nodes, Disperse Factor 6, Redundancy Factor 2."
                          "Withstands loss of 2 drives or 1 node")
            return True
        if scale_out == 2 and count % 12 == 0:
            self.log.info("Medium for 6 Nodes, Disperse Factor 6, Redundancy Factor 2. "
                          "Withstands loss of 2 drives or 2 nodes")
            return True
        if scale_out == 3 and count % 12 == 0:
            self.log.info("High for 6 Nodes, Disperse Factor 12, Redundancy Factor 4."
                          " Withstands loss of 4 drives or 2 nodes")
            return True

        return False

    def get_all_nodes_hostids(self, storage_pool_name):
        """
            Gets a list of MA ids associated with a particular Storage Pool
            Args:
                storage_pool_name (str) -- Name of the Storage Pool
            Return:
                rows -- List of all MA ids associated with the given Storage Pool
        """
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        storage_pool_properties = storage_pool_details._storage_pool_properties
        storage_pool_node_detail_list = storage_pool_properties['storagePoolDetails']['storagePoolNodeDetailsList']
        node_id_list = []
        for details in storage_pool_node_detail_list:
            data = details['mediaAgent']['mediaAgentId']
            if data not in node_id_list:
                node_id_list.append(data)
        self.log.info("host ids %s for pool %s ", node_id_list, storage_pool_name)
        return node_id_list

    def get_disk_uuid(self, mas):
        """
        Get disk uuid information present on all the mas
        Args:
            mas(list) --> list of Media Agents
        Return:
            Dictionary of data
        """

        self.log.info("getting uuid information")
        command1 = r"lsblk | grep -i /ws/disk | awk '{print $1,$7}'"
        # sdb | awk '{print $2}'"
        ma_dict = {}
        for media_agent in mas:
            ma_session = Machine(media_agent, self.commcell)
            self.log.info("UUID's of disks for %s ", media_agent)
            self.log.info(command1)
            output1 = ma_session.execute_command(command1)
            disks = output1.output.split("\n")
            disks.remove("")
            self.log.info(disks)
            hostid = self.get_host_id(media_agent)
            for disk in disks:
                command2 = r"blkid | grep -i "
                data1, data2 = disk.split()
                command2 = command2 + data1 + " | awk '{print $2}'"
                # self.log.info("UUID for %s is", disk)
                output2 = ma_session.execute_command(command2)
                uuid = output2.output
                # self.log.info(output2.output)
                key = media_agent + "sds:" + data2 + "/ws_brick"
                ma_dict[key] = [media_agent, data1, uuid.replace("\n", ""), data2, hostid]

        for key in ma_dict:
            self.log.info("%s %s", key, ma_dict[key])
        return ma_dict

    def gluster_brick_information(self, media_agent):
        """
        Get bricks for the gluster
        Args:
            media_agent(name)  --> Media Agent
        Return:
            list of bricks
        """
        self.log.info("getting disk information for gluster")
        ma_session = Machine(media_agent, self.commcell)
        command2 = r"gluster v info | grep -i 'sds' | awk '{print $2}'"
        self.log.info(command2)
        time.sleep(200)
        output1 = ma_session.execute_command(command2)
        self.log.info(output1.output)
        if output1.output is "":
            self.log.error("no gluster on node %s ", media_agent)
            raise Exception("No Gluster information present on ma")
        bricks = output1.output.split("\n")
        bricks = bricks[:len(bricks) - 2]
        return bricks

    def verify_gluster_disk_uuids(self, media_agent, ma_dict):
        """
        Verify if same disks were used for gluster creation or not,
        disks which were present on each node and bricks after gluster creation
        Args:
            media_agent(name) --> Media Agent (for gluster information)
            ma_dict(dictionary) --> output of get_disk_uuid ( before creation)
        Return:
            True/False
        """

        self.log.info("getting disk uuids for Gluster")
        ma_session = Machine(media_agent, self.commcell)
        command = r"gluster v info | grep -i 'auth.allow:' | awk '{print $2}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        nodes = output.output.replace("\n", "").replace("sds", "").split(",")
        gluster_disk_ids = self.get_disk_uuid(nodes)
        bricks = self.gluster_brick_information(media_agent)
        status = True
        for brick in bricks:
            if gluster_disk_ids[brick][2] != ma_dict[brick][2] or gluster_disk_ids[brick][1] != ma_dict[brick][1]:
                self.log.info("For node %s, uuid are different %s != %s or disk name different %s != %s",
                              gluster_disk_ids[brick][0], gluster_disk_ids[brick][2],
                              ma_dict[brick][2], gluster_disk_ids[brick][3], ma_dict[brick][3])
                status = False
            self.log.info("Ids same for %s or %s, %s equals %s", gluster_disk_ids[brick][3],
                          gluster_disk_ids[brick][1],
                          gluster_disk_ids[brick][2], ma_dict[brick][2])
        self.log.info("brick ids checked and are same for gluster and node disk used")
        return status

    def gluster_healthy_brick_status(self, host_id, device_os):
        """
        Check gluster brick healthy or not from brick information
        Args:
            host_id(name) --> Client id for Media Agent
            device_os(name) --> path for brick
        Return:
            True/False
        """
        brick = self.node_brick_info(host_id, device_os)
        if not int(brick[0]) & 1:
            self.log.error("{0} Brick does not have the flag set to 1".format(brick[3]))
            return False
        if int(brick[1]) != 23:
            self.log.error("{0} Brick does not have the BrickHealth set to 23".format(brick[3]))
            return False
        if int(brick[2]) != 15:
            self.log.error("{0} Brick does not have the BlockDeviceHealth set to 15".format(brick[3]))
            return False
        self.log.info("Flags, brickHealth and BlockDeviceHealth correctly set for {0}".format(brick[3]))
        return True

    def gluster_disk_health(self, mas, ma_dict):
        """
        Verify if all the bricks are healthy for the gluster
        or not.
        Args:
            mas(list) --> list of media agents
             ma_dict(dictionary) --> output of get_disk_uuid ( before creation)
        Return:
            True/False
        """
        bricks = self.gluster_brick_information(mas[0][1])
        for brick in bricks:
            host_id = ma_dict[brick][4]
            disk = ma_dict[brick][3]
            self.log.info("Host id %s and disk %s ", host_id, disk)
            if not self.gluster_healthy_brick_status(host_id, disk):
                self.log.error("Flags, brickHealth and BlockDeviceHealth bad for %s and %s",
                               host_id, disk)
                return False
        return True

    def update_storage_policy(self, name, rename, sql_login, sql_sa_password):
        """
        Updating storage policy name for the given storage pool
        Args:
            name(name) --> Storage Pool Name
            rename(name) --> New name for Storage Pool
            sql_login(name) --> SQL login username
            sql_sa_password(name) --> SQL login password
        Return:
            New name for Storage Pool Policy
        """
        self.log.info("Updating GDSP for %s", name)
        storage_pool_details = self.get_storage_pool_details(name)
        gdspid = storage_pool_details.storage_pool_id
        # rename = "rename " + name
        query = """update archGroup  
                set name = '{0}' 
                where id = {1}""".format(str(rename), str(gdspid))
        self.execute_update_query(query, sql_login, sql_sa_password)
        self.commcell.refresh()
        self.commcell.storage_pools.refresh()
        return rename

    def get_hostname(self, host_id):
        """
        Get hostname for the hostid
        Args:
            host_id (int) -- hostid
        Return:
            hostname
        """
        self.log.info("Getting hostname for id %s", host_id)
        query = " select net_hostname from App_Client where id = " + str(host_id)
        self.log.info(query)
        self.csdb.execute(query)
        hostname = self.csdb.fetch_one_row()
        self.log.info("For id %s, hostname is %s", host_id, hostname[0])
        return hostname[0]

    def gluster_vol_info(self, media_agent):
        """
        Getting gluster volume information form node
        Args:
            media_agent (name) -- node
        Return:
            Volume information
        """
        self.log.info("getting volume information")
        ma_session = Machine(media_agent, self.commcell)
        command = "df -h | grep '/ws/glus'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        if output.output:
            glus_info = output.output.split("\n")[-2].split()
            self.log.info("Glus info %s", glus_info)
            return glus_info
        self.log.error("No gluster /ws/glus present on node %s", media_agent)
        return []

    def check_new_gluster_volume_size(self, media_agent):
        """
        Checking gluster volume permissible size
        Args:
            media_agent (name) -- host
        Return:
            True/False
        """
        glus_metadata_factor = 4  # percentage
        self.log.info("Verifying gluster volume size against glus_metadata_factor %s (percentage)",
                      glus_metadata_factor)
        glus_info = self.gluster_vol_info(media_agent)
        total_size = float(glus_info[1].lower().replace('g', ''))
        metadata_used = float(glus_info[2].lower().replace('g', ''))
        permissible_size = (total_size * glus_metadata_factor) / 100
        self.log.info("Total size of gluster %s and metadata used %s and permissible is %s"
                      , total_size, metadata_used, permissible_size)
        if metadata_used <= permissible_size:
            self.log.info("Gluster size permissible for node %s", media_agent)
            return True
        self.log.error("Gluster size not permissible for node %s", media_agent)
        return False

    def node_brick_info(self, host_id, device_os):
        """
        Get information for the brick present on the node
        Args:
            host_id(name) --> Clint id for the Media Agent
            device_os(name) --> path for the brick
        Return:
            Brick information if present (list)
        """

        query = """select flags,BrickHealth, BlockDeviceHealth, deviceospath, diskId, deviceId from mmdiskhwinfo where 
                mountpathusagetype = 2 and hostid = {0} and deviceOSPath = '{1}'""".format(str(host_id), str(device_os))
        self.log.info(query)
        self.csdb.execute(query)
        brick = self.csdb.fetch_one_row()
        if not str(brick[0]):
            self.log.info("Brick %s not present on node %s", device_os, host_id)
            raise Exception("Brick not present on node")
        self.log.info("Brick %s present on node %s ", brick, host_id)
        return brick

    def gluster_single_brick_info(self, media_agent, pool, brick):
        """
        Get brick information for the gluster if it is part of the gluster or not
        Args:
            media_agent(name) --> Media Agent
            pool(name) --> Storage Pool name
            brick(name) --> brick name
        Return:
            Brick information which is part of gluster (list)
        """

        self.log.info("Checing brick %s part of gluster or not", brick)
        pool = self.gluster_name_for_storage_pool(pool)
        ma_session = Machine(media_agent, self.commcell)
        search = media_agent.lower() + "sds:" + str(brick)
        command = "gluster v status " + str(pool) + " | grep '" + search + "'| awk '{print $1,$2,$5,$6}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        if output.exception:
            self.log.error("brick %s not part of gluster %s", brick, pool)
            # raise Exception("Brick not part of gluster")
            self.log.info(output.exception)
            return []
        brick_info = output.output.replace("\n", "").split()
        if len(brick_info) != 4:
            brick_info = []
            command = "gluster v status " + str(pool) + " | grep '" + search + "' -A 1 | awk '{print $1,$2,$4,$5}'"
            self.log.info(command)
            output = ma_session.execute_command(command)
            brick_output = output.output.replace("\n", "").split()
            brick_output[1] = brick_output[1] + brick_output[2]
            brick_info.extend([brick_output[0], brick_output[1], brick_output[4], brick_output[5]])
        self.log.info("Brick part of gluster %s \nBrick information %s ", pool, brick_info)
        return brick_info

    def check_gluster_brick_online(self, media_agent, pool, brick):
        """
            Verify disk part of gluster or not and online or not
            Args:
                media_agent(name) --> Media Agent of pool
                pool(name) --> storage pool name
                brick(name) --> brick to be verified
            Return:
                 True/False
        """
        gluster_brick = self.gluster_single_brick_info(media_agent, pool, brick)
        if len(gluster_brick) != 0:
            if str(gluster_brick[2]) == "Y":
                self.log.info("Brick %s online %s for gluster %s ", brick,
                              gluster_brick[2], pool)
                return True
        self.log.error("Brick %s not online for gluster %s ", brick, pool)
        return False

    def brick_flag_status(self, hostid, device_os):
        """
        Get flag information for the brick and identify over the flag
       Args:
           hostid(name) --> Clint id for the Media Agent
           device_os(name) --> path for the brick
       Return:
          Flag value
               """
        self.log.info("Checking Flag status for brick")
        brick = self.node_brick_info(hostid, device_os)
        flag = int(brick[0])
        if flag & 0:
            self.log.info("Flag %s Brick %s not configured for host %s ", flag, device_os, hostid)
            raise Exception("brick not configured")
        if flag & 1:
            self.log.info("Flag %s Brick %s configured for host %s ", flag, device_os, hostid)
        if flag & 2:
            self.log.info("Flag %s Replace initiated for\t Brick %s configured for host %s ", flag, device_os, hostid)
        if flag & 4:
            self.log.info("Flag %s Replace submitted for\t Brick %s configured for host %s ", flag, device_os, hostid)
        if flag & 8:
            self.log.info("Flag %s Replace Failed for \t Brick %s configured for host %s ", flag, device_os, hostid)
        if flag & 16:
            self.log.info("Flag %s Brick missing in health update, broken or offline for \t"
                          "Brick %s configured for host %s ", flag, device_os, hostid)
        self.log.info("Flag value %s", flag)
        return flag

    def replace_brick(self, media_agent, brick_id, pool):
        """
                Trigger disk replacement
                        Args:
                           pool (name) -- storage_pool_name
                           media_agent (name) -- media_agent
                           brick_id (int) -- diskId
                        Return:
                            status, response
        """
        self.log.info("Triggering Disk replacement over node %s"
                      "on brick %s", media_agent, brick_id)

        storage_details = self.get_storage_pool_details(pool)

        storage_details.hyperscale_replace_disk(brick_id, media_agent, pool)
        time.sleep(20)
        flag = self.get_disk_by_disk_id(brick_id)[0]
        self.commcell.storage_pools.refresh()
        response = self.get_storage_pool_details(pool)
        self.log.info(response)
        return flag, response

    def replace_log_status(self, media_agent):
        """
        Getting log details for replace brick
        flag = 0 no log lines,
        flag = 1 Replace Failure
        flag = 2 Replace Success
        """
        status = False
        flag = 0
        self.log.info("Getting Replace status response from node %s", media_agent)
        ma_session = Machine(media_agent, self.commcell)
        client = self.commcell.clients.get(media_agent)
        version_info = int(client.version)
        service_pack = client.service_pack
        self.log.info("Version information %s, Service Pack information %s", version_info, service_pack)
        sp_info = int(service_pack)
        self.log.info("Service Pack info %s", sp_info)
        command = "gluster v status | grep 'Status of volume:' | awk '{print $4}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        gluster_name = output.output.replace("\n", "")
        # command_fail = 'bzgrep "CVMAWorker::ProcessGlusterFSReplaceActionOperation"' \
        #                ' /var/log/commvault/Log_Files/CVMA.log | tail -4 | grep "Replace brick failure"'
        command_fail = 'bzgrep "CVMAGlusStoragePool::ReplaceDisk" ' \
                       '/var/log/commvault/Log_Files/CVMA.log | tail -8'
        self.log.info("Waiting")
        time.sleep(120)
        self.log.info(command_fail)
        output = ma_session.execute_command(command_fail)
        log_line = output.output.split("\n")
        log_line = log_line[: len(log_line) - 1]
        for index in range(len(log_line)):
            log_line[index] = " ".join(log_line[index].split())
            if "Replace brick failure" in log_line[index]:
                status = False
                flag = 1
            elif "Reset brick failure" in log_line[index] and int(sp_info) >= 16:
                status = False
                flag = 1
        if flag == 1:
            return status, log_line
        time.sleep(30)
        # command_success = 'bzgrep "CVMAWorker::ProcessGlusterFSReplaceActionOperation"' \
        #                   ' /var/log/commvault/Log_Files/CVMA.log | tail -4 |  grep "Replace Success"'

        command_success = 'bzgrep "CVMAGlusStoragePool::ReplaceDisk "' \
                          ' /var/log/commvault/Log_Files/CVMA.log | tail -7'
        self.log.info(command_success)
        self.log.info("Waiting")
        time.sleep(120)
        output = ma_session.execute_command(command_success)
        log_line = output.output.split("\n")
        log_line = log_line[: len(log_line) - 1]
        for index in range(len(log_line)):
            log_line[index] = " ".join(log_line[index].split())
            if "Replace Success" in log_line[index] and int(sp_info) < 16:
                status = True
                flag = 2
            elif "Reset Success" in log_line[index] and int(sp_info) >= 16:
                status = True
                flag = 2
        if flag == 2:
            return status, log_line
        self.log.info("Cannot grab success or failure status and logs")
        return status, []

    def replace_brick_available(self, media_agent):
        """
            Disk present for replace or not
            Args:
                media_agent(name) -- Node/media agent
            Return:
                True/False
        """
        status = True
        self.log.info("Checking raw disk availability or not ")
        time.sleep(120)
        ma_session = Machine(media_agent, self.commcell)
        command = r'bzgrep "CVMAGlusStoragePool::ReplaceDisk" ' \
                  r'/var/log/commvault/Log_Files/CVMA.log | tail -4 | ' \
                  r'grep " Replace brick failure, not able to find replacing brick"'
        self.log.info(command)
        output = ma_session.execute(command)
        if output.output:
            status = False
            log_line = output.output.split("\n")
            log_line = log_line[: len(log_line) - 1]
            for index in range(len(log_line)):
                log_line[index] = " ".join(log_line[index].split())
            self.log.info(log_line)
            return status
        self.log.info("Disk available for replacement")
        return status

    def get_disk_by_disk_id(self, disk_id):
        """
        Get disk information for the given disk Id
        Args:
            disk_id(name) --> Disk Id for the disk
        Return:
            disk information (list)
        """
        self.log.info("Getting brick information for diskId %s", disk_id)
        query = """select flags,BrickHealth, BlockDeviceHealth, deviceospath, hostId, deviceId from mmdiskhwinfo where  
                mountpathusagetype = 2 and diskId = {0}""".format(str(disk_id))
        self.log.info(query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        self.log.info(row)
        return row

    def heal_disk_entry(self, media_agent, new_path):
        """
        Check if disk replaced is healed or not, entry present in the directory
        /opt/commvault/iDataAgent/jobResults/ScaleOut/ReplaceBricksFile
        Args:
            media_agent(name) --> media Agent on which replace carried out
            new_path(name) --> disk path
        Return:
            True/False
        """
        self.log.info("Checking entry %s for disk heal", new_path)
        status = False
        ma_session = Machine(media_agent, self.commcell)
        command = "bzgrep '" + str(new_path) + "' /opt/commvault/iDataAgent/jobResults/ScaleOut/ReplaceBricksFile"
        self.log.info(command)
        output = ma_session.execute_command(command)
        if output.output:
            status = False
            self.log.info("Gluster heal in progress for %s", new_path)
            self.log.info(output.output)
            return status

        status = True
        self.log.info("Gluster heal complete for %s ", new_path)
        self.log.info(output.output)
        return status

    def check_fstab_entry(self, media_agent, disk):
        """
            Checking entry in /etc/fstab for the disk on given ma
            Args:
                media_agent (name) -- MediaAgent
                disk (name) -- Brick/disk on Ma
            Return:
                list output of entry
        """
        self.log.info("Checking Fstab entry %s on node %s", disk, media_agent)
        ma_session = Machine(media_agent, self.commcell)
        command = "bzgrep '" + str(disk) + "' /etc/fstab"
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        return output.output.replace("\n", "").split()

    def replace_fstab_entry(self, media_agent, old_entry, new_entry):
        """
        CHecks fstab entry being replaced or not after brick repalcement
        Args:
            media_agent(name) --> Media Agent
            old_entry(list) --> list input for old entry
            new_entry(list) --> list input for new entry
        Return:
            True/False
        """
        self.log.info(" Fstab entries for old disk %s and new disk %s", old_entry,
                      new_entry)

        fstab_replace_status = self.verify_fstab_replace_entry(media_agent, new_entry)
        if new_entry and fstab_replace_status:
            if old_entry[0] != new_entry[0]:
                res = "Disk Replacement success, Verified ids for new disk entry {0} " \
                      "\nRemoved old entry {1} and added new entry {2}".format(new_entry,
                                                                               old_entry, new_entry)
                self.log.info(res)
                return True, res
            if old_entry[0] == new_entry[0]:
                res = "old {0} " \
                      "same as new {1}. Verified the ids for the disk {2}".format(old_entry, new_entry, new_entry)
                self.log.info(res)
                return True, res
        if old_entry[0] == new_entry[0]:
            res = "Old entry {0} not removed from fstab".format(old_entry)
            self.log.info(res)
            # raise Exception("Old entry not removed")
            return False, res
        if not fstab_replace_status:
            res = "Ids not same for the entry and sd name mounted as"
            self.log.info(res)
            return False, res

    def verify_fstab_replace_entry(self, media_agent, fstab_info):
        """
            Verify same disk ids, for entry in fstab and mounted on gluster
            Args:
                media_agent(name) -- MediaAgent
                fstab_info (name string) -- fstab entry of disk
        """
        self.log.info("Verifying uuid for entry %s", fstab_info)
        ma_session = Machine(media_agent, self.commcell)
        command = "df -h | grep " + str(fstab_info[1])
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        sd_name = output.output.replace("\n", " ").split()
        self.log.info("For disk %s, mounted with sd name %s", fstab_info[1], sd_name[0])
        self.log.info("Getting uuid for %s", sd_name[0])
        command2 = "blkid " + str(sd_name[0])
        self.log.info(command2)
        output_id = ma_session.execute_command(command2)
        self.log.info(output_id.output)
        id_info = output_id.output.replace("\n", " ").split()
        self.log.info("Verifying ids same for %s which is mounted as %s", fstab_info[1], sd_name[0])
        if fstab_info[0] == id_info[1].replace('"', ""):
            self.log.info("Fstab entry for %s is same for mounted as %s with id as %s and %s", fstab_info[1],
                          sd_name[0], fstab_info[0], id_info[1])
            return True
        self.log.info("Fstab entry for %s is NOT same for mounted as %s with id as %s and %s", fstab_info[1],
                      sd_name[0], fstab_info[0], id_info[1])
        return False

    def gluster_heal_entries(self, media_agent, pool):
        """
            Checking Heal entries, value should reach 0 after heal process
            Args:
                media_agent(name) -- media Agent/ node
                pool(name) -- storage pool name
            Return:
                True/False
        """
        status = False
        self.log.info("Checking heal entries and making sure they turn to 0 for heal process")
        pool = self.gluster_name_for_storage_pool(pool)
        ma_session = Machine(media_agent, self.commcell)
        command = r"gluster v heal " + str(pool) + " info | grep 'Number of entries:'"
        while status is not True:
            self.log.info("Checking gluster heal entries: ")
            output = ma_session.execute(command)
            self.log.info(output.output)
            entries = output.output.replace("\n", " ").split()
            entry_sum = 0
            for entry in entries:
                if entry.isdigit():
                    entry_sum += entry_sum + int(entry)
            if entry_sum == 0:
                status = True
                break
            time.sleep(30)
        return status

    def verify_replace_gluster_size(self, before_replace, after_replace):
        """
                Checking gluster volume  size
                Args:
                    before_replace (list) -- gluster info before replace
                    after_replace (list) -- gluster info after replace
                Return:
                    True/False
                """
        self.log.info("Verifying Gluster size info before replacement request and after replacement request\n"
                      "Gluster size before replacement request %s \n Gluster size after replacement request %s"
                      , before_replace, after_replace)
        total_size_before = float(before_replace[1].lower().replace('g', ''))
        used_before = float(before_replace[2].lower().replace('g', ''))
        available_before = float(before_replace[3].lower().replace('g', ''))
        total_size_after = float(after_replace[1].lower().replace('g', ''))
        used_after = float(after_replace[2].lower().replace('g', ''))
        available_after = float(after_replace[3].lower().replace('g', ''))
        self.log.info("\nTotal gluster size before %s, used %s, available %s\n"
                      "Total gluster size after replace is %s, used %s, available %s\n", total_size_before,
                      used_before, available_before, total_size_after, used_after, available_after)

        if (total_size_before == total_size_after) and (used_before == used_after) and \
                (available_before == available_after):
            self.log.info("Gluster size same after replacement request ")
            return True
        self.log.info("Gluster size different after replace request")
        return False

    def get_subvolume_for_pool(self, storage_pool_name):
        """
            Getting subvolume names for the storage_pool
            Args:
                storage_pool_name (name) --> storage_pool_name
            Returns:
                sub_volume_name (list)
        """
        self.log.info("Getting subVolume names for SP %s", storage_pool_name)
        time.sleep(120)
        self.commcell.storage_pools.refresh()
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        host_ids = self.get_all_nodes_hostids(storage_pool_name)
        ids = ",".join(str(id) for id in host_ids)
        query = """ select distinct SubVolume 
                    from MMDiskHWInfo 
                    where hostId in ({0})""".format(str(ids))
        self.log.info(query)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        self.log.info("Sub Volumes for pool %s are %s", storage_pool_name, rows)
        return rows

    def get_all_bricks_for_subvolume(self, sub_volume):
        """
            Get all bricks belonging to a sub volume
            Args:
                sub_volume(name) --> SubVolume
            Returns:
                bricks(list)
        """
        self.log.info("Getting all bricks belonging to the sub volume %s", sub_volume)
        time.sleep(300)
        query = """select flags,BrickHealth, BlockDeviceHealth, deviceospath, hostId, deviceId, diskId 
                 from mmdiskhwinfo where  
                mountpathusagetype = 2 and SubVolume = '{0}'""".format(str(sub_volume))
        self.log.info(query)
        self.csdb.execute(query)
        bricks = self.csdb.fetch_all_rows()
        self.log.info("Bricks belonging to sub volume %s are %s", sub_volume, bricks)
        return bricks

    def ma_service_down(self, media_agent):
        """
            Making MA services for node offline and not accessible
            Args:
                media_agent(name) --> Media Agent
            Return:
                Machine Object
        """
        self.log.info("Making media agent services down for %s", media_agent)
        client = self.commcell.clients.get(media_agent)
        self.log.info("Bringing down Commvault Media Mount Manager MA services")
        client.execute_command("pkill CvMountd")
        self.log.info("Node %s Media Mount Manager  Down", media_agent)

    def ma_service_up(self, media_agent):
        """
            Making MA services for node online and accessible again
            Args:
                media_agent(name) --> Media Agent
        """
        self.log.info("Making media agent services up for %s", media_agent)
        ma_session = Machine(media_agent, self.commcell)
        client = self.commcell.clients.get(media_agent)
        self.log.info("Bringing up Commvault Media Mount Manager MA services")
        client.start_service("CvMountd")
        try:
            output = ma_session.execute_command("commvault restart")
        except Exception as exp:
            self.log.info("Exception %s", exp)
            self.log.info("Restarted Services")
            pass
        time.sleep(120)
        self.log.info("Node %s Commvault Media Mount Manager Up", media_agent)

    def check_restore_folders(self, media_agent, backup_path, restore_path):
        """
            Create machine object for media_agent and compare backup and restore
            Here source and destination machine are same
            Args:
                media_agent(name) -> Media_Agent
                backup_path(name) -> source destination of file
                restore_path(name) -> restore destination of file
            Return:
                Output(list)
        """
        self.log.info("Verifying Restore data")
        ma_session = Machine(media_agent, self.commcell)
        destination_ma = Machine(media_agent, self.commcell)
        output = ma_session.compare_files(destination_ma, backup_path, restore_path)
        self.log.info(output)
        return output

    def gluster_mount(self, media_agent):
        """
            Check /ws/glus mounted on ma or not
            Args:
                media_agent(name) --> media Agent
            Return:
                True/False
        """
        self.log.info("checking /ws/glus mount on %s", media_agent)
        ma_session = Machine(media_agent, self.commcell)
        # checking /ws/ddb mount on node or not
        mount = r"df -h | grep '/ws/glus' | awk '{print $1}'"
        mount_output = ma_session.execute_command(mount)
        if not mount_output.output:
            self.log.error("No mount path for /ws/glus over %s ", media_agent)
            return False
        self.log.info("/ws/glus is mounted")
        return True

    def unmount_brick(self, media_agent, disk, pool):
        """
            Un mounting brick from gluster
            Args:
                media_agent(name) --> node
                disk(name) --> brick to un mount
                pool(name) --> storage pool
        """
        self.log.info("Un Mounting %s from gluster on node %s ", disk, media_agent)
        brick_info = self.gluster_single_brick_info(media_agent, pool, disk)
        pid = brick_info[-1]
        self.log.info("Pid to kill %s and un mount disk %s", pid, disk)
        ma_session = Machine(media_agent, self.commcell)
        if pid != "N/A":
            command = "kill -9 " + str(pid)
            self.log.info(command)
            output = ma_session.execute_command(command)
            self.log.info(output.output)
            ma_session.execute_command("umount " + str(disk))

        self.log.info("Checking gluster brick status")
        status = self.check_gluster_brick_online(media_agent, pool, disk)
        if not status:
            self.log.info("gluster disk %s not online", disk)
        try:
            output = ma_session.execute_command("commvault restart")
        except Exception as exp:
            self.log.info("Exception %s", exp)
            self.log.info("Restarted Services")
            pass
        time.sleep(120)
        self.log.info("gluster disk un mounted")

    def mount_brick(self, media_agent, disk, pool):
        """
            Mounting disk back to gluster
            Args:
                media_agent(name) --> node
                disk(name) --> brick to un mount
                pool(name) --> storage pool
        """
        self.log.info("Mounting %s back to gluster on node %s ", disk, media_agent)
        gluster_name = self.gluster_name_for_storage_pool(pool)
        self.log.info("Checking gluster brick status")
        status = self.check_gluster_brick_online(media_agent, pool, disk)
        if not status:
            self.log.info("gluster disk %s not online", disk)
        ma_session = Machine(media_agent, self.commcell)
        command = "mount " + str(disk)
        ma_session.execute_command(command)
        self.log.info("Restarting gluster to get brick online")

        # Run echo "y" | gluster v stop storage_pool_name
        output = ma_session.execute_command("echo \"y\" | gluster v stop " + str(gluster_name))
        self.log.info(output.output)
        # gluster v start storage_pool_name
        output = ma_session.execute_command("gluster v start " + str(gluster_name)
                                            + " force")
        self.log.info(output.output)
        self.log.info("Checking gluster brick status")
        status = self.check_gluster_brick_online(media_agent, pool, disk)
        if not status:
            self.log.info("gluster disk %s not online", disk)
        try:
            output = ma_session.execute_command("commvault restart")
        except Exception as exp:
            self.log.info("Exception %s", exp)
            self.log.info("Restarted Services")
            pass
        time.sleep(120)
        self.log.info("gluster disk mounted back")

    def verify_reset_brick(self, media_agent, pool, disk):
        """
            Verifying reset-brick used over SP16+
            Args:
                media_agent(name) --> Media Agent/ Node
                pool(name) --> Storage Pool name
                disk(name) --> Disk replaced
            Return:
                True/False
        """
        self.log.info("Verifying reset-brick used over SP16+")
        pool = self.gluster_name_for_storage_pool(pool)
        disk_name = str(media_agent) + "sds:" + str(disk) + "/ws_brick"
        regex = "reset-brick " + str(pool) + " " + str(disk_name)
        command = "bzgrep '" + str(regex) + "' /var/log/commvault/Log_Files/CVMA.log | tail -2"
        ma_session = Machine(media_agent, self.commcell)
        client = self.commcell.clients.get(media_agent)
        version_info = int(client.version)
        service_pack = client.service_pack
        self.log.info("Version information %s, Service Pack information %s", version_info, service_pack)
        sp_info = int(service_pack)
        output = ma_session.execute_command(command)
        log_info = output.output.split("\n")
        log_info = log_info[:len(log_info) - 1]
        if len(output.output) != 0:
            for log in log_info:
                self.log.info(log)
            if int(sp_info) >= 16:
                self.log.info("Service pack version %s higher than 16", sp_info)
                self.log.info("Reset command used")
                return True
            self.log.info("Service pack version %s lower than 16", sp_info)

        self.log.info("Reset command not used")
        return False

    def brick_data_being_written(self, media_agent, disk):
        """
            Get brick data written while backup is running
            Args:
                media_agent(name) --> Media Agent
                disk(name) --> brick of gluster
            Return:
                list of data and disk
        """
        self.log.info("Check disk %s is in use or not", disk)
        ma_session = Machine(media_agent, self.commcell)
        command = "du " + str(disk) + " | awk '{print $1,$2}'"
        output = ma_session.execute_command(command)
        res = output.output.split("\n")
        res = res[: len(res) - 1]
        check = res[-1].split()
        self.log.info("Data usage over %s is %s", disk, check)
        return check

    def check_brick_in_use(self, media_agent, disk):
        """
            Check if brick under use or not
            Args:
                media_agent(name) --> Media Agent
                disk(name) --> brick of gluster
            Return:
                True/False
        """
        old = self.brick_data_being_written(media_agent, disk)
        self.log.info("Wait some time")
        time.sleep(30)
        new = self.brick_data_being_written(media_agent, disk)
        if old[0] != new[0]:
            self.log.info("Disk %s under usage, old %s and new %s", disk, old, new)
            return True
        return False

    def check_gluster_metadata(self, storage_pool_name):
        """
            Check gluster metadata present after pool expansion
            Args:
                storage_pool_name(name) --> Storage Pool Name
            Return:
                True/Exception
        """
        self.log.info("Checking gluster meta data over expansion")
        commserve = self.commcell.commserv_name
        ma_session = Machine(commserve, self.commcell)
        media_agents = self.get_associated_mas(storage_pool_name)
        client = self.commcell.clients.get(self.commcell.commserv_name)
        regex = "C:\\temp\\glusterd__{0}.tar.gz"
        regex2 = client.job_results_directory + "\\ScaleOut\\glusterd__{0}.tar.gz"
        for media_agent in media_agents:
            hostid = self.get_host_id(media_agent[0])
            self.log.info("Checking gluster metadata for hostid %s ", hostid)
            self.log.info("checking path %s or %s", regex2.format(hostid), regex2.format(hostid))
            if not ma_session.check_file_exists(regex.format(hostid)) and \
                    not ma_session.check_file_exists(regex2.format(hostid)):
                self.log.error("%s not present for %s", regex.format(hostid), media_agent[0])
                self.log.error("OR %s not present for %s", regex2.format(hostid), media_agent[0])
                raise Exception("Glustermeta data not present")
        return True

    def get_ddb_hosts(self, storage_pool_name):
        """
        Get hosts, for the ddb partitions
        Args:
            storage_pool_name(name) --> Storage Pool Name
        Return:
            list of host ids
        """
        self.log.info("Getting hosts for dedup partitions present for pool %s", storage_pool_name)
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        gdsp = storage_pool_details.storage_pool_id
        query = """
            select distinct substore.ClientId from 
            IdxSIDBSubStore as substore, IdxSIDBStore as idx, archGroup as arc, MMSDSStoragePool as storage, 
            archCopySIDBStore sidb 
            where storage.GDSPId = arc.id and
            arc.defaultCopy = sidb.CopyId and
            sidb.SIDBStoreId = idx.SIDBStoreId and 
            idx.SIDBStoreId = substore.SIDBStoreId and  
            storage.GDSPId = {0}
        """.format(gdsp)
        self.log.info(query)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        self.log.info("Hosts with ddb partitions present are %s", rows)
        return rows

    def check_control_node(self, media_agent):
        """
        Check controlnode or datanode for host
        Args:
            media_agent(name) --> Media Agent
        Return:
            True/False
        """
        self.log.info("Checking control node or not")
        ma_session = Machine(media_agent, self.commcell)
        command = "bzgrep 'HyperScaleNodeType' /etc/CommVaultRegistry/Galaxy/Instance001/MediaAgent/.properties |" \
                  " awk '{print $2}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        formatted_output = output.output.replace("\n", "").lower()
        if formatted_output == 'control':
            self.log.info("Host %s is control node", media_agent)
            return True
        self.log.info("Host %s is data node", media_agent)
        return False

    def check_ddb_move_job(self, storage_pool_name):
        """
        Check job initiated and status for ddb move over expansion
        Args:
            storage_pool_name(name) --> Storage Pool Name
        Return:
            list of jobs
        """
        self.log.info("checking ddb move job over expansion")
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        gdsp = storage_pool_details.storage_pool_id
        query = """
        select jobId, status from JMAdminJobStatsTable where opType = 99 and archGrpID = {0}
        """.format(gdsp)
        self.log.info(query)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        if len(rows[0][0]) > 0:
            self.log.info("DDB Move job initiated with jobs %s ", rows)
            for job in rows:
                status = job[1]
                if int(status) == 1:
                    self.log.info("Job %s is completed with status %s", job[0], job[1])
                else:
                    self.log.info("Job %s with status %s", job[0], job[1])
            return True
        self.log.info("DDB Move job not initiated")
        return False

    def check_sds_entry(self, entry, media_agent):
        """
        check sds entry populated over ma
        Args:
            entry(name) --> media agent entry to check
            media_agent(name) --> Media Agent on which to check
        Return:
            True/False
        """
        self.log.info("Checking %s sds entry on %s", entry, media_agent)
        regex = entry + "sds"
        ma_sessin = Machine(media_agent, self.commcell)
        command = "bzgrep '{0}' /etc/hosts".format(regex)
        self.log.info(command)
        output = ma_sessin.execute_command(command)
        if output.output:
            self.log.info("Entry %s sds present on host %s", entry, media_agent)
            return True
        self.log.info("Entry %s sds not present on host %s", entry, media_agent)
        return False

    def bkp_job_details(self, job_id):
        """
        Get Job details for the job id
        Args:
            job_id(name) --> Job Id
        Return:
            job details
        """
        self.log.info("Getting job details for job %s", job_id)
        job_id = int(job_id)
        job_obj = self.commcell.job_controller.get(job_id)
        details = job_obj._get_job_details()
        job_details = details['jobDetail']['generalInfo']
        for detail in job_details:
            self.log.info("Details: %s --> %s", detail, job_details[detail])

        return job_details

    def admin_job_details(self, job_id):
        """
        Get Job details for the job id
        Args:
            job_id(name) --> Job Id
        Return:
            job details
        """
        self.log.info("Getting job details for job %s", job_id)
        job_id = int(job_id)
        job_obj = self.commcell.job_controller.get(job_id)
        details = job_obj._get_job_summary()
        # job_details = details['jobDetail']['generalInfo']
        for detail in details:
            self.log.info("Details: %s --> %s", detail, details[detail])

        return details

    def kill_sds_network(self, media_agent):
        """
        Kill sds network for media agent
        2 commands,
        ---->1. eth interface
        ---->2. ens interface
        ---->3. bond1 interface
        Args:
            media_agent(name) --> Media Agent
        """
        self.log.info("Killing sds network for %s", media_agent)
        networks = ['ens224', 'eth1', 'bond1']
        ma_session = Machine(media_agent, self.commcell)
        for network in networks:
            command = "ip link set {0} down".format(network)
            self.log.info("Command: %s", command)
            ma_session.execute_command(command)

    def start_sds_network(self, media_agent):
        """
        Restore and start sds network back online again
        2 commands
        ---->1. eth interface
        ---->2. ens interface
        ---->3. bond interface
        Args:
            media_agent(name) --> Media Agent
        """
        self.log.info("Bringing sds network online")
        networks = ['ens224', 'eth1', 'bond1']
        ma_session = Machine(media_agent, self.commcell)
        for network in networks:
            command = "ip link set {0} up".format(network)
            self.log.info("Command: %s", command)
            ma_session.execute_command(command)
        ma_session.execute_command("systemctl restart network")

    def fill_gluster_volume(self, media_agent):
        """
        Fill gluster volume with random generated data
        bs --> buffer size (1Gb = 1073741824)
        Args:
            media_agent(name) --> Media Agent
        Return:
            Response
        """
        self.log.info("Filling gluster with random data")
        glus_info = self.gluster_vol_info(media_agent)
        available_size = int(glus_info[3].lower().replace('g', ''))
        self.log.info("Gluster availble size is %s", available_size)
        reserve_space = 2
        fill_size = available_size - reserve_space
        self.log.info("Filling gluster with size %s and leaving reserve space %s", fill_size, reserve_space)
        ma_session = Machine(media_agent, self.commcell)
        self.log.info("Creating dummy folder under /ws/glus")
        ma_session.execute_command("mkdir /ws/glus/dummy")
        for i in range(fill_size):
            command = "dd if=/dev/zero of=/ws/glus/dummy/file{0}.txt count=1 bs=1073741824".format(i)
            self.log.info(command)
            ma_session.execute_command(command)
        self.log.info("FIlled gluster with size %s", available_size)

    def gluster_name_for_storage_pool(self, name):
        """
        Get gluster name, for the storage pool
        Args:
            name(str) --> storage Pool
        Return:
            Gluster_name(str)
        """
        self.log.info("Gluster name for the pool %s", name)
        details = self.get_storage_pool_details(name)
        gdsp = details.storage_pool_id
        media_agents = self.get_associated_mas(name)
        media_agent = media_agents[0][0]
        count = 5
        ma_session = None
        while count >= 0:
            try:
                ma_session = Machine(media_agent, self.commcell)
                status = True
            except Exception as exp:
                result = str(exp)
                self.log.exception("Exception occurred connecting to host %s ", result)
                count -= 1
                status = False
                self.log.info("Trying again")
                time.sleep(90)
                pass
            if status:
                break
        command = "gluster v status | grep 'Status of volume:' | awk '{print $4}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        gluster_name = output.output.replace("\n", "")
        return gluster_name

    def sd_name_for_disk(self, media_agent, disk):
        """
        Getting sd/scuzzy name for the disk
        Args:
            media_agent --> Media Agent
            disk --> disk to format
        """
        self.log.info("Getting sd_name or scuzzy name for the disk %s", disk)
        ma_session = Machine(media_agent, self.commcell)
        command = "lsblk | grep '{0}'".format(disk)
        self.log.info(command)
        output = ma_session.execute_command(command)
        sd_name = output.output.split("\n")[0].split()[0]
        self.log.info("sd name for disk %s is %s", disk, sd_name)
        return sd_name

    def formatting_replaced_disk(self, media_agent, scuzzy_name):
        """
        Formatting disk and removing all data and file system present over it
        Args:
            media_agent --> Media Agent
           scuzzy_name --> sd name for the disk to be formatted
        """
        self.log.info("Formatting replaced disk %s as sd name on media agent %s", scuzzy_name, media_agent)
        ma_session = Machine(media_agent, self.commcell)
        command1 = "dd if=/dev/zero of=/dev/{0}  bs=512  count=1".format(scuzzy_name)
        self.log.info(command1)
        output = ma_session.execute_command(command1)
        self.log.info(output.output)

    def size_of_disk_used_in_storagepool(self, media_agent, storage_pool_name):
        """
        Getting disk size for /ws/disk on the node used in storage pool
        Args:
            media_agent --> Media agent
            storage_pool_name --> storage pool name
        Return:
            size of disk used in gluster (int)
        """
        self.log.info("Getting disk size for %s on node %s", storage_pool_name, media_agent)
        ma_session = Machine(media_agent, self.commcell)
        command = "df -h | grep '/ws/disk'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        disks = output.formatted_output
        disks = sorted(disks, key=itemgetter(3))
        gluster_disk = []
        for disk in disks:
            gluster_disk_info = self.gluster_single_brick_info(media_agent, storage_pool_name, disk[5])
            if len(gluster_disk_info) != 0:
                gluster_disk = disk
                break
        disk_size = gluster_disk[3]
        self.log.info("/ws/disk size used in gluster %s is %s for gluster disk %s",
                      storage_pool_name, disk_size, gluster_disk)
        return disk_size

    def disk_for_vertical_scaleout(self, media_agent, storage_pool_name):
        """
           Identifying disks available on node for Vertical scale out
           Args:
               media_agent --> Media Agent
               storage_pool_name --> storage pool name
            Return:
                list of disks available(list)
        """
        self.log.info("Disks available on node %s for vertical scale out", media_agent)
        ma_session = Machine(media_agent, self.commcell)
        command = "echo \"q\" | (cd /opt/commvault/Base && ./CVSDS -d)"
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        disk_size_req = self.size_of_disk_used_in_storagepool(media_agent, storage_pool_name)
        disk_size_req = float(''.join(list(value for value in disk_size_req if value.isdigit())))
        disk_available = []
        for data in output.formatted_output:
            if '/dev/sd' in ''.join(data) \
                    and float(''.join(list(value for value in data[1] if value.isdigit()))) >= disk_size_req:
                data.append(media_agent)
                disk_available.append(data)
        self.log.info("disk available for vertical scale out are %s", disk_available)
        return disk_available

    def minimum_disks_for_vertical_scaleout(self, storage_pool_name):
        """
        Getting minimum number of disks required to be present on each node for
        vertical scale out
        Args:
            storage_pool_name --> storage pool name
        Return:
            dictionary of hosts with disks, min required for vertical scaleout
            0 --> False(no apt disks present for Vertical scaleout)
        """
        self.log.info("Verifying nodes have minimum number of disks available for vertical scale out")
        media_agents = self.get_associated_mas(storage_pool_name)
        self.log.info("Associated media agents are %s", media_agents)
        for media_agent in media_agents:
            self.log.info(media_agent[0])
        vertical_scaleout_disks = {}
        host, min_used = 'media_agent', sys.maxsize
        for media_agent in media_agents:
            if media_agent[0] not in vertical_scaleout_disks:
                vertical_scaleout_disks[media_agent[0]] = []
            vertical_scaleout_disks[media_agent[0]] = self.disk_for_vertical_scaleout(media_agent[0],
                                                                                      storage_pool_name)
            req = len(vertical_scaleout_disks[media_agent[0]])
            req = (req // 2) * 2
            if req <= min_used:
                min_used = req
            if req == 0:
                host = media_agent[0]
        if min_used == 0:
            self.log.error("Not correct number of disks present on host %s for vertical scaleout, disk count", host,
                           len(vertical_scaleout_disks[host]))
            return vertical_scaleout_disks, min_used
        self.log.info("Minimum number of disk required for vertical scaleout %s", min_used)
        return vertical_scaleout_disks, min_used

    def adding_disk_vertical_scaleout(self, media_agent, disk):
        """
        Adding disks for vertical scale out
        Args:
            media_agent --> Media agent
            disk --> disk to be added for VS (/dev/sdx)
        Return:
            True/False
        """
        self.log.info("Adding disk %s over media agent %s", disk, media_agent)
        ma_session = Machine(media_agent, self.commcell)
        command = "echo -e \"{0}\\nq\" | (cd /opt/commvault/Base && ./CVSDS -d)".format(disk)
        self.log.info(command)
        output = ma_session.execute_command(command)
        self.log.info(output.output)
        if (HYPERSCALE_CONSTANTS['vertical_scaleout_add_disk_success'] in ''.join(output.formatted_output[-1]).lower()) \
                and (disk in ''.join(output.formatted_output[-1]).lower()):
            self.log.info("Added disk, Configure status for disk %s is %s", disk, output.formatted_output[-1])
            return True
        self.log.error(output.output)
        self.log.info("configure status False")
        return False

    def mount_details(self, media_agent, path):
        """
        Get mount details for the path, for linux ma
        Args:
            media_agent --> media agent
            path --> path (which is to be verified and mount details fetched)
        Return:
            True/False --> if path mounted
            details for mount path(list)
        """
        self.log.info("Getting mount details for path %s on MA %s", path, media_agent)
        ma_session = Machine(media_agent, self.commcell)
        mount_status = ma_session.is_path_mounted(path)
        self.log.info("Path mount status for %s is %s", path, mount_status)
        if mount_status:
            command = "df -h | grep '{0}'".format(path)
            self.log.info(command)
            output = ma_session.execute_command(command)
            mount_details = output.formatted_output.split()
            self.log.info("Mount details for %s is %s", path, mount_details)
            return mount_status, mount_details
        self.log.info("Disk %s is not mounted on %s", path, media_agent)
        return mount_status, []

    def vertical_scaleout_log_status(self, media_agent):
        """
        Getting status for vertical scale out from logs on media agent
        Args:
            media_agent --> media agent
        Return:
            flag = 0 no log lines,
            flag = 2 Success
        """
        self.log.info("Getting vertical scaleout status logs from MA %s", media_agent)
        flag = 0
        ma_session = Machine(media_agent, self.commcell)
        command = "gluster v status | grep 'Status of volume:' | awk '{print $4}'"
        self.log.info(command)
        output = ma_session.execute_command(command)
        gluster_name = output.output.replace("\n", "")
        regex = 'bzgrep "CvProcess::system() - gluster volume add-brick\\' \
                '|CVMAWorker::add_bricks_to_gluster_volume\\' \
                '|CVMAWorker::AddBricksToGlusterVolume" /var/log/commvault/Log_Files/CVMA.log | tail -5'
        self.log.info(regex)
        output = ma_session.execute_command(regex)
        if output.output:
            for log in output.formatted_output:
                if HYPERSCALE_CONSTANTS['Add_Bricks_Success'] in " ".join(log).lower() and \
                        gluster_name.lower() in " ".join(log).lower():
                    self.log.info("Add Bricks To Gluster Volume Success, %s",
                                  HYPERSCALE_CONSTANTS['Add_Bricks_Success'])
                    for log_data in output.formatted_output:
                        log_data = ' '.join(log_data)
                        self.log.info(log_data)
                    flag = 2
                    return flag

            if flag != 2:
                self.log.info("No logs found!!")
            return flag
        self.log.info("No logs found!!")
        return flag

    def get_vm_object(self, server, username, password, name):
        """
        Get VM object for the vm name, for the server
        Args:
            server --> server url
            username --> username
            password --> password
            name --> name of vm
        Return:
            vm object/ Exception
        """
        hv_obj = HypervisorHelper.Hypervisor([server], username, password, "vmware", self.commcell)
        vm_help_obj = HypervisorHelper.VMHelper.HypervisorVM(hv_obj, name)
        return vm_help_obj

    def get_ddb_reconstruction_job(self, storage_pool_name):
        """
        Get ddb reconstruction job for storage pool
        Args:
            storage_pool_name(name) --> Storage Pool Name
        Return:
            list of jobs
        """
        self.log.info("Getting ddb reconstruction job")
        storage_pool_details = self.get_storage_pool_details(storage_pool_name)
        gdsp = storage_pool_details.storage_pool_id
        query = """
        select jobId, status from JMAdminJobInfoTable where servStart = 
        (select max(servStart) from JMAdminJobInfoTable where opType = 80) and archGrpID = {0}
        """.format(gdsp)
        self.log.info(query)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        if len(rows[0][0]) <= 0:
            self.log.info("No ddb reconstruction job running at present")
        else:
            job = rows[0]
            self.log.info("Job %s with status %s", job[0], job[1])
            return True, job
        self.log.info("Checking ddb reconstruction job if completed")
        query = """
                select jobId, status from JMAdminJobStatsTable where servStart = 
                (select max(servStart) from JMAdminJobStatsTable where opType = 80) and archGrpID = {0}
                """.format(gdsp)
        self.log.info(query)
        self.csdb.execute(query)
        rows = self.csdb.fetch_all_rows()
        if len(rows[0][0]) <= 0:
            self.log.info("No ddb reconstruction jobs")
            return False, []
        job = rows[0]
        self.log.info("Job %s with status %s", job[0], job[1])
        status = job[1]
        if int(status) == 1:
            self.log.info("Job %s is completed with status %s", job[0], job[1])
        else:
            self.log.info("Job %s with status %s", job[0], job[1])
        return True, job



