# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the functions to validate software cache/ remote cache media

CommServeCache_Validation -- Class to validate software cache media
RemoteCache_Validation    -- Class to validate remote cache media

CommServeCache_Validation
=========================

    __init__()                          -- initialize instance of the CommServeCache_Validation class
    get_files_in_cache()                -- returns list of files in the given cache path
    _get_name_from_osid()                -- returns os name for the given os id
    compute_windows_third_party_packages_to_sync()      -- compute Third Party Packages to Sync based on the available
                                                           packages on CS's cache and those configured for RC
    get_binary_info_dict()              -- converts BinaryInfo.xml to dict
    list_from_binary_info_dict()        -- returns list of cab files and MSIs from Binaryinfo.xml dictionary
    get_os_in_cs_cache()                -- returns list of os ids present in cs cache
    expected_package_list()             -- returns all Packages to be synced


RemoteCache_Validation
=========================

     __init__()                                       --  initialize instance of the RemoteCache_Validation class
     validate_remote_cache()                          -- validate remote cache media
     get_files_to_be_present_in_remote_cache()        -- computes list of files to be present in the remote cache
     get_os_configured_on_remote_cache()              -- returns list of oses to be synced to remote cache
"""
from xml.etree import ElementTree as et
from AutomationUtils import database_helper
from AutomationUtils import logger
from AutomationUtils import cvhelper
from AutomationUtils.machine import Machine
from Install import installer_constants
from Install.softwarecache_helper import SoftwareCache
from Install import installer_utils


class CommServeCache:
    """Download validator class to validate commserve cache"""

    def __init__(self, client_obj, commcell, **kwargs):
        """
        Initialize instance of the CommServeCache class.

        Args:
            commcell_obj -- commcell object

            client_obj   -- client object

        **kwargs: (dict) -- Key value pairs for supporting conditional initializations
            Supported -
            media (str) - media which needs to be validated

        """

        self.commcell = commcell
        self.csdb = database_helper.get_csdb()
        self.log = logger.get_log()
        self.client_obj = client_obj
        self.machine_obj = Machine(self.client_obj)
        self.software_cache_obj = SoftwareCache(self.commcell, self.client_obj)
        self.cs_client = Machine(self.commcell.commserv_client)
        self.cs_cache_path = self.software_cache_obj.get_cs_cache_path()
        self.media = kwargs.get('media', self.software_cache_obj.get_cs_installed_media())
        self.ignore_list = installer_constants.IGNORE_LIST
        self.filtered_dict = {}

    def get_files_in_cache(self, cache_machine_obj=None, media=None):
        """
        Returns list of files in the given cache path.
        Args:
            cache_machine_obj - machine obj for the machine where the cache is present
            media (str) - media to check E.g. SP19_2482227_R737

        Returns:
            list of files in the cache path

        Raises:
            exception if unable to find path
        """
        try:
            if not cache_machine_obj:
                cache_machine = self.cs_client
                if media:
                    folder_path = (f'{self.cs_cache_path}\\{installer_constants.DOWNLOAD_SOFTWARE_DEFAULT_MEDIA}'
                                   f'\\{self.software_cache_obj.get_version()}\\{self.media}')
                else:
                    folder_path = self.cs_cache_path
            else:
                cache_machine = cache_machine_obj
                folder_path = self.software_cache_obj.get_remote_cache_path_to_validate(self.media)
            files_in_cache = set()
            files = cache_machine.get_files_in_path(folder_path=folder_path)
            for each_file in files:
                is_found = False
                for each in self.ignore_list:
                    if each in each_file:
                        is_found = True
                        break
                if not is_found and "11.0.0" in each_file:
                    if 'windows' in cache_machine.os_info.lower():
                        files_in_cache.add(each_file.split("11.0.0\\")[1])
                    else:
                        files_in_cache.add(each_file.split("11.0.0/")[1].replace("/", "\\"))
            return list(files_in_cache)

        except Exception as exp:
            raise Exception("Exception while getting list of files from cache path %s - %s", folder_path, exp)

    def _get_name_from_osid(self, os_id):
        """
        Returns os name for the given os id
        Args:
            os_id (int) - os id

        Returns:
            os name for the given os id
        """

        if os_id != 0:
            query = (f"select simBinarySet.processorName,simBinarySet.name from PatchUpdateOS,simBinarySet where "
                     f"PatchUpdateOS.BinarySetID = simBinarySet.id and PatchUpdateOS.NewUpdateOSID = {os_id}")
            self.csdb.execute(query)
            os_name = self.csdb.fetch_one_row()
        else:
            os_name = installer_constants.OSNAME_LIST[os_id]
        base_os = "Windows" if ("win" in os_name[0]) else "Unix"
        return os_name[0], base_os, os_name[1]

    def compute_windows_third_party_packages_to_sync(self):
        """
        Compute Third Party Packages to Sync based on the available packages on CS's cache and those configured for RC

        Raises:
            exception if unable to parse/get winpackages.xml

        """
        try:
            win_package_xml_path = (f"{self.cs_cache_path}\\{installer_constants.DOWNLOAD_SOFTWARE_DEFAULT_MEDIA}\\"
                                    f"{self.software_cache_obj.get_version()}\\{self.media}\\Windows\\WinPackages.xml")
            win_package_xml_path_unc = installer_utils.convert_unc(self.commcell.commserv_hostname,
                                                                   win_package_xml_path)
            third_party_list = []

            for key, value in self.filtered_dict.items():
                if int(key) == 1:
                    third_party_list = third_party_list + self.software_cache_obj._get_third_party_packages(
                        win_package_xml_path_unc, 1, value)
                elif int(key) == 3:
                    third_party_list = third_party_list + self.software_cache_obj._get_third_party_packages(
                        win_package_xml_path_unc, 3, value)
            computed_third_party_list = list(set(third_party_list))
            self.filtered_dict[0] = computed_third_party_list
            self.log.info("Third party package id are %s", computed_third_party_list)

        except Exception as exp:
            self.log.exception("Exception in Computing ThirdParty Packages to sync - %s", exp)

    def get_binary_info_dict(self, binary_info_xml_path):
        """
        Converts BinaryInfo.xml to dict
        Args:
           binary_info_xml_path - xml path of binaryinfo.xml
        returns:
            binary info dictionary
        """

        tree = et.parse(binary_info_xml_path)
        root = tree.getroot()
        binary_info_dict = self.software_cache_obj.binary_info_xml_to_dict(root)
        return binary_info_dict

    def list_from_binary_info_dict(self, binary_info_dict, binary_set_id, packages):
        """
        Create list of cab files and MSIs from Binaryinfo.xml dictionary
        Args:
           binary_info_dict -- binary info dict
           binary_set_id    -- binary set id found in the xml
           packages         -- list of package ids
        returns:
            filtered list with only required cab files and MSIs
        """

        final_list = []
        os_dict = binary_info_dict[str(binary_set_id)]
        for key, value in os_dict.items():
            try:
                if int(key) in packages:
                    package_dict = os_dict[key]
                    for key, value in package_dict.items():
                        if "Common\\OEM" in key:
                            found = False
                            for each in installer_constants.EXCLUDE_OEM:
                                if each in key:
                                    found = True
                                    break
                            if not found:
                                file = key
                        else:
                            file = key
                        if file:
                            listval = file
                            final_list.append(listval)
            except Exception as exp:
                self.log.exception("Exception in getting package id and retrieving list of files %s", exp)
        return final_list

    def get_os_in_cs_cache(self, service_pack=None):
        """
        returns list of oses present in cs cache
        Args:
            service_pack (str) -- Service pack E.g. 19

        Returns:
            returns list of os ids present in cs cache (list)

        """
        if not service_pack:
            service_pack = self.commcell.commserv_client.service_pack
        cs_os_list = []
        query = f"select OSId from PatchMultiCache where ClientId = 2 and HighestSP = {service_pack}"
        self.csdb.execute(query)
        data = self.csdb.fetch_all_rows()
        for each in data:
            cs_os_list.append(each[0])
        return cs_os_list

    def expected_package_list(self, package_list, optype=0,
                              client_type=0, client_id=2, prev_rel_id=16, new_rel_id=16, os_id=3):
        """
        returns all Packages to be synced
        Args:
            package_list (list)     - packages list to be installed
            optype (int)            - sync =0 install/update/upgrade = 1
            client_type (int)       - remote cache = 0 client = 1
            client_id (int)         - client id
            prev_rel_id (int)       - V11=16, V10=15, V9=14
            new_rel_id (int)        - current release id
            os_id (int)             - patchupdateos id of the os involved in sync/install
        returns:
            new packages list(integer list)
        raises:
            exception if unable to execute stored procedure
        """

        try:
            encrypted_password = self.cs_client.get_registry_value(r"Database", "pAccess")
            sql_password = cvhelper.format_string(self.commcell, encrypted_password).split("_cv")[1]
            sql_instancename = "{0}\\commvault".format(self.commcell.commserv_hostname.lower())
            db = database_helper.MSSQL(
                server=sql_instancename,
                user='sqladmin_cv',
                password=sql_password,
                database='commserv',
                as_dict=False
            )
            for os, packages in package_list.items():
                self.log.info(
                    "Adjusting input package list %s for client ID %s and OS %s", packages, client_id, os_id)
                parameters = "'%s', %d, %d, %d, %d, %d, %d" % (', '.join(
                    map(str, packages)), optype, client_type, int(client_id), prev_rel_id, new_rel_id, int(os))
                expected_packages = db.execute(
                    f'exec simAdjustPkgListForRemoteOperation {parameters}')
                expected_packages = expected_packages.rows[0][0].split(",")
                package_list[os] = [int(package) for package in expected_packages]
            return package_list

        except Exception as exp:
            raise Exception("Failed to obtain adjusted package list - %s", exp)


class RemoteCache(CommServeCache):
    """
    Class to validate Remote cache media
    """

    def __init__(self, client_obj, commcell, **kwargs):
        """
        Initialize instance of the RemoteCache class.

        Args:
            commcell_obj -- commcell object

            client_obj   -- client object

        **kwargs: (dict) -- Key value pairs for supporting conditional initializations
            Supported -
            os_ids (list of integers) -- osids to be synced for lower sp media
        """
        super(RemoteCache, self).__init__(client_obj, commcell, **kwargs)
        self.base_path = (f"{self.cs_cache_path}\\{installer_constants.DOWNLOAD_SOFTWARE_DEFAULT_MEDIA}\\"
                          f"{self.software_cache_obj.get_version()}\\{self.media}")
        self.os_ids = kwargs.get('os_ids')

    def validate_remote_cache(self, configured_os_pkg_list, sync_all=False):
        """
        Validates remote cache media
        Args:
            configured_os_pkg_list(dict) -  Configured list of oses and packages
            sync_all (boolean)           -  If RC is configured to sync all oses

        Raises:
            exception if unable to get the list of files in the cache
        """

        self.log.info("Media to check: %s", self.media)

        if sync_all:
            files_in_remote_cache = self.get_files_in_cache(self.machine_obj, media=self.media)
            self.log.info(
                "Files in %s remote cache:%s", self.client_obj.client_name, files_in_remote_cache)
            files_in_cs_cache = self.get_files_in_cache(media=self.media)
            self.log.info("Files in CS cache:%s", files_in_cs_cache)
            ret = self.machine_obj.compare_lists(files_in_cs_cache, files_in_remote_cache, sort_list=True)
            if not ret:
                self.log.error("Client side Remote cache media validation failed")
            else:
                self.log.info("Remote cache Validation passed for client %s", self.client_obj.client_name)

        else:
            self.configured_os_pkg_list = configured_os_pkg_list
            files_in_remote_cache = self.get_files_in_cache(self.machine_obj)
            self.log.info(
                "Files in %s remote cache:%s", self.client_obj.client_name,
                files_in_remote_cache)
            files_to_be_present_in_remote_cache = self.get_files_to_be_present_in_remote_cache()
            self.log.info("Files to be present in RC:%s", files_to_be_present_in_remote_cache)
            ret = self.machine_obj.compare_lists(
                list(
                    set(files_to_be_present_in_remote_cache)), list(
                    set(files_in_remote_cache)), sort_list=True)
            if not ret:
                self.log.error("Client side Remote cache media validation failed")
            else:
                self.log.info("Remote cache Validation passed for client %s", self.client_obj.client_name)

    def get_files_to_be_present_in_remote_cache(self):
        """
        Computes list of files to be present in the remote cache.
        Returns:
            list of files to be present in the remote cache
        Raises:
            exception if unable to compute list of files
        """

        files_to_be_present = []
        to_delete = []
        self.log.info("Filtering based on OS present in CS cache")
        cs_os_id = list(map(int, self.get_os_in_cs_cache(self.media[2:4])))
        rc_configured_os = self.get_os_configured_on_remote_cache()
        for key, value in rc_configured_os.items():
            if int(key) in cs_os_id:
                self.filtered_dict[key] = rc_configured_os[key]
        self.log.info("Computing Windows Third party packages to sync")
        self.compute_windows_third_party_packages_to_sync()
        remote_cache_path = self.software_cache_obj.get_remote_cache_path_to_validate(self.media)
        for key, value in self.filtered_dict.items():
            os_xml_name, base_os, os_binary_info_name = self._get_name_from_osid(int(key))
            self.log.info("OS Name for int(key) is %s", os_xml_name)
            binary_info_xml_path = self.software_cache_obj.get_binary_info_path(
                base_os, self.base_path, os_xml_name)
            binary_info_dict = self.get_binary_info_dict(
                installer_utils.convert_unc(
                    self.commcell.commserv_hostname,
                    binary_info_xml_path))
            if int(key) != 0:
                query = f"select BinarySetID from PatchUpdateOS where NewUpdateOSID = {key}"
                self.csdb.execute(query)
                binary_set_id = self.csdb.fetch_one_row()[0]
            else:
                binary_set_id = 0
            filtered_binary_info_dict = self.list_from_binary_info_dict(binary_info_dict, binary_set_id, value)
            for each in filtered_binary_info_dict:
                fullpath = each.replace("/", "\\")
                mediapath = f"{remote_cache_path}\\{base_os}\\{os_xml_name}" if (
                    base_os == "Unix") else f"{remote_cache_path}\\{base_os}"
                fullpath = mediapath + "\\" + fullpath
                files_to_be_present.append(fullpath)
        win_root_list, unix_root_list = self.software_cache_obj.get_root_files(remote_cache_path)
        if self.machine_obj.os_info.upper() == "WINDOWS":
            if not any(int(key) in self.filtered_dict for key in [1, 3, 0]):
                files_to_be_present = files_to_be_present + unix_root_list
            files_to_be_present = files_to_be_present + win_root_list
        else:
            if any(int(key) in self.filtered_dict for key in [1, 3]):
                files_to_be_present = files_to_be_present + win_root_list
            files_to_be_present = files_to_be_present + unix_root_list
        for count, element in enumerate(files_to_be_present):
            files_to_be_present[count] = element.replace("/", '\\')
        for count, element in enumerate(files_to_be_present):
            for each in self.ignore_list:
                if each in str(element):
                    to_delete.append(count)
                    break
                else:
                    corrpath = element.split("11.0.0\\")[1]
                    files_to_be_present[count] = corrpath

        for file in sorted(to_delete, reverse=True):
            del files_to_be_present[file]

        return files_to_be_present

    def get_os_configured_on_remote_cache(self):
        """
        returns list of oses to be synced

        Returns:
            returns dictionary of list of oses and packages to be synced to the remote cache
        raises:
            exception if unable to compute list of oses to be synced

        """
        try:

            final_dict = {}
            client_list = []
            client_os_pkg_dict = {}
            self.log.info("Getting clients associated to RC")
            query = f"select componentNameId from App_clientprop where attrname = 'UPDATE CACHE AGENT ID' and" \
                f" attrVal = (select id from PatchUpdateAgentInfo where clientid = {self.client_obj.client_id})"
            self.csdb.execute(query)
            data = self.csdb.fetch_all_rows()
            for each in data:
                client_list.append(each[0])
            for client in client_list:
                query = f"select NewUpdateOSID from PatchUpdateOS where BinarySetID = " \
                    f"(select attrVal from APP_ClientProp where (attrName = 'Binary Set ID' AND componentNameId ={client}))"
                self.csdb.execute(query)
                client_os_id = int(self.csdb.fetch_one_row()[0])
                client_package = self.software_cache_obj.get_packages(client)
                client_os_pkg_dict[client_os_id] = client_package
            self.log.info("Combining the OS and packages dict required for sync")
            for key, value in self.configured_os_pkg_list.items():
                if key in client_os_pkg_dict.keys():
                    final_dict[key] = list(set(self.configured_os_pkg_list[key] + client_os_pkg_dict[key]))
                else:
                    final_dict[key] = self.configured_os_pkg_list[key]
            for key, value in client_os_pkg_dict.items():
                if key in final_dict.keys():
                    final_dict[key] = list(set(final_dict[key] + client_os_pkg_dict[key]))
                else:
                    final_dict[key] = client_os_pkg_dict[key]
            final_dict = self.expected_package_list(
                package_list=final_dict,
                client_id=self.client_obj.client_id)
            for key, value in final_dict.items():
                self.log.info("The package selected for OS %s is %s", key, value)
            if self.media != self.software_cache_obj.get_cs_installed_media():
                self.log.info("filtering the os ids based on side by side cache logic")
                final_dict = dict([(key, final_dict[key]) for key in final_dict if key in self.os_ids])
            return final_dict

        except Exception as exp:
            raise Exception(exp)
