# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Helper file for performing install operations

SimCallHelper is the only class defined in this file

SimCallHelper: Helper class to perform Install operations

SimCallHelper:

    generate_xml()          -- To generate xml for simCall and store it in the given path

    execute_sim_call()      -- To generate the XML for the sim call operation and execute the sim call

    install_new_client()    -- To install a new client on the commcell machine

"""

import re
import xml.etree.ElementTree as ET

from AutomationUtils.machine import Machine
from AutomationUtils import constants
from AutomationUtils.logger import get_log
from AutomationUtils.options_selector import OptionsSelector
from Install.InstallUtils.installer_constants import (CLIENT_USERNAME, CLIENT_PASSWORD, SIM_USER)


class SimCallHelper:
    """
    Helper file for Sim call to perform install operations

    """

    def __init__(self, commcell):
        """
        Initialize the SimCallHelper object

        Args:
            commcell    (object)    -- object of commcell class

        """
        self.commcell = commcell
        self.log = get_log()
        self.temp_dir = constants.TEMP_DIR
        self.commcell_install_directory = self.commcell.commserv_client.install_directory

        self.controller_machine = Machine()
        self.commcell_machine = Machine(self.commcell.commserv_client)
        self.options = OptionsSelector(self.commcell)

    def generate_xml(
            self,
            path=None,
            commserv_hostname=None,
            client_name=None,
            client_hostname=None,
            username=None,
            password=None,
            auth_code=None):
        """
        To generate the XML for the sim call operation

        Args:
            path                    (str)       -- Full path to generate the XML file

            commserv_hostname       (str)       -- Full hostname of the commserv machine

            client_name             (str)       -- Name of the client to be created

            client_hostname         (str)       -- Full hostname of the client machine

            username                (str)       -- Username for the commcell

            password                (str)       -- Encrypted password for the commcell machine

            auth_code               (str)       -- Organization authcode to be used for installation

        """
        xml = fr"""
            <CVInstallManager_ClientSetup Focus="Instance001" Operationtype="0" RemoteClient="1" requestType="1">
               <CommServeInfo>
                  <CommserveHostInfo _type_="3" clientName="" hostName="{commserv_hostname}" />
               </CommServeInfo>
               <ClientAuthentication AuthenticationEnabled="1" DomainConfigured="0" PrincipalName=""
                ProviderID="1" SSOEnabled="0">
                  <userAccountToLogin domainName="" password="{password or CLIENT_PASSWORD}"
                   userName="{username or CLIENT_USERNAME}" />
               </ClientAuthentication>
               <clientComposition activateClient="1" packageDeliveryOption="0">
                  <clientInfo>
                     <client clientPassword="{password or CLIENT_PASSWORD}"
                      cvdPort="8400" installDirectory="C:\Program Files\Commvault\ContentStore">
                        <clientEntity _type_="3" clientName="{client_name}" hostName="{client_hostname}" />
                        <osInfo PlatformType="X64" SubType="Server" Type="Windows" Version="6.2">
                           <OsDisplayInfo OSBuild="9200" OSName="Windows Server 2012 R2 Datacenter"
                            ProcessorType="WinX64" />
                        </osInfo>
                        <jobResulsDir path="C:\Program Files\Commvault\ContentStore\iDataAgent\JobResults" />
                        <versionInfo GalaxyBuildNumber="BUILD80">
                           <GalaxyRelease ReleaseString="11.0" _type_="58" />
                        </versionInfo>
                     </client>
                     <clientProps BinarySetID="3" ClientInterface=""
                      byteOrder="Little-endian" />
                  </clientInfo>
                  <components>
                     <componentInfo ComponentId="1" ComponentName="File System Core" _type_="60" clientSidePackage="1"
                      consumeLicense="1" />
                     <commonInfo>
                        <storagePolicyToUse _type_="17" storagePolicyId="1" storagePolicyName="CV_DEFAULT" />
                     </commonInfo>
                  </components>
                  <patchInformation friendlyName="" spVersion="{self.commcell.commserv_version}">
                     <packagePatches pkgId="1"/>
                  </patchInformation>
               </clientComposition>
               <installFlags activateAllUserProfiles="0" />
                <organizationProperties/>
            </CVInstallManager_ClientSetup>
        """

        xml = ET.fromstring(xml)

        # To convert to an ElementTree
        xml_tree = ET.ElementTree(xml)

        # To set Authcode if given
        if auth_code:
            xml_tree.find('organizationProperties').set('authCode', auth_code)

        # To store it in the given path
        xml_tree.write(path)

    def execute_sim_call(
            self,
            input_path=None,
            output_path=None):
        """
        To generate the XML for the sim call operation and execute the sim call

        Args:
            input_path              (str)       -- Full path to generate the XML file

            output_path             (str)       -- Full path to store the response

        Returns:
            None

        Raises:
            Exception

                If sim call failed

        """
        sim_path = self.commcell_machine.join_path(self.commcell_install_directory,
                                                   'Base',
                                                   'SIMCallWrapper.exe')

        # Command to execute simcall
        command = f'&"{sim_path}" -enc {SIM_USER} -input "{input_path}" -output "{output_path}"'

        # To execute the command on the controller machine
        output = self.commcell_machine.execute_command(command)

        # To read the output file for error level
        file_output = self.commcell_machine.read_file(output_path)

        if 'SimCallWrapper completed' not in output.formatted_output and 'ErrorCode="0"' not in file_output:
            error = re.search('<(.*)SimError (.*?)/>', file_output, re.I)
            self.log.error('SimError: %s', error.group(2))
            raise Exception(f'SimError: {error.group(2)}')

    def install_new_client(
            self,
            client_name=None,
            client_hostname=None,
            username=None,
            password=None,
            auth_code=None):
        """
        To install a new client on the commcell machine

        Args:
            client_name     (str)   -- Name of the client to be created

            client_hostname (str)   -- Hostname of the client machine

            auth_code       (str)   -- Organization authcode to be used for installation

            username        (str)   -- Username to install the new client

            password        (str)   -- Password to be used to install the new client

        Returns:
            (str,str)     -- (clientname, hostname)

        """
        if not client_name and not client_hostname:
            # To get a randomly generated client name and hostname
            client_name = client_hostname = self.options.get_custom_str('client')
        self.log.info('client name: "%s" \n client hostname: "%s"', client_name, client_hostname)

        # To create temp directory in Automation folder if not present on controller machine
        if not self.controller_machine.check_directory_exists(self.temp_dir):
            self.controller_machine.create_directory(self.temp_dir)
            self.log.info('Successfully created temporary directory for Automation on controller machine')

        # Path to save the generated xml file on the controller machine
        path = f"{self.controller_machine.join_path(self.temp_dir, 'Simcall.xml')}"

        # To generate xml for sim call
        self.generate_xml(
            path=path,
            commserv_hostname=self.commcell.commserv_hostname,
            client_name=client_name,
            client_hostname=client_hostname,
            username=username,
            password=password,
            auth_code=auth_code)

        # To generate input path for the sim call
        input_path = self.commcell_machine.join_path(self.commcell_install_directory, 'Temp')
        self.log.info('Input path for the Sim call: %s', input_path)

        # To copy the xml file form the controller machine to the commcell machine
        self.commcell_machine.copy_from_local(path, input_path)

        # Full Path of the input path
        input_path = self.commcell_machine.join_path(input_path, 'Simcall.xml')

        # To generate output path for the sim call
        output_path = self.commcell_machine.join_path(self.commcell_install_directory,
                                                      'Temp',
                                                      'Simcall_output.xml')
        self.log.info('Output path for the Sim call: %s', output_path)

        # To use sim call to install a new client
        self.execute_sim_call(input_path=input_path, output_path=output_path)
        self.log.info('Successfully installed Client: %s on Commserv: %s',
                      client_name,
                      self.commcell.commserv_name)

        # To delete the xml file from the controller machine
        self.controller_machine.delete_file(
            f"{self.controller_machine.join_path(self.temp_dir, 'Simcall.xml')}")

        # To delete the input path
        self.commcell_machine.delete_file(input_path)

        # To delete the output path
        self.commcell_machine.delete_file(output_path)

        return client_name, client_hostname
