# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations for silent install

SilentinstallHelper
===================

    __init__()                          --  initialize instance of the silent install class
    silent_install                      --  Main module for fresh silent installation
    get_commands_to_remote_execute      --  Commands to execute on the remote machine for silent installation
    download_file_from_http_server      --  This method is used to download configuration xml files from server.

    _get_packages_to_install            --  Dictionary of packages(packages id : package name) to be installed
    _get_hard_dependent_packages        --  Get the hard depenedent packages for selected package
    _get_soft_dependent_packages        --  Get the list of soft dependent packages for any package
    _get_pkg_displayname                --  Frame dictionary of names of hard and soft depenedent package
"""
import os
import xmltodict
from AutomationUtils import config, logger, constants
from AutomationUtils.machine import Machine
from Install import installer_constants, installer_utils
from Install.bootstrapper_helper import BootstrapperHelper
from Install.install_xml_helper import InstallXMLGenerator


class SilentInstallHelper:

    @staticmethod
    def create_installer_object(client_name, feature_release, machine_obj, inputs):
        """Returns the instance of one of the Subclasses WindowsInstallation /
        DebianInstallation based on the OS details of the remote client.

        """

        if machine_obj.os_info == 'WINDOWS':
            obj = WindowsInstallation(client_name, feature_release, machine_obj, inputs)
        else:
            obj = DebianInstallation(client_name, feature_release, machine_obj, inputs)

        return obj

    def __init__(self, client_name, feature_release, machine_obj, inputs):
        """
            Initialize instance of the SilentInstallHelper class.
                Args:
                    client_name -- Client Name provided for installation

                    feature_release -- feature release of the bootstrapper

                    machine_obj -- machine object

                    inputs (dict)
                    --  Inputs for Installation should be in a dictionary
                            Supported key / value for inputs:

                            Mandatory:
                                csClientName        (str)        Commserve Client Name
                                csHostname          (str)        Commserve Hostname
                                commservePassword:  (str)        Commserve encrypted login password

                            Optional:
                                commserveUsername   (str)        Commserve Username
                                useExistingDump     (str)        Use existing dump for Dm2 / Workflow Engine ("0" or 1")
                                useExsitingCSdump   (str)        Use CS dump ("0" or "1")
                                CommservDumpPath    (str)        Dump path for Commserve
                                install32base       (str)        install 32bit software on 64bit Machine ("0" or "1")
                                restoreOnlyAgents   (str)       "0 or "1"
                                DM2DumpPath         (str)        Dump path for DM2 webservice
                                WFEngineDumpPath    (str)        Dump path for Workflow Engine
                                decoupledInstall    (str)        "0 or "1"
                                enableFirewallConfig (str)       "0 or "1"
                                firewallConnectionType (str)     "0" or "1" or "2"
                                httpProxyConfigurationType(str)  "0" or "1" or "2"
                                httpProxyPortNumber (str)        Port Number eg: "6256"
                                httpProxyHostName   (str)        Hostname of the proxy machine
                                portNumber          (str)        Port number to connect to the CVD eg:8410
                                enableProxyClient   (str)        Enable Client as a proxy "0" or "1"
                                proxyClientName     (str)        Proxy Client Name / Client Name on the commcell
                                proxyHostname       (str)        Proxy Client Hostname
                                proxyPortNumber     (str)        Proxy client Port Number to be used
                                sqlSaPassword       (str)        Sa (user) password for SQL access
                                installDirectoryUnix(str)        Path on which software to be installed on Unix Client
                                installDirectoryWindows(str)     Path on which software to be installed on Win Client
                                mediaPath           (str)        Filer Path required for Unix installations
                                                                 (Path till CVMedia)

            Note:
                Unix Installation Requires Filer Path/ Media Path ( Path till CVMedia) -- mediaPath
        """
        self.log = logger.get_log()
        self.config_json = config.get_config()
        self.machine_obj = machine_obj
        self.local_machine = Machine()
        self.feature_release = feature_release
        self.bootstrapper = None
        self.os_type = self.machine_obj.os_info.lower()
        self.inputs = inputs
        self.client_name = client_name
        self.test_results_path = constants.TEMP_DIR
        self.remote_dir = installer_constants.REMOTE_FILE_COPY_LOC
        self.download_server = self.config_json.Install.download_server
        self.xml_helper = InstallXMLGenerator(client_name, self.machine_obj, inputs)
        self.remote_media_path = None

    def silent_install(self, packages_list):
        """
            Installs the client on the remote machine depending on the feature release using bootstrapper.
            :arg
                packages_list (list of package id)   -- list of features to be installed
                                                        eg: packages_list=[1, 51, 702] for Windows
                                                        eg: packages_list=[1002, 1101] for Unix
        """
        raise NotImplementedError("Module not implemented for the class")

    def get_packages_to_install(self, packages_list, only_hard_dependent_packages=False):
        """
            Get the dictionary of packages(packages id : package name) to be installed on Remote machine
            Args:
                    packages_list     (list)  --  List of packages selected by the user

                    only_hard_dependent_packages    (bool)  --  Select only hard dependent packages
                                                                :Default False

            Returns:
                Dict  -   dict of packages id with package names
        """
        raise NotImplementedError("Module not implemented for the class")

    def download_file_from_http_server(self, file_path):
        """
            This method is used to download configuration xml files from server.
        """
        try:
            import requests, urllib
            try:
                req = requests.get(file_path)

            except Exception as err:
                self.log.error("Failed to get Hotfix Configuration xml from devserver - %s" % str(err))
                return None

            base_filename = os.path.basename(file_path)
            base_filepath = os.path.join(self.test_results_path, base_filename)
            try:
                fp = open(base_filepath, "wb")

            except Exception as exp:
                self.log.error("Opening XML failed:%s" % exp)
                fp = open(base_filepath, "wb")

            fp.write(req.content)
            fp.close()
            return base_filepath

        except Exception as exp:
            self.log.exception("Exception occurred in Downloading the file from http server %s" % str(exp))
            return None

    def _get_hard_dependent_packages(self, pkgdict, packages_list):
        """
            Get the hard depenedent packages for selected package

            Args:
                 pkgdict        (dict)   --  Xml of packages info convereted to Dict

                 packages_list  (list)   --  List of packages selected by the user

            Returns:
                List  -   List of hard dependent package ids
        """
        try:
            self.log.info("Selected Packages from user is %s" % packages_list)
            hard_list = []
            for each in pkgdict['UpdatePatches_PackageDetails']['PackageInfoList']['pkginfo']:
                if (int(each['@pkgID']) in packages_list) and (each['@binarySetId'] != "1"):
                    if 'RequiresHard' in each:
                        if len(each['RequiresHard']) == 1:
                            hard_list.append(int(each['RequiresHard']['@pkgID']))
                        elif len(each['RequiresHard']) > 1:
                            for every in each['RequiresHard']:
                                if int(every['@pkgID']) not in packages_list:
                                    hard_list.append(int(every['@pkgID']))
            self.log.info("Hard dependent packages are " + str(hard_list))
            return hard_list

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("An exception occurred in getting hard dependent packages")

    def _get_soft_dependent_packages(self, pkgdict, packages_list):
        """
            Get the list of soft dependent packages for any package

            Args:
                 pkgdict        (dict)   --  Xml of packages info convereted to Dict

                 packages_list  (list)   --  List of packages selected by the user

            Returns:
                List  -   List of soft dependent package ids

        """
        try:
            soft_list = []
            for each in pkgdict['UpdatePatches_PackageDetails']['PackageInfoList']['pkginfo']:
                if (int(each['@pkgID']) in packages_list) and (each['@binarySetId'] != "1"):
                    if 'RequiresSoft' in each:
                        if len(each['RequiresSoft']) == 1:
                            soft_list.append(int(each['RequiresSoft']['@pkgID']))
                        elif len(each['RequiresSoft']) > 1:
                            for every in each['RequiresSoft']:
                                if int(every['@pkgID']) not in packages_list:
                                    soft_list.append(int(every['@pkgID']))

            self.log.info("Soft dependent packages are " + str(soft_list))
            return soft_list

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("An exception occurred in getting soft dependent packages")

    def _get_pkg_displayname(self, pkgdict, pkg_list):
        """
               frame dictionary of hard and soft depenedent package
        """
        try:
            self.log.info("Getting the packages names from WinPackages.xml")
            packages_info = {}
            for each in pkgdict['UpdatePatches_PackageDetails']['PkgList']['pkg']:
                if int(each['@id']) in pkg_list:
                    packages_info[each['@id']] = each['@DisplayName']

            self.log.info("Package ID display Name dict is  " + str(packages_info))
            return packages_info

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("An Exception occurred while getting dictionary of name and package id")

    def get_commands_to_remote_execute(self, media_path):
        """
            Commands to be executed on the remote Machine
        :param
            media_path: Path of the Media on remote machine (Path where Setup.exe / silent_install is Present)
        :return:
            bat file that has all the list of commands
        """
        raise Exception("Method not implemented for this Class")


class WindowsInstallation(SilentInstallHelper):
    """Class for performing Install operations on a Windows client."""

    def __init__(self, client_name, feature_release, machine_obj, inputs):
        """
            Args:
                    client_name -- Client Name provided for installation

                    feature_release -- feature release of the bootstrapper

                    machine_obj -- machine object

                    inputs (dict) --  Inputs for Installation should be in a dictionary

            Note:
                Unix Installation Requires Filer Path/ Media Path (Path till CVMedia) -- mediaPath

        """
        super(WindowsInstallation, self).__init__(client_name, feature_release, machine_obj, inputs)

    def silent_install(self, packages_list):
        """
            Installs the client on the remote machine depending on the feature release using bootstrapper.
            :arg
                packages_list (list of package id)   -- list of features to be installed
                                                        eg: packages_list=[1, 51, 702] for Windows

        """
        try:
            # Windows use Bootstrapper to Install the packages
            self.bootstrapper = BootstrapperHelper(self.feature_release, self.machine_obj)
            self.bootstrapper.extract_bootstrapper()
            self.remote_media_path = self.bootstrapper.remote_drive + \
                installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH

            packages_to_be_installed = self.get_packages_to_install(packages_list)
            self.log.info("Packages to be installed on the remote machine : %s" % str(packages_to_be_installed))
            root = self.xml_helper.silent_install_xml(packages_to_be_installed)
            xml_path = self.xml_helper.write_xml_to_file(root, "silent_install.xml")
            install_batch_file = self.get_commands_to_remote_execute(self.remote_media_path)
            self.log.info("Copying file [{0}] on Client [{1}] at [{2}]"
                          "".format(install_batch_file, self.machine_obj.machine_name, self.remote_dir))

            # Copying batch file and the XML on the remote machine
            self.machine_obj.copy_from_local(install_batch_file, self.remote_dir)
            self.machine_obj.copy_from_local(xml_path, self.remote_dir)
            _command = os.path.join(self.remote_dir, os.path.basename(install_batch_file))

            self.log.info("Executing command [{0}] on client [{1}]".format(_command, self.machine_obj.machine_name))
            return_code = self.machine_obj.execute_command(_command)

            if not return_code.exit_code == 0:
                self.log.error("Silent Installation Failed")
                raise Exception("Failed to Execute the bat file: %s" % return_code.formatted_output)

            self.log.info(return_code.formatted_output)
            self.log.info("Silent Installation Successful for the machine")

        except Exception as err:
            self.log.exception(str(err))
            raise Exception(f"Silent Installation Failed of the machine {self.machine_obj.machine_name}")

    def get_packages_to_install(self, packages_list, only_hard_dependent_packages=False):
        """
            Get the dictionary of packages(packages id : package name) to be installed on Remote machine
            Args:
                    packages_list     (list)  --  List of packages selected by the user

                    only_hard_dependent_packages    (bool)  --  Select only hard dependent packages
                                                                :Default False

            Returns:
                Dict  -   dict of packages id with package names
        """
        try:
            self.log.info("Packages selected by the user: %s" % packages_list)
            xml_path = self.get_pkg_info_path
            pkg_xml_path = self.download_file_from_http_server(xml_path)

            with open(pkg_xml_path) as fd:
                pkgdict = xmltodict.parse(fd.read())

            if not only_hard_dependent_packages:
                self.log.info("Finding the hard dependent packages")
                hard_packages = self._get_hard_dependent_packages(pkgdict, packages_list)
                initial_soft_packages = self._get_soft_dependent_packages(pkgdict, packages_list)
                packages_list = list(set(packages_list) | set(initial_soft_packages))
                final_soft_packages = self._get_hard_dependent_packages(pkgdict, packages_list)

            else:
                self.log.info("Finding only the hard dependent packages")
                hard_packages = self._get_hard_dependent_packages(pkgdict, packages_list)
                final_soft_packages = []
                self.log.info("Soft packages wont be installed as it is not selected")

            final_pkg_list = list(set(packages_list) | set(hard_packages) | set(final_soft_packages))
            final_pkg_dict = self._get_pkg_displayname(pkgdict, final_pkg_list)
            return final_pkg_dict

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("An error occurred while getting list of packages to be installed")

    def get_commands_to_remote_execute(self, media_path):
        """
            Commands to be executed on the remote Machine
        :param
            media_path: Path of the Media on remote machine (Path where Setup.exe / silent_install is Present)
        :return:
            bat file that has all the list of commands
        """
        try:
            cmd_list=[]
            cmd_list.append(media_path + "\\Setup.exe /silent /install /resume /play \"" +
                            installer_constants.REMOTE_FILE_COPY_LOC + "\\silent_install.xml\"" +
                            " >> " + installer_utils.get_batch_output_file())

            cmd_list.append("set exitcode=%ERRORLEVEL%" + ">> " + installer_utils.get_batch_output_file())
            cmd_list.append("EXIT %exitcode%" + ">> " + installer_utils.get_batch_output_file())
            install_batch_file = installer_utils.create_batch_file_for_remote(cmd_list, file_name="silent_install.bat")
            return install_batch_file

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("Exception in creating a Batch file")

    @property
    def get_pkg_info_path(self):
        return f"http://{self.download_server}/CVMedia/{installer_constants.CURRENT_RELEASE_VERSION}/" \
               f"{installer_constants.CURRENT_BUILD_VERSION}/{self.feature_release}DVD/Windows/WinPackages.xml"


class DebianInstallation(SilentInstallHelper):
    """Class for performing Install operations on a debian/Mac client."""

    def __init__(self, client_name, feature_release, machine_obj, inputs):
        """
        Args:
                    client_name -- Client Name provided for installation

                    feature_release -- feature release of the bootstrapper

                    machine_obj -- machine object

                    inputs (dict) --  Inputs for Installation should be in a dictionary

            Note:
                Unix Installation Requires Filer Path/ Media Path ( Path till CVMedia) -- mediaPath

        """
        super(DebianInstallation, self).__init__(client_name, feature_release, machine_obj, inputs)

    def silent_install(self, packages_list):
        """
            Installs the client on the remote machine depending on the feature release using bootstrapper.
            :arg
                packages_list (list of package id)   -- list of features to be installed
                                                        eg: packages_list=[1, 51, 702] for Windows
                                                        eg: packages_list=[1002, 1101] for Unix
        """
        try:
            self.remote_dir = installer_constants.UNIX_REMOTE_FILE_COPY_LOC
            if self.inputs.get("mediaPath"):
                self.remote_media_path = self.inputs.get("mediaPath").replace("\\", "/")
            else:
                raise Exception("Please specify the path for download on Testcase Inputs")

            packages_to_be_installed = self.get_packages_to_install(packages_list)
            self.log.info("Packages to be installed on the remote machine : %s" % str(packages_to_be_installed))
            root = self.xml_helper.silent_install_xml(packages_to_be_installed)
            xml_path = self.xml_helper.write_xml_to_file(root, "silent_install.xml")
            install_batch_file = self.get_commands_to_remote_execute(self.remote_media_path)
            self.log.info("Copying file [{0}] on Client [{1}] at [{2}]"
                          "".format(install_batch_file, self.machine_obj.machine_name, self.remote_dir))

            self.machine_obj.copy_from_local(install_batch_file, self.remote_dir)
            self.machine_obj.copy_from_local(xml_path, self.remote_dir)
            _command = os.path.join(self.remote_dir, os.path.basename(install_batch_file))
            _command = _command.replace("\\", "/")
            self.machine_obj.execute_command("chmod -R 0775 " + self.remote_dir)

            self.log.info("Executing command [{0}] on client [{1}]".format(_command, self.machine_obj.machine_name))
            return_code = self.machine_obj.execute_command(_command)



            if not return_code.exit_code == 0:
                self.log.error("Silent Installation Failed")
                raise Exception("Failed to Execute the bat file: %s" % return_code.formatted_output)

            self.log.info(return_code.formatted_output)
            self.log.info("Silent Installation Successful for the machine")

        except Exception as err:
            self.log.exception(str(err))
            raise Exception(f"Silent Installation Failed of the machine {self.machine_obj.machine_name}")

    def get_packages_to_install(self, packages_list, only_hard_dependent_packages=False):
        """
            Get the dictionary of packages(packages id : package name) to be installed on Remote machine
            Args:
                    packages_list     (list)  --  List of packages selected by the user

                    only_hard_dependent_packages    (bool)  --  Select only hard dependent packages
                                                                :Default False

            Returns:
                Dict  -   dict of packages id with package names
        """
        try:
            self.log.info("Packages selected by the user: %s" % packages_list)
            xml_path = self.get_pkg_info_path
            pkg_xml_path = self.download_file_from_http_server(xml_path)

            with open(pkg_xml_path) as fd:
                pkgdict = xmltodict.parse(fd.read())

            if not only_hard_dependent_packages:
                self.log.info("Finding the hard dependent packages")
                hard_packages = self._get_hard_dependent_packages(pkgdict, packages_list)
                initial_soft_packages = self._get_soft_dependent_packages(pkgdict, packages_list)
                packages_list = list(set(packages_list) | set(initial_soft_packages))
                final_soft_packages = self._get_hard_dependent_packages(pkgdict, packages_list)

            else:
                self.log.info("Finding only the hard dependent packages")
                hard_packages = self._get_hard_dependent_packages(pkgdict, packages_list)
                final_soft_packages = []
                self.log.info("Soft packages wont be installed as it is not selected")

            final_pkg_list = list(set(packages_list) | set(hard_packages) | set(final_soft_packages))
            final_pkg_dict = self._get_pkg_displayname(pkgdict, final_pkg_list)
            return final_pkg_dict

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("An error occurred while getting list of packages to be installed")

    def get_commands_to_remote_execute(self, media_path):
        """
            Commands to be executed on the remote Machine
        :param
            media_path: Path of the Media on remote machine (Path where Setup.exe / silent_install is Present)
        :return:
            bat file that has all the list of commands
        """
        try:
            cmd_list = []
            if media_path[:2] == "//":
                if self.machine_obj.check_directory_exists("/cvbuild"):
                    self.machine_obj.execute_command("umount /cvbuild")

                else:
                    self.machine_obj.create_directory(directory_name="/cvbuild", force_create=True)
                mount_op = self.machine_obj.execute_command("mount -t cifs " + media_path + " -o username=" +
                                             self.config_json.Install.dvd_username.split("\\")[1] +
                                             ",password=" + self.config_json.Install.dvd_password + ",vers=2.0" +
                                             ",domain=" + self.config_json.Install.dvd_username.split("\\")[0] +
                                             " /cvbuild")
                if mount_op.exit_code != 0:
                    raise Exception("Mount failed.. Check network path and credentials")
                cmd_list.append(f"/cvbuild/silent_install -p " + installer_constants.UNIX_REMOTE_FILE_COPY_LOC +
                            "/silent_install.xml -jobid 836")
            else:
                cmd_list.append(f"{media_path}/silent_install -p " + installer_constants.UNIX_REMOTE_FILE_COPY_LOC +
                                "/silent_install.xml -jobid 836")

            install_batch_file = installer_utils.create_batch_file_for_remote(cmd_list, file_name="silent_install.bat")
            return install_batch_file

        except Exception as err:
            self.log.exception(str(err))
            raise Exception("Exception in creating a Batch file")

    @property
    def get_pkg_info_path(self):
        return f"http://{self.download_server}/CVMedia/{installer_constants.CURRENT_RELEASE_VERSION}/" \
               f"{installer_constants.CURRENT_BUILD_VERSION}/{self.feature_release}DVD/Unix/pkg.xml"
