# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""File for the Utils that Support Install Automation

get_current_time()              --  Returns the current time in UnixTime Format
get_batch_working_directory()   --  Returns the Directory where Batch File is Copied
get_batch_output_file()         --  Returns the Path where Output of the Batch file is Printed
output_pipe_path_inside_batch() --  Appends the Path where output of batch file is Printed
create_batch_file_for_remote()  --  This method creates a batch file with the list of commands given
run_batch_file()                --  This Method Runs the Batch File on the current machine
is_path_local()                 --  This Method check if the Path given is Remote or on Local Machine
convert_unc()                   --  Convert a file path on a host to a UNC path.
"""


import os
import time
import subprocess
from Install import installer_constants
from AutomationUtils import logger, constants


def get_current_time():
    """Returns the current time in UnixTime Format"""
    return str(int(round(time.time(), 0)))


def get_batch_working_directory():
    """Returns the Directory where Batch File is Copied """
    return installer_constants.REMOTE_FILE_COPY_LOC


def get_batch_output_file():
    """ Returns the Path where Output of the Batch file is Printed"""
    return get_batch_working_directory() + "\\AllOutputFile.txt"


def output_pipe_path_inside_batch():
    """Appends the Path where output of batch file is Printed"""
    return " >> " + get_batch_output_file()


def create_batch_file_for_remote(commands=None, file_name="tempremoteexec.bat"):
    """
        This method creates a batch file with the list of commands given

            Args:
                commands     (list)  --  List of commands to be present in batch file
                    default: None

                file_name    (str)   --  Name of the Batch File
                    default: tempremoteexec.bat

            Returns:
                The File Path where Batch file is created.

    """
    log = logger.get_log()
    try:
        _path = constants.TEMP_DIR
        if file_name:
            _filepath = os.path.join(_path, file_name)
        _filehandle = open(_filepath, "w")
        if commands:
            for command in commands:
                _filehandle.write(command + "\n")
        _filehandle.close()

        log.info("Created batch file [{0}]".format(_filepath))

        return _filepath
    except Exception as err:
        log.exception("Exception raised at createBatchFileForRemoteExec: %s", str(err))
        raise err


def run_batch_file(file_path):
    """
    This Method Runs the Batch File on the current machine
        Args:
            file_path     (str)  --  File Path of the Batch File to be Triggered

        Returns:
            (Str) Output Obtained from the Batch File

        Raises:
            SDKException:
                if it fails to trigger the Batch File
    """
    log = logger.get_log()
    try:
        log.info("Executing batch file [{0}]".format(file_path))

        result = subprocess.call(file_path, shell=True)
        return result

    except Exception as err:
        log.exception("Exception in run_batch_file: %s" % str(err))
        raise err


def is_path_local(path):
    """
    This Method check if the Path given is Remote or on Local Machine
        Args:
            path     (str)  --  Path to check if it is local or Remote

        Returns:
            True            --  If the Path is Local
            False           --  If the Path is Remote

        Raises:
            SDKException:
                if it is unable to determine the Path Format
    """
    if path.startswith("\\"):
        return False

    return True


def convert_unc(hostname, path):
    """
    Convert a file path on a host to a UNC path.
    Args:
        hostname - hostname where the file exists
        path - path of file\folder

    Returns:
        returns converted path or False
    """
    try:
        return ''.join(['\\\\', hostname, '\\', path.replace(':', '$')])
    except Exception as exp:
        raise Exception("Exception raised at convert_unc")
