# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Helper file for performing Install validation

InstallValidator
================

    __new__()               --  based on the OS details Windows / UNIX,
                                    initialize the class instance accordingly

    __init__()              --  initializes Install validator helper object

    validate_baseline()     --  to validate if baseline is 1 or not

    validate_sp_version()   --  to validate if package sp version matches that of commserv

Attributes
----------

    **package_id**      --  returns the package id list based on the package names

"""

from abc import ABCMeta, abstractmethod

from AutomationUtils.machine import Machine
from Install.installer_constants import WINDOWS_SERVICES, UNIX_SERVICES


class InstallValidator():
    """Install validator for windows and unix machines"""
    __metaclass__ = ABCMeta

    def __new__(cls, machine_name, testcase_object):
        """
        Returns the instance of one of the Subclasses WindowsValidator / UnixValidator,
        based on the OS details of the remote client.

        """
        machine = Machine(machine_name, testcase_object.commcell)
        if 'windows' in machine.os_info.lower():
            instance = object.__new__(WindowsValidator)
        elif 'unix' in machine.os_info.lower():
            instance = object.__new__(UnixValidator)
        else:
            raise Exception("Validation is not implemented for this os")

        instance.machine = machine
        return instance

    def __init__(self, machine_name, testcase_object):
        """
        Initializes instance of the InstallValidator class.

        Args:
            machine_name    (str)           -- machine to validate install on

            testcase_object    (object)     -- object of TestCase class

        """
        self.machine_name = machine_name
        self.testcase = testcase_object
        self.log = testcase_object.log
        self.csdb = testcase_object.csdb
        self.commcell = testcase_object.commcell
        self.client_id = self.commcell.clients.get(machine_name).client_id
        self._package_id = []
        self.os_info = None

    def __repr__(self):
        """
        String representation of the instance of this class.

        Returns:
            str - string about the details of the Machine class instance

        """
        return f"InstallValidator class instance of Host {self.machine_name}"

    def __enter__(self):
        """Returns the current instance.

            Returns:
                object  -   the initialized instance referred by self

        """
        return self

    def __del__(self):
        """
        destructor of the class InstallValidator

        """
        del self.machine_name
        del self.testcase
        del self.log
        del self.csdb
        del self.commcell
        del self.client_id
        del self._package_id

    @property
    def package_id(self):
        """"Returns the list of package id's"""
        if not self._package_id:
            query = f"select simPackageID from simInstalledPackages where ClientId = '{self.client_id}'"
            self.csdb.execute(query)
            cur = self.csdb.fetch_all_rows()
            for row in cur:
                self._package_id.append(int(row[0]))

        return self._package_id

    @package_id.setter
    def package_id(self, package_names):
        """Sets the value of package_id attribute."""
        try:
            for package in package_names:
                if isinstance(package, str):
                    package = package.strip()
                    query = f"select id from simPackage where replace(Name,' ','')= '{package.replace(' ','')}'"
                    self.csdb.execute(query)
                    cur = self.csdb.fetch_one_row()
                    self._package_id.append(int(cur[0]))
                else:
                    self._package_id.append(int(package))
        except Exception:
            self.log.info("Invalid package name given")
            raise Exception("Invalid package name given")

    @abstractmethod
    def validate_services(self):
        """
        To validate if services are running on the client machine or not

        Raises:
             Exception:

                if service is not running

        """
        raise NotImplementedError('Method Not Implemented by the Child Class')

    def validate_baseline(self):
        """
        To validate if baseline for the installed packages is 1 or not

        Raises:
             Exception:

                if baseline of the package is not 1

        """
        packages = self.package_id
        for package in packages:
            query = f"select  baseline from simInstalledPackages where " \
                    f"ClientId={self.client_id} and simPackageID={package}"
            self.csdb.execute(query)
            if self.csdb.fetch_one_row()[0] != '1':
                raise Exception(f'Baseline of package {package} is not 1 on '
                                f'client {self.machine_name}')

    def validate_sp_version(self):
        """
        To validate if sp version for the installed packages matches the commserv version or not

        Raises:
             Exception:

                if sp version is not same as commserv version

        """
        sp_version = self.commcell.commserv_version
        packages = self.package_id
        for package in packages:
            query = f"select HighestSP from simInstalledPackages where" \
                    f" ClientId={self.client_id} and simPackageID={package}"
            self.csdb.execute(query)
            if self.csdb.fetch_one_row()[0] != str(sp_version):
                raise Exception(f'SP version of package {package} is not same as commcell on '
                                f'client {self.machine_name}')


class WindowsValidator(InstallValidator):
    """Class for performing validation on the Windows machine"""

    def __init__(self, machine_name, testcase_object):
        """
        Initializes instance of the InstallValidator class

        Args:
            machine_name    (str)           -- machine to validate install on

            testcase_object    (object)     -- object of TestCase class

        """
        super(WindowsValidator, self).__init__(machine_name, testcase_object)
        self.os_info = 'WINDOWS'

    def validate_services(self):
        """
        To validate if services are running on the client machine or not

        Raises:
             Exception:

                if service is not running

        """
        services = []
        windows_services = WINDOWS_SERVICES
        packages = self.package_id

        [services.append(f'{service}({self.machine.instance})')
         for package in packages if package in windows_services
         for service in windows_services[package] if service not in services]

        machine_services = self.machine.execute_command('Get-Service | Where Status'
                                                        ' -eq "Running" | select Name')
        running_services = [''.join(service) for service in machine_services.formatted_output]

        if not set(services).issubset(running_services):
            raise Exception(f'Windows services for machine {self.machine_name} are down')


class UnixValidator(InstallValidator):
    """Class for performing validation on the Unix machine"""

    def __init__(self, machine_name, testcase_object):
        """
        Initializes instance of the InstallValidator class

        Args:
            machine_name    (str)           -- machine to validate install on

            testcase_object    (object)     -- object of TestCase class

        """
        super(UnixValidator, self).__init__(machine_name, testcase_object)
        self.os_info = 'UNIX'

    def validate_services(self):
        """
        To validate if services are running on the client machine or not

        Raises:
             Exception:

                if service is not running

        """
        services = []
        unix_services = UNIX_SERVICES
        packages = self.package_id
        [services.append(service) for package in packages if package in unix_services
         for service in unix_services[package] if service not in services]
        for service in services:
            cmd = 'ps -A | grep ' + service
            output = self.machine.execute_command(cmd)
            if str(output.output).find(service) > 0:
                self.log.info("Service created successfully for %s" % service)
            else:
                raise Exception(f'Unix services for machine {self.machine_name} are down')
