# -*- coding: utf-8 -*-
# ---------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# ---------------------------------------------------------------------------

"""Helper file for performing install operations

InstallHelper: Helper class to perform Install operations

InstallHelper:

    wait_for_services()             --  waits for the communication services to come up
                                        in the commcell machine

    revert_snap()                   --  To revert the snap of a virtual machine

    execute_command()               --  To execute the command on the remote machine using PsExec.exe

    restart_services()              --  Restart services on the client

    install_software()              --  Install specific feature packages on a client

    uninstall_client()              -- Module to uninstall client and hard delete from commcell

UnixInstallHelper:

    install_software()              --  Install specific feature packages on a client

    uninstall_client()              -- Module to uninstall client and hard delete from commcell

    silent_install()                -- Unattended installation of client based on feature release

    restart_services()              --  Restart services on the client

WindowsInstallHelper:

    install_software()              --  Install specific feature packages on a client

    uninstall_client()              -- Module to uninstall client and hard delete from commcell

    silent_install()                -- Unattended installation of client based on feature release

    restart_services()              --  Restart services on the client

"""
import os
import time
from base64 import b64encode
import subprocess
from cvpysdk.deployment.deploymentconstants import WindowsDownloadFeatures
from cvpysdk.deployment.deploymentconstants import UnixDownloadFeatures
from AutomationUtils import logger
from AutomationUtils.machine import Machine
from AutomationUtils.constants import AUTOMATION_DIRECTORY
from AutomationUtils import config, constants
from AutomationUtils.options_selector import OptionsSelector
from Install import installer_utils, installer_constants
from Install.silent_install_helper import SilentInstallHelper


class InstallHelper:
    """Helper class to perform  install operations"""

    def __new__(cls, commcell, machine_obj=None):
        """
        Returns the respective class object based on the platform OS

        Args:
           commcell   -- commcell object

           machine_obj -- machine object

        Returns (obj) -- Return the class object based on the OS

        """
        if cls is not __class__ or machine_obj is None:
            return super().__new__(cls)

        if 'windows' in machine_obj.os_info.lower():
            return object.__new__(WindowsInstallHelper)

        elif 'unix' in machine_obj.os_info.lower():
            return object.__new__(UnixInstallHelper)

    def __init__(self, commcell, machine_obj=None):
        """
        constructor for install related files
        """
        self.log = logger.get_log()
        self.commcell = commcell
        self.commserv = self.commcell.commserv_client
        self.machine = Machine()
        self.config_json = config.get_config()
        self.options_selector = OptionsSelector(self.commcell)
        self.client_machine_obj = machine_obj
        self.test_results_path = constants.TEMP_DIR

    def wait_for_services(self, wait_time=3600, retry=180):
        """
        waits for the communication services on the commcell machine to come up

        Args:
            wait_time  (int)    -- Time to wait for the services to come up

            retry      (int)    -- Retry interval for checking the services

        Returns:
            None

        Raises:
            Exception

            if communication services are down after a threshold time

        """
        self.log.info("waiting for services on the commcell machine")
        start_time = time.time()
        while time.time() - start_time < wait_time:
            try:
                if self.commserv.is_ready:
                    self.log.info("communication services are up and running")
                    return
            except Exception:
                continue

            time.sleep(retry)
        raise Exception("Communication services are down")

    def execute_command(
            self,
            hostname=None,
            username=None,
            password=None,
            command=None):
        """
        To execute the command on the remote machine using PsExec.exe

        Args:
            hostname    (str)   -- Full hostname of the machine to execute command on
            username    (str)   -- Username to connect to the machine
            password    (str)   -- Password to connect to the machine
            command     (str)   -- Command to execute on the remote machine

        Returns:
            (int)       -- Return code of the command executed

        """
        exe_path = self.machine.join_path(AUTOMATION_DIRECTORY, 'CompiledBins', 'PsExec.exe')

        command = (f'"{exe_path}"'
                   ' -i 1'
                   f' "\\\\{hostname}"'
                   f' -u "{username}"'
                   f' -p "{password}"'
                   f' {command}')

        return subprocess.call(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)

    def revert_snap(
            self,
            server_name=None,
            username=None,
            password=None,
            vm_name=None,
            snap_name='fresh'):
        """
        To revert the snap of a VM

        Args:
            server_name     (str)   -- Full hostname of the Server machine

            username        (str)   -- Username to login

            password        (str)   -- password of the machine

            vm_name         (str)   -- Name of the VM to revert snap

            snap_name       (str)   -- Snap name to revert the VM to
                                        default: 'fresh'

        """
        machine = Machine(server_name, username=username, password=password)
        command = {
            "server_name": server_name,
            "vm_name": vm_name,
            "operation": "RevertSnap",
            "extra_args": snap_name,
            "vhd_name": "$null"
        }

        script_path = machine.join_path(
            AUTOMATION_DIRECTORY,
            "VirtualServer",
            "VSAUtils",
            "HyperVOperation.ps1"
        )
        output = machine.execute_script(script_path, command)

        command['operation'] = "PowerOn"
        output = machine.execute_script(script_path, command)

        if '0' in output.formatted_output:
            self.log.info('Successfully reverted VM %s to snap %s', vm_name, snap_name)
            self.log.info('VM is powered on successfully')

            # To wait for the machine to come up
            self.log.info('Sleeping for 3 minutes for the machine %s to be up', vm_name)
            time.sleep(180)
        else:
            self.log.error('Failed to Revert VM %s, please check the logs', vm_name)
            raise Exception(f'Failed to Revert VM {vm_name}, please check the logs')

    def get_machine_objects(self, type_of_machines=0):
        """
            This Method returns the list of machine objects
                Args:
                    type_of_machines    (int)  --  Create Machine Class Object

                            0 : Both the machines (Windows and Unix Machine are created)
                            1: Only Windows Machine is created
                            2: Only Unix Machine is
                            default : 0

                Returns:
                    list  -   list of machine objects are returned

                Raise :
                    Exception:
                        - if failed to get the machine object
                        - type_of_machines >2  or <0

                Note:
                    Details to be mentioned on the config.json file under Install.

                        Machine_user_name   (str): Client UserName

                        Machine_password    (str): Client Password

                        Machine_host_name   (str): Client Hostname
        """

        config_json = config.get_config()
        list_of_machine_objects = []

        if type_of_machines not in [0, 1, 2]:
            raise Exception("Type_machines selected invalid")

        if type_of_machines in [0, 1]:
            windows_machine_obj = self.options_selector.get_machine_object(
                machine=config_json.Install.windows_client.machine_host,
                username=config_json.Install.windows_client.machine_username,
                password=config_json.Install.windows_client.machine_password)

            self.log.info("Windows machine object created")
            list_of_machine_objects.append(windows_machine_obj)

        if type_of_machines in [0, 2]:
            unix_machine_obj = self.options_selector.get_machine_object(
                machine=config_json.Install.unix_client.machine_host,
                username=config_json.Install.unix_client.machine_username,
                password=config_json.Install.unix_client.machine_password)

            self.log.info("Unix machine object created")
            list_of_machine_objects.append(unix_machine_obj)

        return list_of_machine_objects

    def install_software(
            self,
            client_computers=None,
            features=None,
            username=None,
            password=None,
            install_path=None,
            **kwargs):
        raise NotImplementedError("Module not implemented for the class")

    def uninstall_client(self, delete_client=True):
        """ Module to uninstall client and hard delete from commcell.
            Deletes the Instance001 associated to the Commcell / Decoupled Client.

         Args:
            delete_client (bool)       : Delete client from commcell?

                                        False : Client in de-configured state on commcell

                                              : Client not part of Commcell (Decoupled Uninstallation)


        Exception:

            Machine Object not created with Credentials.

            If failed to uninstall client

            Failed to delete the client from Commcell : As client not part of commcell (Decoupled Client Instance)


        Note:
            Machine object should always be created with Machine credentials.

                Details to be mentioned on the config.json file under Install.

                        Machine_user_name   (str): Client UserName

                        Machine_password    (str): Client Password

                        Machine_host_name   (str): Client Hostname

            Delete client should be False for Uninstalling decoupled client

        """
        raise NotImplementedError("Module not implemented for the class")

    def silent_install(self, client_name, tcinputs, feature_release=None, packages=None):
        """
        Installs the client on the remote machine depending on the giver user inputs/ Server selected

            Args:
                    client_name (str)    -- Client Name provided for installation

                    tcinputs     (dict)  -- testcase inputs / Dictionary with supported keys

                                         --  Inputs for Installation should be in a dictionary
                            Supported key / value for inputs:

                            Mandatory:
                                commservePassword:  (str)        Commserve encrypted login password
                                mediaPath           (str)        Filer Path required for Unix installations
                                                                (Path till media)

                            Optional:
                                commserveUsername   (str)        Commserve Username
                                useExistingDump     (str)        Use existing dump for Dm2 / Workflow Engine ("0" or 1")
                                useExsitingCSdump   (str)        Use CS dump ("0" or "1")
                                CommservDumpPath    (str)        Dump path for Commserve
                                install32base       (str)        install 32bit software on 64bit Machine ("0" or "1")
                                restoreOnlyAgents   (str)       "0 or "1"
                                DM2DumpPath         (str)        Dump path for DM2 webservice
                                WFEngineDumpPath    (str)        Dump path for Workflow Engine
                                decoupledInstall    (str)        "0 or "1"
                                enableFirewallConfig (str)       "0 or "1"
                                firewallConnectionType (str)     "0" or "1" or "2"
                                httpProxyConfigurationType(str)  "0" or "1" or "2"
                                httpProxyPortNumber (str)        Port Number eg: "6256"
                                httpProxyHostName   (str)        Hostname of the proxy machine
                                portNumber          (str)        Port number to connect to the CVD eg:8410
                                enableProxyClient   (str)        Enable Client as a proxy "0" or "1"
                                proxyClientName     (str)        Proxy Client Name / Client Name on the commcell
                                proxyHostname       (str)        Proxy Client Hostname
                                proxyPortNumber     (str)        Proxy client Port Number to be used
                                sqlSaPassword       (str)        Sa (user) password for SQL access
                                installDirectoryUnix(str)        Path on which software to be installed on Unix Client
                                installDirectoryWindows(str)     Path on which software to be installed on Win Client
                                force_ipv4(unix)       (int)    0 for both IPv4 and IPv6 support
                                                                1 for IPv4 support only
                                                                2 for IPv6 support only


                    feature_release(str) -- feature release of the bootstrapper
                                            eg: SP21, SP22

                    packages(list)       -- list of features to be installed

            Note:
                Unix Installation Requires Filer Path/ Media Path ( Path till CVMedia) -- mediaPath

        """
        raise NotImplementedError("Module not implemented for the class")

    def restart_services(self):
        """
            This Method starts the service on the client when the client is not reachable

                Raises:
                SDKException:
                    Response was not success

        """
        raise NotImplementedError("Module not implemented for the class")

    @property
    def client_host(self):
        """ Read only attribute for client host """
        raise NotImplementedError(
            'Property Not Implemented by the Child Class')

    @property
    def get_machine_creds(self):
        """ Read only attribute for client host """
        raise NotImplementedError(
            'Property Not Implemented by the Child Class')


class UnixInstallHelper(InstallHelper):
    """Helper class to perform Unix install operations"""

    def __init__(self, commcell, machine_obj):
        """
        Initialises the UnixInstallHelper class

        Args:

            commcell   -- commcell object

            machine_obj -- machine object
        """
        super(UnixInstallHelper, self).__init__(commcell, machine_obj)

    def install_software(
            self,
            client_computers=None,
            features=None,
            username=None,
            password=None,
            install_path=None,
            **kwargs):
        """
            client_computers    (list)      -- client hostname list

            features (list of features)   -- list of features to be installed
                                             default - ['FILE_SYSTEM']

            username    (str)             -- username of the machine to install features on

                default : None

            password    (str)             -- password

                default : None

            install_path (str)            -- Software Installation Path

                default :  None

            **kwargs: (dict) -- Key value pairs for supporting conditional initializations
            Supported -
            install_flags (dict) - dictionary of install flag values
            Ex : install_flags = {"preferredIPfamily":2, "install32Base":True}

        """
        if client_computers is None:
            client_computers = [self.client_host]

        if features is None:
            features = ['FILE_SYSTEM']

        feature_vals = []
        for feature in features:
            feature_vals.append(getattr(UnixDownloadFeatures, feature).value)

        if username is None or password is None:
            username = self.config_json.Install.unix_client.machine_username
            password = self.config_json.Install.unix_client.machine_password

        self.log.info("Installing software for features {0} on client {1}".format(features, client_computers))

        return self.commcell.install_software(
            client_computers=client_computers,
            windows_features=None,
            unix_features=feature_vals,
            username=username,
            password=b64encode(password.encode()).decode(),
            install_path=install_path,
            **kwargs
        )

    def uninstall_client(self, delete_client=True):
        """ Module to uninstall client and hard delete from commcell.
            Deletes the Instance001 associated to the Commcell / Decoupled Client.

         Args:
            delete_client (bool)       : Delete client from commcell?

                                        False : Client in de-configured state on commcell

                                              : Client not part of Commcell (Decoupled Uninstallation)


        Exception:

            Machine Object not created with Credentials.

            If failed to uninstall client

            Failed to delete the client from Commcell : As client not part of commcell (Decoupled Client Instance)


        Note:
            Machine object should always be created with Machine credentials.

                Details to be mentioned on the config.json file under Install.

                        Machine_user_name   (str): Client UserName

                        Machine_password    (str): Client Password

                        Machine_host_name   (str): Client Hostname

            Delete client should be False for Uninstalling decoupled client

        """
        client_hostname = self.client_machine_obj.machine_name
        try:
            # This done to make sure we get the response code after executing on CLI
            if not (self.client_machine_obj.username and self.client_machine_obj.password):
                self.log.info("Creating machine object with credentials")
                self.client_machine_obj = self.options_selector.get_machine_object(
                    machine=self.config_json.Install.unix_client.machine_host,
                    username=self.config_json.Install.unix_client.machine_username,
                    password=self.config_json.Install.unix_client.machine_password)

            self.log.info("Checking if instance exists on client [{0}]".format(client_hostname))
            cmd_to_run = r'ls -ld /etc/CommVaultRegistry/Galaxy/Instance001 2>/dev/null | wc -l'
            unix_output_obj = self.client_machine_obj.execute_command(cmd_to_run)

            if "1" in unix_output_obj.output:
                self.log.info("Uninstalling existing Instance001 on client [{0}]".format(client_hostname))

                if "darwin" in self.client_machine_obj.os_flavour.lower():
                    cmd_to_run = r'echo ' + "\'" + self.client_machine_obj.password + "\'" + \
                                  r' | sudo -S /usr/local/bin/cvpkgrm -i Instance001'

                else:
                    cmd_to_run = '/usr/bin/cvpkgrm -i Instance001'

                self.log.info("Executing command [{0}] on client [{1}]"
                              .format(cmd_to_run, client_hostname))

                unix_output_obj = self.client_machine_obj.execute_command(cmd_to_run)

                if not (str(unix_output_obj.exit_code) == "0" or "Password" in str(unix_output_obj.output[0])):
                    raise Exception("Failed to uninstall existing instance on client [{0}]".format(client_hostname))

                if delete_client:
                    client_obj = self.commcell.clients.get(client_hostname)
                    self.options_selector.delete_client(client_obj.client_name)

            else:
                self.log.info("No Instance found on the client [{0}]".format(client_hostname))

            self.log.info("Uninstallation for client [{0}] completed successfully".format(client_hostname))

        except Exception as excp:
            self.log.error("Failed to uninstall existing commvault instance on [{0}]"
                           .format(client_hostname))
            raise Exception("\n [{0}]".format(str(excp)))

    def silent_install(self, client_name, tcinputs, feature_release=None, packages=None):
        """
        Installs the client on the remote machine depending on the giver user inputs/ Server selected

            Args:
                    client_name (str)    -- Client Name provided for installation

                    tcinputs     (dict)  -- testcase inputs / Dictionary with supported keys

                                         --  Inputs for Installation should be in a dictionary
                            Supported key / value for inputs:

                            Mandatory:
                                csClientName        (str)        Commserve Client Name
                                csHostname          (str)        Commserve Hostname
                                commservePassword   (str)        Commserve encrypted login password
                                mediaPath           (str)        Filer Path required for Unix installations
                                                                (Path till media)


                            Optional:
                                commserveUsername   (str)        Commserve Username
                                useExistingDump     (str)        Use existing dump for Dm2 / Workflow Engine ("0" or 1")
                                useExsitingCSdump   (str)        Use CS dump ("0" or "1")
                                CommservDumpPath    (str)        Dump path for Commserve
                                install32base       (str)        install 32bit software on 64bit Machine ("0" or "1")
                                restoreOnlyAgents   (str)       "0 or "1"
                                DM2DumpPath         (str)        Dump path for DM2 webservice
                                WFEngineDumpPath    (str)        Dump path for Workflow Engine
                                decoupledInstall    (str)        "0 or "1"
                                enableFirewallConfig (str)       "0 or "1"
                                firewallConnectionType (str)     "0" or "1" or "2"
                                httpProxyConfigurationType(str)  "0" or "1" or "2"
                                httpProxyPortNumber (str)        Port Number eg: "6256"
                                httpProxyHostName   (str)        Hostname of the proxy machine
                                portNumber          (str)        Port number to connect to the CVD eg:8410
                                enableProxyClient   (str)        Enable Client as a proxy "0" or "1"
                                proxyClientName     (str)        Proxy Client Name / Client Name on the commcell
                                proxyHostname       (str)        Proxy Client Hostname
                                proxyPortNumber     (str)        Proxy client Port Number to be used
                                sqlSaPassword       (str)        Sa (user) password for SQL access
                                installDirectoryUnix(str)        Path on which software to be installed on Unix Client
                                installDirectoryWindows(str)     Path on which software to be installed on Win Client
                                force_ipv4(unix)       (int)     0 for both IPv4 and IPv6 support
                                                                 1 for IPv4 support only
                                                                 2 for IPv6 support only


                    feature_release(str) -- feature release of the bootstrapper
                                            eg: SP20, SP21

                    packages(list)       -- list of features to be installed
                                            eg: features=['FILE_SYSTEM', 'MEDIA_AGENT']
                                            default - ['FILE_SYSTEM']
            Note:
                Unix Installation Requires Filer Path/ Media Path ( Path till CVMedia) -- mediaPath
        """
        self.log.info("Starting installation for Unix Machine")
        try:
            if not (self.client_machine_obj.username and self.client_machine_obj.password):
                self.log.info("Creating machine object with credentials.")
                self.client_machine_obj = self.options_selector.get_machine_object(
                    machine=self.config_json.Install.unix_client.machine_host,
                    username=self.config_json.Install.unix_client.machine_username,
                    password=self.config_json.Install.unix_client.machine_password)

            if feature_release is None:
                feature_release = "SP" + str(self.commcell.commserv_version)

            if packages is None:
                packages = ['FILE_SYSTEM']

            package_list = []
            for package in packages:
                package_list.append(getattr(UnixDownloadFeatures, package).value)

            silent_helper = SilentInstallHelper.create_installer_object(client_name,
                                                                        feature_release.upper(),
                                                                        self.client_machine_obj,
                                                                        tcinputs)
            silent_helper.silent_install(package_list)
            self.commcell.refresh()
            self.log.info("Silent installation complete on Unix machine")

        except Exception as exp:
            self.log.error("Failed to install commvault software on :%s" % self.client_machine_obj)
            raise Exception("\n [{0}]".format(str(exp)))

    def restart_services(self):
        """
            This Method starts the service on the client when the client is not reachable

            Raises:
                SDKException:
                    Response was not success

        """

        self.client_machine_obj.execute_command("commvault -all restart")
        client_obj = self.commcell.clients.get(self.client_host)
        if not client_obj.is_ready:
            raise Exception("Failed to restart services on the client:%s", client_obj.client_name)

        self.log.info("Successfully Restarted services on the Client: %s", client_obj.client_name)

    @property
    def client_host(self):
        """ Read only attribute for client host """
        return self.config_json.Install.unix_client.machine_host

    @property
    def get_machine_creds(self):
        """ Read only attribute for client host """
        return (self.config_json.Install.unix_client.machine_username,
                self.config_json.Install.unix_client.machine_password)


class WindowsInstallHelper(InstallHelper):
    """Helper class to perform Windows install operations"""

    def __init__(self, commcell, machine_obj):
        """
        Initialises the WindowsInstallHelper class

        Args:

            commcell   -- commcell object

           machine_obj -- machine object
        """
        super(WindowsInstallHelper, self).__init__(commcell, machine_obj)
        self.remote_dir = installer_constants.REMOTE_FILE_COPY_LOC

    def _copy_files_to_client(self, file_name):
        """Copies the created Batch file to the Client Machine
                Args:
                        file_name     (str)  --  Name of the file to be copied to Client
                """

        self.log.info("Copying file [{0}] on Client [{1}] at [{2}]"
                      "".format(file_name, self.client_machine_obj.machine_name, self.remote_dir))

        self.options_selector.create_directory(self.client_machine_obj, self.remote_dir)
        self.client_machine_obj.copy_from_local(file_name, self.remote_dir)

        self.log.info("Successfully copied file [{0}] on the client [{1}]"
                      .format(file_name, self.client_machine_obj.machine_name))

    def install_software(
            self,
            client_computers=None,
            features=None,
            username=None,
            password=None,
            install_path=None,
            **kwargs):
        """
            client_computers    (list)      -- client hostname list

            features (list of features)   -- list of features to be installed
                                            eg: features=['FILE_SYSTEM', 'MEDIA_AGENT']
                                            default - ['FILE_SYSTEM']

            username    (str)             -- username of the machine to install features on

                default : None

            password    (str)             -- password

                default : None

            install_path (str)             -- Software Installation Path

                default : None

            **kwargs: (dict) -- Key value pairs for supporting conditional initializations
            Supported -
            install_flags (dict) - dictionary of install flag values
            Ex : install_flags = {"preferredIPfamily":2, "install32Base":True}
        """
        if client_computers is None:
            client_computers = [self.client_host]

        if features is None:
            features = ['FILE_SYSTEM']

        feature_vals = []
        for feature in features:
            feature_vals.append(getattr(WindowsDownloadFeatures, feature).value)

        if username is None or password is None:
            username = self.config_json.Install.windows_client.machine_username
            password = self.config_json.Install.windows_client.machine_password

        self.log.info("Installing software for features {0} on clients {1}".format(features, client_computers))

        return self.commcell.install_software(
            client_computers=client_computers,
            windows_features=feature_vals,
            unix_features=None,
            username=username,
            password=b64encode(password.encode()).decode(),
            install_path=install_path,
            **kwargs
        )

    def uninstall_client(self, delete_client=True):
        """ Module to uninstall client and hard delete from commcell.
            Deletes the Instance001 associated to the Commcell / Decoupled Client.

         Args:
            delete_client (bool)       : Delete client from commcell?

                                        False : Client in de-configured state on commcell

                                              : Client not part of Commcell (Decoupled Uninstallation)


        Exception:

            Machine Object not created with Credentials.

            If failed to uninstall client

            Failed to delete the client from Commcell : As client not part of commcell (Decoupled Client Instance)


        Note:
            Machine object should always be created with Machine credentials.

                Details to be mentioned on the config.json file under Install.

                        Machine_user_name   (str): Client UserName

                        Machine_password    (str): Client Password

                        Machine_host_name   (str): Client Hostname

            Delete client should be False for Uninstalling decoupled client

        """
        client_hostname = self.client_machine_obj.machine_name
        try:
            # This done to make sure we get the response code/message after executing on CLI
            if not (self.client_machine_obj.username and self.client_machine_obj.password):
                self.log.info("Creating machine object with credentials.")
                self.client_machine_obj = self.options_selector.get_machine_object(
                    machine=self.config_json.Install.windows_client.machine_host,
                    username=self.config_json.Install.windows_client.machine_username,
                    password=self.config_json.Install.windows_client.machine_password)

            self.log.info("Uninstalling Instance001 on client [{0}]".format(client_hostname))
            command_list = []
            command_list.append(r"SET regPath=%WINDIR%\System32")
            command_list.append(
                "FOR /f \"usebackq tokens=3 skip=2\" %%L IN "
                "(`%regPath%\REG QUERY "
                "\"HKLM\SOFTWARE\CommVault Systems\Galaxy\Instance001\InstalledPackages\" "
                "/v BundleProviderKey 2^>null`) DO SET bundlekey=%%L")
            command_list.append(
                "IF NOT \"%bundlekey%\"==\"\" "
                "(START \"\" /wait \"%ALLUSERSPROFILE%\Package Cache\%bundlekey%\Setup.exe\" "
                "/uninstall /silent /instance Instance001)")

            self.log.info("Uninstall command: Setup.exe /uninstall /silent /instance instance001")

            local_file = os.path.join(self.test_results_path, "cvuninstall.bat")
            install_batch_file = installer_utils.create_batch_file_for_remote(
                commands=command_list, file_name=local_file)
            self._copy_files_to_client(install_batch_file)
            _command = os.path.join(self.remote_dir, os.path.basename(install_batch_file))

            self.log.info("Executing command [{0}] on client [{1}]".format(_command, client_hostname))

            return_code = self.client_machine_obj.execute_command(_command)

            if return_code.exit_code != 0:
                self.log.info("Result String is " + installer_constants.QINSTALLER_RETURNCODES[return_code])
                raise Exception("Failed to uninstall client [{0}].Please check install.log of client"
                                "".format(client_hostname))

            self.log.info("Uninstallation for client [{0}] completed successfully".format(client_hostname))

            if delete_client:
                client_obj = self.commcell.clients.get(client_hostname)
                self.options_selector.delete_client(client_obj.client_name)

        except Exception as excp:
            self.log.error("Failed to uninstall existing commvault instance on [{0}]".format(client_hostname))
            raise Exception("\n [{0}]".format(str(excp)))

    def silent_install(self, client_name, tcinputs, feature_release=None, packages=None):
        """
        Installs the client on the remote machine depending on the giver user inputs/ Server selected

            Args:
                    client_name (str)    -- Client Name provided for installation

                    tcinputs    (dict)   -- testcase inputs / Dictionary with supported keys

                                         --  Inputs for Installation should be in a dictionary
                            Supported key / value for inputs:

                            Mandatory:
                                csClientName        (str)        Commserve Client Name
                                csHostname          (str)        Commserve Hostname
                                commservePassword:  (str)        Commserve encrypted login password

                            Optional:
                                commserveUsername   (str)        Commserve Username
                                useExistingDump     (str)        Use existing dump for Dm2 / Workflow Engine ("0" or 1")
                                useExsitingCSdump   (str)        Use CS dump ("0" or "1")
                                CommservDumpPath    (str)        Dump path for Commserve
                                install32base       (str)        install 32bit software on 64bit Machine ("0" or "1")
                                restoreOnlyAgents   (str)       "0 or "1"
                                DM2DumpPath         (str)        Dump path for DM2 webservice
                                WFEngineDumpPath    (str)        Dump path for Workflow Engine
                                decoupledInstall    (str)        "0 or "1"
                                enableFirewallConfig (str)       "0 or "1"
                                firewallConnectionType (str)     "0" or "1" or "2"
                                httpProxyConfigurationType(str)  "0" or "1" or "2"
                                httpProxyPortNumber (str)        Port Number eg: "6256"
                                httpProxyHostName   (str)        Hostname of the proxy machine
                                portNumber          (str)        Port number to connect to the CVD eg:8410
                                enableProxyClient   (str)        Enable Client as a proxy "0" or "1"
                                proxyClientName     (str)        Proxy Client Name / Client Name on the commcell
                                proxyHostname       (str)        Proxy Client Hostname
                                proxyPortNumber     (str)        Proxy client Port Number to be used
                                sqlSaPassword       (str)        Sa (user) password for SQL access
                                installDirectoryUnix(str)        Path on which software to be installed on Unix Client
                                installDirectoryWindows(str)     Path on which software to be installed on Win Client


                   feature_release(str) -- feature release of the bootstrapper
                                           eg: SP20, SP21

                   packages(list)       -- list of features to be installed
                                            features=['FILE_SYSTEM', 'MEDIA_AGENT']
                                            default - ['FILE_SYSTEM']

            Note:
                Unix Installation Requires Filer Path/ Media Path ( Path till CVMedia) -- mediaPath

        """
        try:
            if not (self.client_machine_obj.username and self.client_machine_obj.password):
                self.log.info("Creating machine object with credentials.")
                self.client_machine_obj = self.options_selector.get_machine_object(
                    machine=self.config_json.Install.windows_client.machine_host,
                    username=self.config_json.Install.windows_client.machine_username,
                    password=self.config_json.Install.windows_client.machine_password)

            if feature_release is None:
                feature_release = "SP" + str(self.commcell.commserv_version)

            if packages is None:
                packages = ['FILE_SYSTEM']

            package_list = []
            for package in packages:
                package_list.append(getattr(WindowsDownloadFeatures, package).value)

            silent_helper = SilentInstallHelper.create_installer_object(client_name,
                                                                        feature_release.upper(),
                                                                        self.client_machine_obj,
                                                                        tcinputs)
            silent_helper.silent_install(package_list)
            self.commcell.refresh()
            self.log.info("Silent installation complete for the Windows machine")

        except Exception as exp:
            self.log.error("Failed to install commvault software on :%s" % self.client_machine_obj)
            raise Exception("\n [{0}]".format(str(exp)))

    def restart_services(self):
        """
            This Method starts the service on the client when the client is not reachable

            Raises:
                SDKException:
                    Response was not success

        """
        self.client_machine_obj.execute_command(r"cd $Env:CV_Instance001;.\gxadmin -console -restartsvcgrp ALL")
        client_obj = self.commcell.clients.get(self.client_host)
        if not client_obj.is_ready:
            raise Exception("Failed to restart services on the client:%s", client_obj.client_name)

        self.log.info("Successfully Restarted services on the Client: %s", client_obj.client_name)

    @property
    def client_host(self):
        """ Read only attribute for client host """
        return self.config_json.Install.windows_client.machine_host

    @property
    def get_machine_creds(self):
        """ Read only attribute for client host """
        return (self.config_json.Install.windows_client.machine_username,
                self.config_json.Install.windows_client.machine_password)
