# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations for bootstrapper download

BootstrapperHelper
===================

    __init__()                          --  initialize instance of the BootstrapperHelper class
    local_machine_drive                 --  Returns local machine drive
    remote_machine_drive                --  Returns remote machine drive
    cleanup                             --  cleans up testcase results
    bootstrapper_download_url           --  returns download server path of the bootstrapper installer
    download_bootstrapper_exe           --  downloads bootstrapper installer .exe file
    download_payload_from_bootstrapper  --  downloads payload from bootstrapper
    extract_bootstrapper                --  extracts and downloads payload from bootstrapper
"""
import re
import shutil
from bs4 import BeautifulSoup as bs
import urllib3
from AutomationUtils import logger
from AutomationUtils import config
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.machine import Machine
from Install import installer_constants


class BootstrapperHelper:
    """BootstrapperHelper helper class to perform bootstrapper related operations"""

    def __init__(self, feature_release, machine_obj, **kwargs):
        """
        Initialize instance of the BootstrapperHelper class.

        Args:
            feature_release -- feature release of the bootstrapper
            machine_obj -- machine object

            **kwargs: (dict) -- Key value pairs for supporting conditional initializations
            Supported -
            download_server (str) - download server from which bootstrapper .exe file will be downloaded
            bootstrapper_download_os (str) - Oses to download E.g. bootstrapper_download_os="Windows,Unix"
        """
        self.log = logger.get_log()
        self.config_json = config.get_config()
        self.machine_obj = machine_obj
        self.local_machine = Machine()
        self.feature_release = feature_release
        self.download_server = kwargs.get('download_server', self.config_json.Install.download_server)
        self.bootstrapper_download_os = kwargs.get('bootstrapper_download_os', 'Windows').split(",")
        self.windows_os_to_download = ",".join(str(bit)
                                               for bit in installer_constants.WINDOWS_BOOTSTRAPPER_DOWNLOAD_OSID)
        self.unix_os_to_download = ",".join(str(bit) for bit in installer_constants.UNIX_BOOTSTRAPPER_DOWNLOAD_OSID)
        self.local_drive = None
        self.remote_drive = None

    @property
    def local_machine_drive(self):
        """
        Returns local machine drive
        """
        if not self.local_drive:
            self.local_drive = OptionsSelector.get_drive(self.local_machine)
        return self.local_drive

    @property
    def remote_machine_drive(self):
        """
        Returns remote machine drive
        """
        if not self.remote_drive:
            self.remote_drive = OptionsSelector.get_drive(self.machine_obj)
        return self.remote_drive

    def cleanup(self):
        """
        Cleans up testcase results
        """
        cmd = f"Remove-Item -LiteralPath {self.local_file_copy_loc}" \
              f" -Force -Recurse -ErrorAction Ignore;" \
              f"Remove-Item -LiteralPath {self.remote_machine_drive}" \
              f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH}" \
              f" -Force -Recurse -ErrorAction Ignore;"
        output = self.local_machine.execute_command(cmd)
        self.log.info(output.formatted_output)

    def bootstrapper_download_url(self):
        """
        returns download server path of the bootstrapper installer
        """
        return f"http://{self.download_server}/CVMedia/11.0.0/BUILD80/{self.feature_release}DVD/BootStrapper/Commvault-1/"

    def download_bootstrapper_exe(self):
        """
        Downloads bootstrapper installer .exe file

        Raises:
            exception if unable to find exe
            exception if unable to download exe
        """
        try:
            http = urllib3.PoolManager()
            url = self.bootstrapper_download_url()
            response = http.request('GET', url)
            soup = bs(response.data)
            link = soup.findAll('a', attrs={'href': re.compile(".exe")})
            self.exe_name = link[0].get('href')
            self.log.info("Downloading %s from %s", self.exe_name, url)
            self.local_machine.create_directory(f"{self.local_file_copy_loc}", force_create=True)
            with http.request('GET', f"{url}{self.exe_name}", preload_content=False) as resp, \
                    open(f"{self.local_file_copy_loc}{self.exe_name}", 'wb') as out_file:
                shutil.copyfileobj(resp, out_file)

        except Exception as exp:
            self.log.exception(exp)
            raise Exception("Unable to download the exe file")

    def download_payload_from_bootstrapper(self):
        """
        download Payload from Bootstrapper
        """
        self.log.info("Downloading from Bootstrapper")
        try:
            for each_os in self.bootstrapper_download_os:
                if each_os.upper() == "UNIX":
                    download_os_path = f"{self.remote_machine_drive}" \
                                       f"{installer_constants.UNIX_BOOTSTRAPPER_DOWNLOADPATH}"
                    os_list = self.unix_os_to_download
                else:
                    download_os_path = f"{self.remote_machine_drive}" \
                                       f"{installer_constants.WINDOWS_BOOTSTRAPPER_DOWNLOADPATH}"
                    os_list = self.windows_os_to_download

                self.log.info("going to Download at %s", download_os_path)
                cmd = f"Remove-Item -LiteralPath \"{download_os_path}\" -Force -Recurse -ErrorAction Ignore\n" \
                    f"ping 127.0.0.1 -n 16\n" \
                    f"$app = \"{self.remote_machine_drive}" \
                      f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH}\\Setup.exe\"\n" \
                    f"$arg = \"/silent /download{each_os} /OSList {os_list} /outputpath {download_os_path}\"\n" \
                    f"Start-Process $app $arg -Wait"
                output = self.machine_obj.execute_command(cmd)
                self.log.info(output.formatted_output)

        except Exception as err:
            self.log.exception(str(err))
            raise Exception(f"Downloading payload failed for os {each_os}")

    def extract_bootstrapper(self):
        """
        Extracts bootstrapper
        """
        self.local_file_copy_loc = f"{self.local_machine_drive}{installer_constants.REMOTEFILECOPYLOC}"
        self.remote_file_copy_loc = f"{self.remote_machine_drive}{installer_constants.REMOTEFILECOPYLOC}"
        self.log.info("Downloading exe file for bootstrapper")
        self.download_bootstrapper_exe()
        self.log.info("Extracting Bootstrapper")
        if self.machine_obj.ip_address != self.local_machine.ip_address:
            self.log.info("Copying to remote machine")
            self.machine_obj.copy_from_local(f"{self.local_file_copy_loc}{self.exe_name}", self.remote_file_copy_loc)
            self.log.info("File copied successfully")
            cmd = f"New-Item -ItemType Directory -Force -Path {self.remote_machine_drive}" \
                  f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH};" \
                  f"$app = \"{self.remote_file_copy_loc}{self.exe_name}\";" \
                  f"$arg = \"/d {self.remote_machine_drive}" \
                  f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH} /silent /noinstall\";" \
                  f"Start-Process $app $arg -Wait"
            output = self.machine_obj.execute_command(cmd)
        else:
            cmd = f"New-Item -ItemType Directory -Force -Path {self.local_machine_drive}" \
                  f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH};" \
                  f"$app = \"{self.local_file_copy_loc}{self.exe_name}\";" \
                  f"$arg = \"/d {self.local_machine_drive}" \
                  f"{installer_constants.BOOTSTRAPPER_EXECUTABLE_EXTRACTPATH} /silent /noinstall\";" \
                  f"Start-Process $app $arg -Wait"
            output = self.local_machine.execute_command(cmd)
        self.log.info("Output from powershell is %out", output.formatted_output)
